;+
;NAME:
; spd_ui_draw_object
;
;PURPOSE:
; This object generates an object graphics tree using the IDLgr* objects that are built
; into IDL.  It generates this tree by querying the loaded data object and the window storage
; object to get the settings generated by the panel.  This object also performs other utility
; operations related to drawing.  These include animations of vertical bars and markers, updating
; the legend by performing a lookup in a hashed version of the plotted data, and reporting information
; about the current display.(Cursor segment selected,  range of displayed data, etc...)
;
; There are a few major functions in this object.
; #1 'update': will update the draw area to correspond to new settings
; #2 'setCursor' : actually makes the draw object do all the animation updates
; #3 'draw' : use it to force the object to redraw the draw area.\
; #4 'getInfo(panelIndex)' : get a struct with information about a panel in the currently drawn display
;                            This includes information like the actual range and scaling for each axis, the number of tics, etc...
;
;NOTES:
;
; 1. In almost all cases ranges will be stored and passed around as 2-element arrays.
;    If they are on log scaled axis, the ranges will be in log space, this tends to make some calculations simpler(and others more complex)
;    By log space, I mean, if scaling eq 1 then range = alog10(normalRange), if scaling eq 2 then range = alog(normalRange)
;    
; 2. The codes used for different scaling values are consistent.  0 indicates linear scaling, 1 indicates base 10 logarithm, 2 indicates natural logarithm
; 
; 3. Positions in this code are generally specified in one of three different coordinate systems.  
;    a.  They are normalized to the draw area.  So [.5,.5] is the center of the draw area.
;    b.  They are normalized to the panel.  So [.5,.5] is the center of a panel.
;    c.  They are specified in pixels.  Often when pixels are used, the value used may actually be 2x or 4x the actual number of screen pixels to prevent aliasing.
;    
;    Somewhat ambiguously, c. is sometimes called points(because it is a scaled number of pixels).  This is not the same as the measurement in note 5.
; 
;    Rectangles(like panels) will normally be specified using two variables, which are each 2 element doubles.
;    xpos = [xstart,xstop],  ypos = [ystart,stop]  These are absolute locations. 
;    The alternative representation [xstart,xsize],[ystart,ysize], is rarely, if ever, used as a parameter,
;    although sizes are used for various computations throughout the code.
;    
; 4. distances that aren't specified in the above coordinates are often specified in pts. 
;    This is a desktop publishing standard, that ideally is consistent across output devices.  1 pt should be 127/360 mm.
;    The goal is to make all output consistent across output devices.
;    
; 
;
;$LastChangedBy: jimm $
;$LastChangedDate: 2014-02-11 10:54:32 -0800 (Tue, 11 Feb 2014) $
;$LastChangedRevision: 14326 $
;$URL: svn+ssh://thmsvn@ambrosia.ssl.berkeley.edu/repos/spdsoft/tags/spedas_5_0/spedas_gui/display/spd_ui_draw_object__define.pro $
;-

;create a new draw object.
;
;destination:
;  window created by widget_draw(graphics_level=2),
;  or other IDLgrDestination object(IDLgrClipboard, IDLgrBuffer, IDLgrPrinter,...)
;  
;statusBar:
;  spd_ui_status_bar object to which status message should be sent
;  
;historyWin
;  spd_ui_historyWin object to which history messages should be sent
;
;
function spd_ui_draw_object::Init,destination,statusbar,historyWin

  compile_opt idl2
  
  if ~obj_valid(destination) then begin
    t=error_message('Invalid destination object passed to spd_ui_draw_object',/traceback)
    return,0
  endif
  
  if ~obj_valid(Statusbar) then begin
    t = error_message('Expected valid status bar',/traceback)
    return,0
  endif
  
  if ~obj_valid(historyWin) then begin
    t = error_message('Expected valid history Window',/traceback)
    return,0
  endif
  
  
  self.destination = destination
  self.statusbar = statusbar
  self.historyWin = historyWin
  
  self.scene = OBJ_NEW('IDLgrScene',color=[255,255,255])
  
  rubberview = obj_new('IDLgrView')
  self.rubberview = rubberview
  rubberview->setProperty,units=3,viewplane_rect=[0.,0.,1.,1.],location=[0.,0.],dimensions=[1.,1.],zclip=[1.,-1.],eye=5,name="rubberview",/transparent,/double
  self.scene->add,rubberview
  
  self.staticViews = obj_new('IDL_Container')
  self.dynamicViews = obj_new('IDL_Container')
   
  self.lineres = 4.0D
  self.specres = 1.0D
  self.pointmax = 4096
  
  self.maxTickNum = 200
  
  return,1
  
end


pro spd_ui_draw_object__define

  struct = { SPD_UI_DRAW_OBJECT, $
    destination:OBJ_NEW(), $ ; The output object
    scene:OBJ_NEW(), $ ; The current scene(private)
    pageview:OBJ_NEW(), $  ; The view for the page includes the title
    panelViews:obj_new(),$ ; view group of panel views
    rubberview:OBJ_NEW(), $  ; The rubber band view
    rubberstart:dblarr(2), $ ; The starting location of a rubberband box
    markerstart:dblarr(2), $  ; The starting location of a marker draw
    cursorloc:dblarr(2), $  ; The current location of the cursor
    markerOn:0, $ ; boolean value indicating whether to draw a marker 0: no marker, 1: single panel marker, 2: all panel markers
    rubberOn:0, $ ; boolean value indicating whether to draw a rubber band
    vBarOn:0, $  ;value indicating whether to draw a vertical bar, 0 = no bar, 1 = single panel bar, 2 = all panel bar
    hBarOn:0, $  ;value indicating whether to draw a horizontal bar, 0 = no bar, 1 = single panel bar, 2 = all panel bar
    legendOn:0, $  ;value indicating whether to draw a legend, 0 = no legend, 1 = single panel legend, 2 = all panel legend
    nRows:1, $ ;number of rows in current display
    nCols:1, $ ;number of columns in current display
    panelInfo:ptr_new(), $ ;an array of structures that describe information in panels that is relevant to drawing annotations
    currentPageSize:[8.5,11], $ ; the current size of the page in inches
    currentMarkers:ptr_new(), $ ;a list of the currently/most recently drawn markers
    markerIdx:ptr_new(), $ ; a list of the marker panel indexes
    statusBar:obj_new(), $  ; reference to the status bar from the main window, for error output
    historyWin:obj_new(), $ ; reference to the history window
    postscript:0B, $  ; indicates whether or not a vector postscript is being generated
    lineres:2D, $ ; a multiplier for line plots to determine how much it is up or down sampled prior to display
    specres:1D, $ ; a multiplier for spectral plots to determine how they are up or down sampled prior to display
    fancompressionfactor:0d,$ ; percentage error to be allowed if fancompression is used during postscript plotting
    pointmax:4096, $ ; the maximum resolution at which a screen render should be made
    staticViews:obj_new(), $ ; This view group contains all visual components that change only during updates
    dynamicViews:obj_new(), $ ; This view group contains all visual components that change between updates and during updates
    maxTickNum:200 $ ; The maximum tick limit before a warning is issued
    }
;margins:dblarr(5) } ; The [left,right,top,bottom,internal] margins in normalized coordinates
    
end


