# INPUTS
#    iGrid   - the input grid
#    igInfo  - the input grid information array
#    oGrid   - the output (expanded) grid
#    ogInfo  - the output grid information array
#    xF      - number of cells to add between input grid cells along X
#    yF      - number of cells to add between input grid cells along Y

package provide TclUtils 1.0

proc TUgridGrow { iGrid igInfo oGrid ogInfo xF yF } {

   upvar $iGrid iG
   upvar $igInfo igI
   upvar $oGrid oG
   upvar $ogInfo ogI

# GROW by an odd number of cells since this gives a true center cell.
#   Check below makes sure and silently corrects any problems

   set xF [expr int($xF)]
   set yF [expr int($yF)]
   if { [expr $xF % 2] == 0 } { incr xF }
   if { [expr $yF % 2] == 0 } { incr yF }

# COMPUTE the length of a super cell in the output grid.  This length
#    includes the known corner cells

   set xBase [expr $xF + 2] 
   set yBase [expr $yF + 2] 

# COMPUTE the offset to the last row or column number in a super cell.  This
#   is actually one less then the length because each super cell builds on
#   the edge(s) of the last.

   set mxF [expr $xF + 1]
   set myF [expr $yF + 1]

# COMPUTE the offset to the super cell center cell

   set xM [expr int($xF / 2) + 1]
   set yM [expr int($yF / 2) + 1]

# FORM the output grid information array.  With the exception of the number 
#   of cells along X and Y we will follow the definition of the input grid.

   for { set I 0 } { $I < 19 } { incr I } { set ogI($I) $igI($I) } 
   set ogI(4) [expr ($xF + 1) * $igI(4) - $xF]
   set ogI(5) [expr ($yF + 1) * $igI(5) - $yF]
   if [string match YES $igI(15)] { set ogI(4) [expr $ogI(4) + $xF] }
   if [string match YES $igI(16)] { set ogI(5) [expr $ogI(5) + $xF] }

#  if { $NotFull == 1 } {
#     set IncX [expr ($ogI(1) - $ogI(0)) / $ogI(4)]
#     set IncY [expr ($ogI(2) - $ogI(3)) / $ogI(5)]
#     set ogI(0) [expr ogI(0) + 2.0 * IncX]
#     set ogI(1) [expr ogI(1) - 2.0 * IncX]
#     set ogI(2) [expr ogI(2) + 2.0 * IncY]
#     set ogI(3) [expr ogI(3) - 2.0 * IncY]
#  }

# ZERO the output grid

   set tG [expr $ogI(4) * $ogI(5)]
   for { set I 0 } { $I < $tG } { incr I } { set oG($I) 0.0 } 

# SET up the triangle iD mask for the supergrid.  This divides the super
#   cell into 4 triangles meeting at the mid-point.  Each cell in the
#   super cell is assigned to one of these triangles.  Cells which form
#   a common border can be assigned to either of the two triangles of
#   which they are a part.

   set bR 0
   set eR $myF
   set c $mxF
   for { set C 0 } { $C <= $xM  } { incr C ; incr c -1 } {
      for { set R $bR } { $R <= $eR } { incr R } {
          set TiD($C,$R) 0
          set TiD($c,$R) 2
      }
      incr bR
      incr eR -1
   }

   set bC 1
   set eC [expr $mxF - 1]
   set r $myF
   for { set R 0 } { $R < $yM  } { incr R ; incr r -1 } {
      for { set C $bC } { $C <= $eC } { incr C } {
          set TiD($C,$R) 3
          set TiD($C,$r) 1
      }
      incr bC
      incr eC -1
   }

# THIS is the array index of the last row and column in the input grid

   set LastC [expr $igI(4) - 1]
   set LastR [expr $igI(5) - 1]
   if [string match YES $igI(15)] { incr LastC }
   if [string match YES $igI(16)] { incr LastR }

# THE center of a super cell always has co-ordinates of 0,0.  This allows
#    the X,Y coordinates of the super cell corners to be set in advance.

   set V(0) [expr -$xM] ; set V(1)  [expr -$yM]
   set V(3) [expr -$xM] ; set V(4)  $yM
   set V(6) $xM         ; set V(7)  $yM
   set V(9) $xM         ; set V(10) [expr -$yM]

# SET the beginning column to start at in the super-cell.  This is always
#   0 for the first column of super cells and 1 for every other column since
#   a super cell fixes the first column of cells in the super cell to its
#   right.

   set cB 0

# OK now fill in the super cells.  This process is separated depending on
#   whether the input grid was stored as a series of COLUMNS or ROWS.  

   if [string match COLUMN $igI(8)] {

# LOOP over the columns of the input grid

      for { set C 0 } { $C < $LastC } { incr C } {

# SET the beginning row to start at in the super-cell.  This is always
#   0 for the first super cell in a column of super cells and 1 for every 
#   super cell in the column since each super cell fixes lowest row of 
#   cells in the super cell above it.

         set rB 0

# SET the grid position of the lower left and right-hand corners of the
#   current super-cell and then get their intensities.
 
         set bLeft  [expr ($C % $igI(4)) * $igI(5)]
         set bRight [expr (($C + 1) % $igI(4)) * $igI(5)]
         set cV(0) $iG($bLeft)
         set cV(3) $iG($bRight)

# This is the grid position of the left-hand corner of the super cell in
#   the output grid array.

         set gN [expr ($C * $mxF) * $ogI(5)]

# LOOP over the rows of the input grid

         for { set R 1 } { $R <= $LastR } { incr R } {

# GET the values in the upper two super cell corners

             set cV(1) $iG([expr $bLeft + ($R % $igI(5))])
             set cV(2) $iG([expr $bRight + ($R % $igI(5))])

# SET the center value of the current super cell.  This is just the average
#   of the values at the 4 corners.

            set mV [expr ($cV(0) + $cV(1) + $cV(2) + $cV(3))/4.0]

# COMPLETE the vectors which define the planes formed by the 4 triangles.
#   Taking the cross product of the two vectors defining the triangle gives
#   the normal to the plane which is used to set the equation of the plane.
#
#                       Ax + By + Cz = D 
#
#   Solve for the equations of each of the 4 planes up top.

            set V(2)  [expr $cV(0) - $mV]
            set V(5)  [expr $cV(1) - $mV]
            set V(8)  [expr $cV(2) - $mV]
            set V(11) [expr $cV(3) - $mV]
            TUvecMath V CROSS V Vc 0 3 0
            TUvecMath V CROSS V Vc 3 6 3
            TUvecMath V CROSS V Vc 6 9 6
            TUvecMath V CROSS V Vc 9 0 9
            set D(0) [expr $mV * $Vc(2)]
            set D(1) [expr $mV * $Vc(5)]
            set D(2) [expr $mV * $Vc(8)]
            set D(3) [expr $mV * $Vc(11)]

# LOOP over each cell in the super cell.  The triangle iD mask computed 
#   near the start indicated which triangle (hence plane) to use in 
#   determining the intensity to assign to that cell.

            for { set c $cB } { $c < $xBase } { incr c } {
               for { set r $rB } { $r < $yBase } { incr r } {

# THIS is the cell triangle iD

                   set N $TiD($c,$r)

# THIS is the location of the super cell sub-cell within the output grid
#   array

                   set P [expr $gN + $r + $c * $ogI(5)]

# THESE are the coordinates of the sub-cell

                   set X [expr $V(0) + $c]
                   set Y [expr $V(1) + $r]

# THIS is the location of A, B, and C for the plane

                   set oX [expr 3 * $N]
                   set oY [expr $oX + 1]
                   set oZ [expr $oY + 1]

# SET the intensity

                   set oG($P) [expr ($D($N) - $Vc($oX) * $X - \
                                              $Vc($oY) * $Y) / $Vc($oZ)]
               }
            }

            set rB 1
            incr gN $myF
            set cV(0) $cV(1)
            set cV(3) $cV(2)
         }
         set cB 1
      }
   } else {
      set dY [expr $myF * $ogI(4)]
      set nG [expr $igI(4) * $igI(5)]
      for { set C 0 } { $C < $LastC } { incr C } {
         set rB 0
         set bLeft  $C
         set bRight [expr ($C + 1) % igI(4)]
         set cV(0) $iG($bLeft)
         set cV(3) $iG($bRight)

         set gN [expr $C * $mxF]

         for { set R 0 } { $R < $LastR } { incr R } {
            set cV(1) $iG([expr $bLeft + ($R % $igI(5)) * $igI(4)])
            set cV(2) $iG([expr $bRight + ($R % $igI(5)) * $igI(4)])
            set mV [expr ($cV(0) + $cV(1) + $cV(2) + $cV(3))/4.0]
            set V(2)  [expr $cV(0) - $mV]
            set V(5)  [expr $cV(1) - $mV]
            set V(8)  [expr $cV(2) - $mV]
            set V(11) [expr $cV(3) - $mV]
            TUvecMath V CROSS V Vc 0 3 0
            TUvecMath V CROSS V Vc 3 6 3
            TUvecMath V CROSS V Vc 6 9 6
            TUvecMath V CROSS V Vc 9 0 9
            set D(0) [expr $mV * $Vc(2)]
            set D(1) [expr $mV * $Vc(5)]
            set D(2) [expr $mV * $Vc(8)]
            set D(3) [expr $mV * $Vc(11)]

            for { set c $cB } { $c < $xBase } { incr c } {
               for { set r $rB } { $r < $yBase } { incr r } {
                   set N $TiD($c,$r)
                   set P [expr $gN + $c + $r * $ogI(4)]
                   set X [expr $V(0) + $c]
                   set Y [expr $V(1) + $r]
                   set oX [expr 3 * $N]
                   set oY [expr $oX + 1]
                   set oZ [expr $oY + 1]
                   set oG($P) [expr ($D($N) - $Vc($oX) * $X - \
                                              $Vc($oY) * $Y) / $Vc($oZ)]
               }
            }

            set rB 1
            incr gN $dY
            set cV(0) $cV(1)
            set cV(3) $cV(2)
         }
         set cB 1
      }
   } 
}
