/*  The C equivalent of TUvecMath,  It consists of a TCL interface over     */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 10 OBJV elements.  These are:                                 */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  Input Vector 1                                             */
/*     OBJV[2]:  Operation                                                  */
/*     OBJV[3]:  Input Vector 2                                             */
/*     OBJV[4]:  Output Vector                                              */
/*     OBJV[5]:  Offset into first vector                                   */
/*     OBJV[6]:  Offset into second vector                                  */
/*     OBJV[7]:  Offset into output vector array                            */
/*     OBJV[8]:  Length of first vector                                     */
/*     OBJV[9]:  Length of second vector                                    */

#include <tcl.h>
#include <stdlib.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int VecMathCmd (ClientData cD, Tcl_Interp *tIntp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr; 
   ReaL_8 *V, *W, *R, rV;
   ByTe_4 vLen, Bytes, RSize = 0, TmP, Roff = 0;
   ByTe_1 Op, *Oper, XFerR;
   int LenA = 3, LenB = 3, OffA = 0, OffB = 0, OffC = 0;
   Tcl_Obj *objPtr;  

/* Make sure that all the IO parameters are present */

   if (objc < 5 ) {
     Tcl_WrongNumArgs(tIntp, 1, objv, "Usage: TUvecMath V1 Op V2 V");
     return TCL_ERROR;
   }

/* Get the input parameters which may or may not be present                */
      
   if (objc >= 6)  { Tcl_GetIntFromObj (tIntp, objv[5], &OffA); }
   if (objc >= 7)  { Tcl_GetIntFromObj (tIntp, objv[6], &OffB); }
   if (objc >= 8)  { Tcl_GetIntFromObj (tIntp, objv[7], &OffC); }
   if (objc >= 9)  { Tcl_GetIntFromObj (tIntp, objv[8], &LenA); }
   if (objc >= 10) { Tcl_GetIntFromObj (tIntp, objv[9], &LenB); }
 
/* Get the non-array input parameters                                      */

   Oper = Tcl_GetStringFromObj(objv[2], (int *)0);
   if ( Oper[0] == 'R') {
      Op = Oper[1];
      Roff = OffB; 
      OffB = (ByTe_4)(OffB / LenB) * LenB; 
   } else { Op = Oper[0]; } 

   switch (Op) {
      case 'A': 
      case 'D': 
      case 'M': 
      case 'S': 
         XFerR = 0;
      break;
      default:  
         XFerR = 1;
         TmP = LenA * LenB;
         RSize = (Op == 'T') ? TmP : (Op == 'C') ? 3 : LenA;
      break;
    }

/* Now lets to all the necessary mallocs at once                           */

   vLen = (LenA > LenB) ? LenA : LenB;
   Bytes = (2 * vLen + LenA * LenB) * sizeof(ReaL_8);
   if ((memPtr = malloc (Bytes)) == NULL)
     return TCL_ERROR;

/* OK now lets assign all the pointers into that malloc                   */

   V = (ReaL_8 *)memPtr; 
   W = (ReaL_8 *)memPtr + vLen; 
   R = (ReaL_8 *)memPtr + 2 * vLen; 

/* Transfer array data from the TCL array to the C array                   */

   if (TclArrayToC (tIntp, objv[1], (void *)V, LenA, OffA, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }
   if (TclArrayToC (tIntp, objv[3], (void *)W, LenB, OffB, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }

/* Call the C routine                                                       */

   rV = VecMath_C (V, Op, W, R, LenA, LenB, Roff);

/* Return the Resultant Array to a Tcl Array                                */

   if (XFerR) { CArrayToTcl (tIntp, objv[4], R, RSize, OffC, 'D'); } 

/* Get rid of the temporary arrays and return                               */

   free (memPtr);

   objPtr = Tcl_GetObjResult(tIntp);
   Tcl_SetDoubleObj (objPtr, (double)rV);

   return TCL_OK;
}


/* THE C ROUTINE                                                            */
/*                                                                          */
/*  In the C routine operations have character representations.  Some of    */
/*  the reductions from the Tcl Operations include:                         */
/*     'C':  CROSS                                                          */
/*     'D':  DOT                                                            */
/*     'M':  MAG                                                            */
/*     'U':  UNIT                                                           */
/*     'T':  TENSOR                                                         */

ReaL_8 VecMath_C (ReaL_8 *V, ByTe_1 oP, ReaL_8 *W, ReaL_8 *R, ByTe_4 LenA,
                  ByTe_4 LenB, ByTe_4 Roff)
{
    ReaL_8 T = 0.0;
    ByTe_4 I, J, K, L;

    switch (oP) {
      case 'C': 
         R[0] = V[1] * W[2] - V[2] * W[1];
         R[1] = V[2] * W[0] - V[0] * W[2];
         R[2] = V[0] * W[1] - V[1] * W[0];
      break;
      case 'D': 
         for (I = 0; I < LenA; ++I ) { T += V[I] * W[I]; }
      break;
      case 'M': 
         for (I = 0; I < LenA; ++I ) { T += V[I] * V[I]; }
         T = sqrt(T);
      break;
      case 'T': 
         L = 0;
         for (I = 0; I < LenA; ++I ) { 
            for (J = 0; J < LenB; ++J ) { 
               R[L++] = V[I] * W[J];
            }
         }
      break;
      case 'U': 
         for (I = 0; I < LenA; ++I ) { T += V[I] * V[I]; }
         T = sqrt(T);
	 if (T != 0.0 ) {
            for (I = 0; I < LenA; ++I ) { R[I] = V[I] / T; }
         }
      break;
      case '*': 
         L = 0; 
         for (I = 0; I < LenA; ++I ) { 
            K = (I + Roff) % LenB;
            R[L++] = V[I] * W[K];
         }
      break;
      case '/': 
         L = 0; 
         for (I = 0; I < LenA; ++I ) { 
            K = (I + Roff) % LenB;
            R[L++] = V[I] / W[K];
         }
      break;
      case '+': 
         L = 0; 
         for (I = 0; I < LenA; ++I ) { 
            K = (I + Roff) % LenB;
            R[L++] = V[I] + W[K];
         }
      break;
      case '-': 
         L = 0; 
         for (I = 0; I < LenA; ++I ) { 
            K = (I + Roff) % LenB;
            R[L++] = V[I] - W[K];
         }
      break;
    }

    return T;
}
