/*  The C equivalent of TUsolveCubic,  It consists of a TCL interface over  */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  Ax^3 + Bx^2 + Cx + D = 0                                                */
/*                                                                          */
/*  There are 7 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  A                                                          */
/*     OBJV[2]:  B                                                          */
/*     OBJV[3]:  C                                                          */
/*     OBJV[4]:  D                                                          */
/*                                                                          */
/*     The resultant Z value is returned through the procedure              */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int SolveCubicCmd (ClientData cD, Tcl_Interp *interp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   ReaL_8 A, B, C, D, rV[6];
   ReaL_8 *dPt;
   ByTe_4 nC, I;

   void **memPtr = NULL;
   ByTe_4 Bytes;
   Tcl_Obj **lObj, *rObj;


/* Make sure that the minimum number of IO parameters are present */

   if (objc < 4 ) {
     Tcl_WrongNumArgs(interp, 1, objv, "Usage: TUsolveCubic A B C D");
     return TCL_ERROR;
   }

/* Get the non array input parameters                                        */

   Tcl_GetDoubleFromObj (interp, objv[1], &A);
   Tcl_GetDoubleFromObj (interp, objv[2], &B);
   Tcl_GetDoubleFromObj (interp, objv[3], &C);
   Tcl_GetDoubleFromObj (interp, objv[4], &D);

/* Call the  straight C array procedure                                   */

   nC = SolveCubic_C (A, B, C, D, rV); 

/* Gather and return results */

   Bytes = nC * sizeof(*lObj);
   if ((memPtr = malloc (Bytes)) == NULL) { return TCL_ERROR; }
   lObj = (Tcl_Obj **)memPtr;

   dPt = (ReaL_8 *)rV;
   for (I = 0; I < nC; ++I) { lObj[I] = Tcl_NewDoubleObj(*dPt++); }

   rObj = Tcl_GetObjResult(interp);
   Tcl_SetListObj(rObj, (int)nC, lObj);

   if (memPtr != NULL) { free(memPtr); }

   return TCL_OK;
}
   
ByTe_4  SolveCubic_C (ReaL_8 A, ReaL_8 B, ReaL_8 C, ReaL_8 D, ReaL_8 *rV)
{
   ReaL_8 a, b, c;
   ReaL_8 Q, R, Q3, R2;
   ReaL_8 Ang, sQ, rA, TwoPI, T, SgN;
   ReaL_8 r1, r2, r3;
   ReaL_8 i2, i3;
   ByTe_4 nC = 3;

   TwoPI = 2.0 * 3.14159265358979323846;

   a = B/A; 
   b = C/A; 
   c = D/A; 

   Q = (a * a - 3.0 * b) / 9.0;
   R = (2.0 * a * a * a - 9.0 * a * b + 27.0 * c) / 54.0;
   
   Q3 = Q * Q * Q;
   R2 = R * R;

   if ( R2 < Q3 ) {

/* THREE real roots */

      Ang = acos(R / sqrt(Q3));
      sQ  = -2.0 * sqrt(Q);
      rA  = a / 3.0;

      r1 = sQ * cos(Ang/3.0) - rA;
      r2 = sQ * cos((Ang + TwoPI) / 3.0) - rA;
      r3 = sQ * cos((Ang - TwoPI) / 3.0) - rA;

      if (r1 > r2) { T = r2; r2 = r1; r1 = T; } 
      if (r1 > r3) { T = r3; r3 = r1; r1 = T; }
      if (r2 > r3) { T = r3; r3 = r2; r2 = T; } 

      *rV++ = r1;
      *rV++ = r2;
      *rV   = r3;
   } else {
       SgN = (R < 0.0) ? -1.0 : 1.0; 
       A = fabs(R) + sqrt(R * R - Q3);
       A = -SgN * exp(log(fabs(A))/3.0);
       B = (A == 0.0) ? 0.0 : Q / A;
       r1 = A + B - a/3.0;
       r2 = -(A + B)/2.0 - a/3.0;
       r3 = -(A + B)/2.0 - a/3.0;
       if ( A != B ) {
          i2 = sqrt(3.0) * (A - B) / 2.0;
          i3 = -sqrt(3.0) * (A - B) / 2.0;
          *rV++ = r1;
          *rV++ = r2;
          *rV++ = r3;
          *rV++ = i2;
          *rV   = i3;
	  nC = 5;
       } else {
         *rV++ = r1;
         *rV++ = r2;
         *rV   = r3;
       }
   }

   return nC;
}
