/*  The C equivalent of TUpolyExp2D,  It consists of a TCL interface over   */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 7 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  The X value (X)                                            */
/*     OBJV[2]:  The Y value (X)                                            */
/*     OBJV[3]:  The number of coefficients (nCoef)                         */
/*     OBJV[4]:  The coefficent array                                       */
/*     OBJV[5]:  The order of the equation                                  */
/*     OBJV[6]:  Offset to start of coefficients (Off)                      */
/*                                                                          */
/*     The resultant Z value is returned through the procedure              */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int PolyExp2DCmd (ClientData cD, Tcl_Interp *interp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void   *memPtr;
   ReaL_8 *CoeF;
   ReaL_8 X, Lims[2];
   int    nCoef, Off = 0, nOrder;
   double Y;
   ByTe_4 Bytes, LChk = 0;

   Tcl_Obj *objPtr;

/* Make sure that the minimum number of IO parameters are present */

   if (objc < 6 ) {
     Tcl_WrongNumArgs(interp, 1, objv, "Usage: TUpolyExp2D X Y ...");
     return TCL_ERROR;
   }

/* Get the non array input parameters                                        */

   Tcl_GetDoubleFromObj (interp, objv[1], &X);
   Tcl_GetDoubleFromObj (interp, objv[2], &Y);
   Tcl_GetIntFromObj (interp, objv[3], &nCoef);
   Tcl_GetIntFromObj (interp, objv[5], &nOrder);
   if (objc == 7)
      Tcl_GetIntFromObj (interp, objv[6], &Off);
   if ( objc == 9) {
      Tcl_GetDoubleFromObj (interp, objv[7], &Lims[0]);
      Tcl_GetDoubleFromObj (interp, objv[8], &Lims[1]);
      LChk = (Lims[0] == Lims[1]) ? 0 : 1;
   } 

/* Get the array inuts                                                      */

   Bytes = sizeof(ReaL_8) * nCoef;
   if ((memPtr = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   CoeF = (ReaL_8 *)memPtr;

   if (TclArrayToC (interp, objv[4], (void *)CoeF, nCoef, Off, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }
   
/* Call the  straight C array procedure                                   */

   Y = PolyExp2D_C (X, Y, nCoef, CoeF, nOrder, LChk, Lims); 

/* Gather output - free data array - and gone                             */

   free (memPtr);

   objPtr = Tcl_GetObjResult(interp);
   Tcl_SetDoubleObj (objPtr, Y); 
   return TCL_OK;
}
   
ReaL_8  PolyExp2D_C (ReaL_8 X, ReaL_8 Y, int N, ReaL_8 *A, int nOrder, 
                     ByTe_4 LChk, ReaL_8 *Lims)
{
   ReaL_8 Z;
   ByTe_4 J, K, L, M, Base;
   ByTe_4 LenX;
   ReaL_8 TmpXY[N];

   Base = 0;
   LenX = 1;
   TmpXY[0] = 1.0;
   L = 1;
   for (J = 0; J < nOrder; ++J) {
      M = Base;
      for (K = 0; K < LenX; ++K) { TmpXY[L++] = TmpXY[M++] * X; }

      --M;
      TmpXY[L++] = TmpXY[M] * Y;
      Base += LenX;
      ++LenX;
   }

   Z = 0.0;
   for (J = 0; J < N; ++J) { Z += A[J] * TmpXY[J]; }

   if (LChk == 1) {
      if (Z > Lims[0]) { Z = Lims[0]; }
      if (Z < Lims[1]) { Z = Lims[1]; }
   }

   return (Z);
}
