/*  The C equivalent of TUmatrixInv,  It consists of a TCL interface over   */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 7 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  The matrix order (N)                                       */
/*     OBJV[2]:  The matrix array (Mat1)                                    */
/*     OBJV[3]:  Offset into matrix array (B1)                              */
/*     OBJV[4]:  Number of solution vectors (M)                             */
/*     OBJV[5]:  The solution vectors array (Mat2)                          */
/*     OBJV[6]:  Offset into solution vector array (B2)                     */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int MatrixInvCmd (ClientData cD, Tcl_Interp *tIntp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr;
   ReaL_8 *MaT1, *MaT2 = NULL;
   int N, B1 = 0, M = 0, B2 = 0, rV;

   ByTe_4 Bytes;

   Tcl_Obj *objPtr;
   
/* Make sure that the minimum number of IO parameters are present          */

   if (objc < 3 ) {
     Tcl_WrongNumArgs(tIntp, 1, objv, "Usage: TUmatrixInv Order Matrix");
     return TCL_ERROR;
   }

/* Get the constant input parameters                                        */

   Tcl_GetIntFromObj (tIntp, objv[1], &N);
   if (objc >= 4)
      Tcl_GetIntFromObj (tIntp, objv[3], &B1);
   if (objc >= 5)
      Tcl_GetIntFromObj (tIntp, objv[4], &M);
   if (objc >= 7)
      Tcl_GetIntFromObj (tIntp, objv[6], &B2);

/* Get the array inputs                                                     */

   Bytes = sizeof(ReaL_8) * (N * N  + N * M);
   if ((memPtr = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   MaT1 = (ReaL_8 *)memPtr;

   if (TclArrayToC (tIntp, objv[2], (void *)MaT1, N*N, B1, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }

   if (M > 0) {
      MaT2 = (ReaL_8 *)memPtr + N * N;
      if (TclArrayToC (tIntp, objv[5], (void *)MaT2, N*M, B2, 'D') == 0) {
         free (memPtr);
         return TCL_OK;
      }
   }

/* Call the  straight C array procedure                                   */

   MaT1 = (ReaL_8 *)memPtr; 
   if (M > 0) { MaT2 = MaT1 + N * N; } 
   rV = MatrixInv_C (N, MaT1, M, MaT2); 

/* Gather output - free data array - and gone                             */

/* Reconvert Matrix and if necessary the solution vectors to Tcl Arrays   */
 
   CArrayToTcl (tIntp, objv[2], MaT1, N*N, B1, 'D');
   if (M > 0) { CArrayToTcl (tIntp, objv[5], MaT2, M*N, B2, 'D'); }

   free (memPtr);

   objPtr = Tcl_GetObjResult(tIntp);
   Tcl_SetIntObj (objPtr, rV); 
   return TCL_OK;
}

   
/*  This is a Matrix Inversion Code - straight C                            */

ByTe_4 MatrixInv_C (ByTe_4 N, ReaL_8 *Mat1, ByTe_4 M, ReaL_8 *Mat2)
{
   void *memPtr;
   register ReaL_8 *A, *B;
   ReaL_8 BiG, TmpD, pivInv, T, aB;
   ByTe_4 *iPiv, *indxr, *indxc;
   ByTe_4 I, J, K, L, Q, Bytes;
   ByTe_4 iC, iR;
   ByTe_4 P, P1, P2;

   A = Mat1;
   B = Mat2;

   Bytes = sizeof(ByTe_4) * N * 3;
   if ((memPtr = malloc (Bytes)) == NULL)
     return -1;
   iPiv =  (ByTe_4 *)memPtr;
   indxr = (ByTe_4 *)memPtr + N;
   indxc = (ByTe_4 *)memPtr + 2 * N;

   for ( J = 0;  J < N; ) { iPiv[J++] = 0; }

   for ( I = 0;  I < N; ++I ) { 
      BiG = 0.0;
      for ( J = 0;  J < N; ++J ) { 
         if ( iPiv[J] != 1 ) {
             P = N * J;
             for ( K = 0;  K < N; ++K, ++P ) { 
               if ( iPiv[K] == 0 ) {
                  aB = fabs(A[P]);
                  if ( aB >= BiG ) {
                      BiG = aB;
                      iR = J;
                      iC = K;
                  } 
               } else { if ( iPiv[K] > 1 ) { return -1; } }
            }
         }
      }

      ++iPiv[iC];
      if ( iC != iR ) {
         P1 = N * iR;
         P2 = N * iC;
         for ( L = 0;  L < N; ++L, ++P1, ++P2 ) { 
            TmpD = A[P1];
            A[P1] = A[P2];
            A[P2] = TmpD;
         }
         P1 = M * iR;
         P2 = M * iC;
         for ( L = 0;  L < M; ++L, ++P1, ++P2 ) { 
            TmpD = B[P1];
            B[P1] = B[P2];
            B[P2] = TmpD;
         }
      }
      indxr[I] = iR;
      indxc[I] = iC;
      P = N * iC + iC;
      if ( A[P] == 0.0 ) { return -2; }
      pivInv = 1.0 / A[P];

      A[P] = 1.0;
      P = N * iC;
      for ( L = 0;  L < N; ++L, ++P ) { A[P] *= pivInv; }
      P = M * iC;
      for ( L = 0;  L < M; ++L, ++P ) { B[P] *= pivInv; }

      for ( Q = 0;  Q < N; ++Q ) { 
         if ( Q != iC ) {
             P = N * Q + iC;
             T = A[P];
             A[P] = 0.0;
             P = N * Q;
             P1 = N * iC;
             for ( L = 0;  L < N; ++L, ++P, ++P1 ) { A[P] -= A[P1] * T; }
             P = M * Q;
             P1 = M * iC;
             for ( L = 0;  L < M; ++L, ++P, ++P1 ) { B[P] -= B[P1] * T; }
         } 
      }
   }

   for ( L = N -1;  L >= 0; --L ) { 
      if ( indxr[L] != indxc[L] ) {
         for ( K = 0;  K < N; ++K ) { 
             P = N * K + indxr[L];
             P1 = N * K + indxc[L];
             TmpD = A[P1];
             A[P1] = A[P];
             A[P] = TmpD;
         }
      }
   }

   free(memPtr);
   return 1;
}
