/*  The C equivalent of TUgeiMag,  It consists of a TCL interface over      */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 4 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  The GEI vector (GeI)                                       */
/*     OBJV[2]:  The transfer information array (xInfo)                     */
/*           Index 0: Mean Julian Day (MjD)                                 */
/*           Index 1: Greenwich mean Siderial Time (GsT)                    */
/*           Index 2: cos(GsT)                                              */
/*           Index 3: sin(GsT)                                              */
/*     OBJV[3]:  The MaG vector (MaG)                                       */

#include <tcl.h>
#include <stdlib.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int GeiToMagCmd (ClientData cD, Tcl_Interp *tIntp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr = NULL; 
   ReaL_8 *MaG, *GeI, *xInfo;

   ByTe_4 Bytes, rV;

/* Make sure that all the IO parameters are present */

   if (objc != 4 ) {
     Tcl_WrongNumArgs(tIntp, 1, objv, "Usage: TUgeiMAG GEI XInfo MAG");
     return TCL_ERROR;
   }

/* Now lets to all the necessary mallocs at once                          */

   Bytes = 16 * sizeof(ReaL_8);
   if ((memPtr = malloc (Bytes)) == NULL)
     return TCL_ERROR;

/* OK now lets assign all the pointers into that malloc                   */

   GeI =   (ReaL_8 *)memPtr; 
   MaG =   (ReaL_8 *)memPtr + 3; 
   xInfo = (ReaL_8 *)memPtr + 6; 

/* Transfer array data from the TCL array to the C array                   */

   if (TclArrayToC (tIntp, objv[1], (void *)GeI, 3, 0, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }
   if (TclArrayToC (tIntp, objv[2], (void *)xInfo, 10, 0, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }

/* Call the C routine                                                       */

   rV =  GeiToMag_C (GeI, xInfo, MaG);

/* Reconvert to the Grid and Norm C Arrays to Tcl Arrays                    */

   CArrayToTcl (tIntp, objv[3], MaG, 3, 0, 'D'); 

/* Get rid of the temporary array and return                                */

   free (memPtr);
   return TCL_OK;
}


/* THE C ROUTINE                                                            */


ByTe_4 GeiToMag_C (ReaL_8 *GeI, ReaL_8 *xI, ReaL_8 *MaG)
{

    register ReaL_8 *fPt;
    ReaL_8 tX, tY, tZ;
    ReaL_8 X, Y, Z;

    tX =  xI[2] * GeI[0] + xI[3] * GeI[1];
    tY = -xI[3] * GeI[0] + xI[2] * GeI[1];
    tZ = GeI[2];

    X =  xI[5] * tX + xI[6] * tY;
    Y = -xI[6] * tX + xI[5] * tY;
    Z = tZ;

    fPt = MaG;
    *fPt++ = xI[8] * X + xI[9] * Z;
    *fPt++ = Y;
    *fPt   = -xI[9] * X + xI[8] * Z;

    return 1;
}
