/*  The C equivalent of TUdataMqCoef,  It consists of a TCL interface over  */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 10 OBJV elements.  These are:                                 */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  Y data array                                    (Y)        */
/*     OBJV[2]:  Standard Deviation Array                        (SiG)      */
/*     OBJV[3]:  Number of data points in array                  (nP)       */
/*     OBJV[4]:  Fitted/Unfitted Component array                 (iA)       */
/*     OBJV[5]:  Number of independent parameters                (nA)       */
/*     OBJV[6]:  Offset to Curvature matrix in oP                (bG1)      */
/*     OBJV[7]:  Offset to scratch space oP                      (bG2)      */
/*     OBJV[8]:  Function Call                                   (FunC)     */
/*     OBJV[9]:  Input/Output and Scratch                        (oP)       */
/*        Element 0:  Fitting Test - set < 0 for first call to let the      */
/*                    procedure do initializations,  set to 0 for last      */
/*                    call to get the covariance and curvature matrices.    */
/*                    Procedure sets varaible during iterations             */
/*        Element 1:  Current Chi Squared Value                             */
/*        Element 2:  Previous Chi Squared Value                            */
/*        Element 3:  The number of coefficients being fit - computed       */
/*                    during  the first call from iA                        */
/*        Element 4:  Input Element to process                              */
/*        Element 5:  Scratch space starts here.  The first mA values hold  */
/*                    the current guess of the coefficients being used by   */
/*                    the procedure.  The next mA*mA coefficients hold the  */
/*                    Covarience Matrix which only has meaning if Ops(0) =  */
/*                    0. The next mA*mA coefficients hold the Curvature     */
/*                    Matrix which only has meaning if Ops(0) = 0.  After   */
/*                    this pure scratch space                               */

#include <tcl.h>
#include <stdlib.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int DataMqCoefCmd (ClientData cD, Tcl_Interp *tIntp, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr1, *memPtr2; 
   int nP, nA, bG1, bG2;
   ReaL_8 *V, *SiG, *A, *oP;
   ByTe_4 rV, Bytes, nASq, nOp;
   ByTe_1 *FunC, *iA;

/* Make sure that all the IO parameters are present */

   if (objc != 10 ) {
     Tcl_WrongNumArgs(tIntp, 1, objv, "Usage: TUdataMqCoef Y SiG ...");
     return TCL_ERROR;
   }

/* Get the constant input parameters                                        */

   Tcl_GetIntFromObj (tIntp, objv[3], &nP);
   Tcl_GetIntFromObj (tIntp, objv[5], &nA);
   Tcl_GetIntFromObj (tIntp, objv[6], &bG1);
   Tcl_GetIntFromObj (tIntp, objv[7], &bG2);
   FunC = Tcl_GetString (objv[8]);
 
/* Establish the Memory for the needed arrays                               */
 
   nASq = nA * nA;
   nOp = 5 + 2 * nA + 4 * nASq;
   Bytes = sizeof(ReaL_8) * (2 * nP + nOp);
   if ((memPtr1 = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   V = (ReaL_8 *)memPtr1;
   SiG = (ReaL_8 *)memPtr1 + nP;
   A = (ReaL_8 *)memPtr1 + 2 * nP;
   oP = (ReaL_8 *)memPtr1 + 2 * nP + nA;
 
   if (TclArrayToC (tIntp, objv[1], (void *)V, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tIntp, objv[2], (void *)SiG, nP, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }
   if (TclArrayToC (tIntp, objv[9], (void *)oP, nOp, 0, 'D') == 0) {
      free (memPtr1);
      return TCL_OK;
   }

   Bytes = sizeof(ByTe_1) * (nA);
   if ((memPtr2 = malloc (Bytes)) == NULL) {
      free (memPtr1);
      return TCL_ERROR;
   }
   iA = (ByTe_1 *)memPtr2;
 
   if (TclArrayToC (tIntp, objv[5], (void *)iA, nA, 0, 'C') == 0) {
      free (memPtr1);
      free (memPtr2);
      return TCL_OK;
   }

/* Call the C routine                                                       */

   rV =  DataMqCoef_C (tIntp, objv[9], V, SiG, (ByTe_4)nP, iA, (ByTe_4)nA, 
                       (ByTe_4)bG1, (ByTe_4)bG2, FunC, oP);

/* Reconvert the Ops Array to a Tcl Array                                   */

   CArrayToTcl (tIntp, objv[8], (void *)oP, nOp, 0, 'D'); 

/* Get rid of the temporary array and return                                */

   free (memPtr1);
   free (memPtr2);
   return TCL_OK;

}


/* THE C ROUTINE                                                            */


ByTe_4 DataMqCoef_C (Tcl_Interp *tI, Tcl_Obj *oV, ReaL_8 *Y, ReaL_8 *SiG, 
                     ByTe_4 nP, ByTe_1 *iA, ByTe_4 nA, ByTe_4 bG1, ByTe_4 bG2, 
                     ByTe_1 *FunC, ReaL_8 *oP)
{
   void *memPtr;
   register ByTe_4 I, J, K, L, M;
   ReaL_8 *d1, *dEnD;
   ReaL_8 SiG2i, dY, *dYdA;
   ReaL_8 rV, WgT;
   ByTe_4 EnD, nC, Bytes, P, P1, XFer;
   int Flg = TCL_LEAVE_ERR_MSG;
   char *vName, TexT[10];
   Tcl_Obj *rVO, *rV1;

   rV1 = Tcl_NewObj();

   vName = Tcl_GetString(oV);
   XFer = 5 + nA;
   nC = (ByTe_4)oP[3];
   Bytes = sizeof(ReaL_8) * (nA + 3);
   if ((memPtr = malloc (Bytes)) == NULL)
      return TCL_ERROR;
   dYdA = (ReaL_8 *)memPtr;

   d1 = &oP[bG1];
   dEnD = d1 + nC * nC;  
   for ( ; d1 < dEnD; ) { *d1++ = 0.0; }

   d1 = &oP[bG2];
   dEnD = d1 + nC;  
   for ( ; d1 < dEnD; ) { *d1++ = 0.0; }

   oP[1] = 0.0;
   CArrayToTcl (tI, oV, (void *)oP, XFer, 0, 'D'); 

   for ( I = 0; I < nP; ++I) {
      oP[4] = I;
      rV1 = Tcl_DuplicateObj(rV1);
      Tcl_SetIntObj (rV1, I);
      Tcl_SetVar2Ex(tI, vName, "4", rV1, Flg);
      Tcl_Eval(tI, FunC);

      if ( (rVO = Tcl_GetVar2Ex(tI, "FuncRet", (char *)0, Flg)) != NULL) {
         Tcl_GetDoubleFromObj (tI, rVO, &rV);
      } else { return -1; }

      for (J = 0; J < nA; ++J ) {
         sprintf (TexT, "%d", J);
         if ( (rVO = Tcl_GetVar2Ex(tI, "dYdA", TexT, Flg)) != NULL) {
            Tcl_GetDoubleFromObj (tI, rVO, &dYdA[J]);
         } else { return -1; }
      }

      SiG2i = 1.0 / (SiG[I] * SiG[I]);
      dY = Y[I] - rV;
      EnD = 1;
      J = 0;
      K = bG2;
      for (L = 0; L < nA; ++L, ++EnD ) {
         if ( iA[L] == 1 ) {
            WgT = dYdA[L] * SiG2i;
            P = bG1 + J * nC;
            for (M = 0; M < EnD; ++M ) {
               if ( iA[M] == 1 ) {
                  oP[P] = oP[P] + WgT * dYdA[M];
                  ++P;
               }
            }
            oP[K] = oP[K] + dY * WgT;
            ++J;
            ++K;
         }
      }
      oP[1] += dY * dY * SiG2i;
   }
 
   for (J = 1; J < nC; ++J ) {
      P = bG1 + J * nC;
      for (K = 0; K < J; ++K, ++P ) {
         P1 = bG1 + K * nC + J;
         oP[P1] = oP[P];
      }
   }

   free (memPtr);
   return 1;
}
