/*  The C equivalent of TUdataGrid,  It consists of a TCL interface over    */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 12 OBJV elements.  These are:                                 */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  Number of data points to put in grid (nP)                  */
/*     OBJV[2]:  X hold value (xHold)                                       */
/*     OBJV[3]:  Array of lower edge X values (X1)                          */
/*     OBJV[4]:  Array of upper edge X values (X2)                          */
/*     OBJV[5]:  Y hold value (yHold)                                       */
/*     OBJV[6]:  Array of lower edge Y values (Y1)                          */
/*     OBJV[7]:  Array of upper edge Y values (Y2)                          */
/*     OBJV[8]:  Intensity array (Vv)                                       */
/*     OBJV[9]:  The grid array (Grid)                                      */
/*     OBJV[10]: The grid normalization array (Norm)                        */
/*     OBJV[11]:  Grid Info Array                                           */
/*                 0:  Beginning X grid position                            */
/*                 1:  Ending X grid position                               */
/*                 2:  Beginning Y grid position                            */
/*                 3:  Ending Y grid position                               */
/*                 4:  Number of grids along X                              */
/*                 5:  Number of grids along Y                              */
/*                 6:  Use POINT or BAND method to store X data             */
/*                 7:  Use POINT or BAND method to store Y data             */
/*                 8:  Storage Method ROW or COLUMN                         */
/*                 9:  IGNORE or KEEP Zeros when averaging                  */
/*                10:  NEW: initialize grid                                 */
/*                     ADD: add data to grid                                */
/*                     END: normalize grid                                  */
/*                     or any combination as ADDEND or NEWADDEND            */
/*                11:  Value to set unfilled grids to                       */
/*                12:  Bad grid value                                       */
/*                13:  Remove data below this value                         */
/*                14:  Remove data above this value                         */
/*                15:  Cyclic in X (YES/NO)                                 */
/*                16:  Cyclic in Y (YES/NO)                                 */
/*                17:  X Axis scaling (LINEAR/LOG)                          */
/*                18:  X Axis scaling (LINEAR/LOG)                          */
/*                                                                          */
/*  For passage into the C routine the ASCII grid info values are           */
/*  pre-processed to integer values.                                        */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int DataGridCmd (ClientData cD, Tcl_Interp *tI, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr = NULL; 
   ReaL_8 *Grid, *Norm, *X1, *X2, *Y1, *Y2, *Vv;
   ReaL_8 gI[19]; 

   int    nP, xHold, yHold;
   ByTe_4 nX, nY, Elem1, Elem2, tGrids;
   ByTe_4 Bytes, Offset, rV;
   ByTe_1 *cPt, *gN, *nN, *Value;
   int    sLen;

   int ActioN;
   register ByTe_4 I;

   int Flg = TCL_LEAVE_ERR_MSG;
   Tcl_Obj *rVO;

   char *aName, aName1[40], aName2[40], Index[10];

/* Make sure that all the IO parameters are present */

   if (objc != 12 ) {
     Tcl_WrongNumArgs(tI, 1, objv, "Usage: TUdataGrid nP xHold X1 X2 ...");
     return TCL_ERROR;
   }

/* Get the non-array inputs                                                 */
 
   Tcl_GetIntFromObj (tI, objv[1], &nP);
   Tcl_GetIntFromObj (tI, objv[2], &xHold);
   Tcl_GetIntFromObj (tI, objv[5], &yHold);

/* Get the grid array inputs - must be handled individually since they are  */
/* a mixture of values and strings                                          */

   aName = Tcl_GetStringFromObj(objv[11],(int *)0);
   for (I = 0; I <= 18; ++I) {
      sprintf (Index, "%d", I);                                            
      if ( (rVO = Tcl_GetVar2Ex(tI, aName, Index, Flg)) != NULL) {
         switch (I) {
            case 0:
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
            case 11:
            case 12:
            case 13:
            case 14:
	      Tcl_GetDoubleFromObj (tI, rVO, &gI[I]);
            break;
            case 6:
            case 7:
	      Value = Tcl_GetStringFromObj (rVO, (int *)0);
              gI[I] = (strcmp("POINT", Value) == 0) ? 0 : 1;
            break;
            case 8:
	      Value = Tcl_GetStringFromObj (rVO, (int *)0);
              gI[8] = (strcmp("COLUMN", Value) == 0) ? 1 : 0;
            break;
            case 9:
	      Value = Tcl_GetStringFromObj (rVO, (int *)0);
              gI[9] = (strcmp("IGNORE", Value) == 0) ? 1 : 0;
            break;
            case 10:
              ActioN = 0;
	      Value = Tcl_GetStringFromObj (rVO, &sLen);
              cPt = Value;
              while (sLen > 0) {
                 if (strncmp("NEW", cPt, 3) == 0) { ActioN += 4; }
                 if (strncmp("ADD", cPt, 3) == 0) { ActioN += 2; }
                 if (strncmp("END", cPt, 3) == 0) { ActioN += 1; }
                 sLen -= 4;
                 cPt += 4;
              }
              gI[10] = ActioN;
            break;
            case 15:
            case 16:
	      Value = Tcl_GetStringFromObj (rVO, (int *)0);
              gI[I] = (strcmp("YES", Value) == 0) ? 1 : 0;
            break;
            case 17:
            case 18:
	      Value = Tcl_GetStringFromObj (rVO, (int *)0);
              gI[I] = (strcmp("LOG", Value) == 0) ? 1 : 0;
            break;
         }
      } 
   }

/*  DO all the necessary mallocs at once.  The Mesh and Normalization    */
/*     arrays are always malloced but the data arrays are only needed    */
/*     if data is being added.                                           */

   tGrids = (ByTe_4)(gI[4] * gI[5]);
   Bytes = sizeof(ReaL_8) * (2 * tGrids);
 
   if ((ActioN & 2) == 2 ) {
      Bytes += sizeof(ReaL_8) * nP;

      strcpy (aName1, Tcl_GetString(objv[3])); 
      strcpy (aName2, Tcl_GetString(objv[4])); 
      nX = (strcmp(aName1, aName2) == 0) ? 1 : 2;

      Elem1 = (xHold < 0) ? -xHold : nP / xHold;
      Bytes += sizeof(ReaL_8) * nX * Elem1;

      strcpy (aName1, Tcl_GetString(objv[6])); 
      strcpy (aName2, Tcl_GetString(objv[7])); 
      nY = (strcmp(aName1, aName2) == 0) ? 1 : 2;
      Elem2 = (yHold < 0) ? -yHold : nP / yHold;
      Bytes += sizeof(ReaL_8) * nY * Elem2;
   }

   if ((memPtr = malloc (Bytes)) == NULL) { return TCL_ERROR; }

/* OK now lets assign all the pointers into that malloc                   */

   Offset = 0;
   if ((ActioN & 2) == 2 ) {
      X1 = (ReaL_8 *)memPtr; 
      Offset += Elem1 * ( nX - 1); 
      X2 = (ReaL_8 *)memPtr + Offset; 
      Offset += Elem1; 
      Y1 = (ReaL_8 *)memPtr + Offset; 
      Offset += Elem2 * ( nY - 1); 
      Y2 = (ReaL_8 *)memPtr + Offset; 
      Offset += Elem2; 
      Vv = (ReaL_8 *)memPtr + Offset; 
      Offset += nP;
   }
   Grid = (ReaL_8 *)memPtr + Offset; 
   Offset += tGrids;
   Norm = (ReaL_8 *)memPtr + Offset; 

/* If we need to transfer data from the TCL arrays to the C arrays do    */
/* that now                                                              */

   if ((ActioN & 2) == 2 ) {
      if (TclArrayToC (tI, objv[3], (void *)X1, Elem1, 0, 'D') == 0) {
         free (memPtr);
         return TCL_OK;
      }
      if (TclArrayToC (tI, objv[4], (void *)X2, Elem1, 0, 'D') == 0) {
         free (memPtr);
         return TCL_OK;
      }
      if (TclArrayToC (tI, objv[6], (void *)Y1, Elem2, 0, 'D') == 0) {
         free (memPtr);
         return TCL_OK;
      }
      if (TclArrayToC (tI, objv[7], (void *)Y2, Elem2, 0, 'D') == 0) {
         free (memPtr);
         return TCL_OK;
      }
      if (TclArrayToC (tI, objv[8], (void *)Vv, nP, 0, 'D') == 0) {
         free (memPtr);
         return TCL_OK;
      }
   }

/* Call the C routine                                                       */

   gN = Tcl_GetString(objv[9]);
   nN = Tcl_GetString(objv[10]);
   rV =  DataGrid_C (tI, nP, xHold, X1, X2, yHold, Y1, Y2, Vv, gN, nN, gI);

/* Get rid of the temporary array and return                                */

   free (memPtr);
   return TCL_OK;
}


/* THE C ROUTINE                                                            */


ByTe_4 DataGrid_C (Tcl_Interp *tI,  int nP, int xHold, ReaL_8 *Xa, ReaL_8 *Xb, 
                           int yHold, ReaL_8 *Ya, ReaL_8 *Yb, 
                           ReaL_8 *V, ByTe_1 *gN, ByTe_1 *nN, ReaL_8 *gI)
{
   ReaL_8 IncX, IncY;
   ReaL_8 cFb, cFe, rFb, rFe; 
   ReaL_8 cTmPb, cTmPe, rTmPb, rTmPe;
   ReaL_8 Frac, F1, F2; 
   ReaL_8 gR, nO, aX, bX, aY, bY; 
   register ByTe_4 I, J, K;
   ByTe_4 TotGrids, nXMesh, nYMesh;
   ByTe_4 xI, yI, Pix, xWrap, yWrap;
   ByTe_4 LastX, LastY;
   ByTe_4 cBeg, cEnd, rBeg, rEnd, pcEnd, prEnd;
   ByTe_1 ActioN, NoeFrac, NobFrac;
   ByTe_1 Index[50], diRx, diRy;
   Tcl_Obj *rVO, *rV1;

   int Flg = TCL_LEAVE_ERR_MSG;

   rV1 = Tcl_NewObj();

/* GET the total number of grids in the mesh and the Action to be taken    */

   TotGrids = (ByTe_4)(gI[4] * gI[5]);

/* GET what needs to be done                                               */

   ActioN = (ByTe_1)gI[10];

/* IF this is a new grid then initialize it.  If only initialization is    */
/*    required then split                                                  */

   if ((ActioN & 4) == 4)  { 
      F1 = 0.0;
      for (I = 0; I < TotGrids; ++I ) {
         sprintf(Index, "%d", I);
	 rV1 = Tcl_DuplicateObj(rV1);
	 Tcl_SetDoubleObj (rV1, F1);
         Tcl_SetVar2Ex(tI, gN, Index, rV1, Flg);
         Tcl_SetVar2Ex(tI, nN, Index, rV1, Flg);
      }
      if (ActioN == 4) { return 1; } 
   }

   if ((ActioN & 2) == 2)  { 

/* ESTABLISH if grid axes are increasing or decreasing                    */

      diRx = (gI[0] > gI[1]) ? 0 : 1;
      diRy = (gI[2] > gI[3]) ? 0 : 1;

/* ESTABLISH some integer Mesh data and set up the X and Y grid sizes in  */
/*   the matrix. The increment depends on the axis scaling.               */
 
      nXMesh = (ByTe_4)gI[4];
      nYMesh = (ByTe_4)gI[5];
      LastX = nXMesh - 1;
      LastY = nYMesh - 1;

      if ( gI[17] > 0.5 ) {
         gI[0] = log10(gI[0]);
         gI[1] = log10(gI[1]);
      }
      if ( gI[18] > 0.5 ) {
         gI[2] = log10(gI[2]);
         gI[3] = log10(gI[3]);
      }

      IncX = (gI[1] - gI[0]) / gI[4];
      IncY = (gI[3] - gI[2]) / gI[5];

/* Set repeats if needed                                                  */

      if (xHold < 0) {
         xWrap = -xHold;
	 xHold = 1;
      } else { xWrap = nP; }

      if (yHold < 0) {
         yWrap = -yHold;
	 yHold = 1;
      } else { yWrap = nP; }

/* OK now loop over the data                                              */

      for (I = 0; I < nP; ++I ) {

/* DON'T save zero data if we are not supposed to or if the data is       */
/* outside the set limits                                                 */

         if ((gI[9] > 0.5) && (V[I] == 0.0))  { continue; } 
         if ( (V[I] < gI[13]) || (V[I] > gI[14]) ) { continue; }

/* SET the the location of the data to grab                               */

         xI = (I / xHold) % xWrap;
         yI = (I / yHold) % yWrap;

/* STORE X data within passed BAND or as a POINT                          */

         if ( gI[17] > 0.5 ) {
            if ( diRx ) {
               aX = log10(Xa[xI]);
               bX = log10(Xb[xI]);
            } else { aX = log10(Xb[xI]) ; bX = log10(Xa[xI]); }
         } else {
            if ( diRx ) {
               aX = Xa[xI];
               bX = Xb[xI];
            } else { aX = Xb[xI] ; bX = Xa[xI]; }
	 }

         cTmPb = (aX - gI[0]) / IncX;
         cTmPe = (bX - gI[0]) / IncX;
         cFb = 1.0;
         cFe = 1.0;
         if ( gI[6] < 0.5 ) {
            cTmPb = (cTmPb + cTmPe)/2.0;
            if ( (cTmPb < 0.0) || (cTmPb >= gI[4]) ) { continue; }
            cTmPe = cTmPb;
            cBeg = (ByTe_4)(cTmPb);
            cEnd = (ByTe_4)(cTmPe);
         } else {
            NobFrac = 0;
            NoeFrac = 0;
            if ( cTmPb < 0.0    ) { cTmPb = 0.0; NobFrac = 1; }
            if ( cTmPe >= gI[4] ) { cTmPe = LastX; NoeFrac = 1; }
            if ( cTmPb > cTmPe ) { continue; } 
            cBeg = (ByTe_4)(cTmPb);
            cEnd = (ByTe_4)(cTmPe);
            if (cBeg != cEnd) {
               if ( !NobFrac ) {
                  cFb = ((gI[0] + (cBeg + 1) * IncX) - aX) / IncX;
               }
               if ( !NoeFrac ) {
                  cFe = (bX - (gI[0] + cEnd * IncX)) / IncX;
               }
            }
         }

         pcEnd = cEnd - 1;

/* NOW do the same thing but for the Y values associated with the data. */

         if ( gI[18] > 0.5 ) {
            if ( diRy ) {
               aY = log10(Ya[yI]);
               bY = log10(Yb[yI]);
            } else { aY = log10(Yb[yI]) ; bY = log10(Ya[yI]); }
         } else {
            if ( diRy ) {
               aY = Ya[yI];
               bY = Yb[yI];
            } else { aY = Yb[yI] ; bY = Ya[yI]; }
	 }

         rTmPb = (aY - gI[2]) / IncY;
         rTmPe = (bY - gI[2]) / IncY;
         rFb = 1.0;
         rFe = 1.0;
         if ( gI[7] < 0.5 ) {
            rTmPb = (rTmPb + rTmPe)/2.0;
            if ( (rTmPb < 0.0) || (rTmPb >= gI[5]) ) { continue; }
            rTmPe = rTmPb;
            rBeg = (ByTe_4)(rTmPb);
            rEnd = (ByTe_4)(rTmPe);
         } else {
            NobFrac = 0;
            NoeFrac = 0;
            if ( rTmPb < 0.0   ) { rTmPb = 0.0; NobFrac = 1; }
            if ( rTmPe >= gI[5]) { rTmPe = LastY; NoeFrac = 1; }
            if ( rTmPb > rTmPe ) { continue; }
            rBeg = (ByTe_4)(rTmPb);
            rEnd = (ByTe_4)(rTmPe);
            if (rBeg != rEnd) {
               if ( !NobFrac ) {
                  rFb = ((gI[2] + (rBeg + 1) * IncY) - aY) / IncY;
               }
               if ( !NoeFrac ) {
                  rFe = (bY - (gI[2] + rEnd * IncY)) / IncY;
               }
            }
         }

        prEnd = rEnd - 1;

/* NOW fill in the matrix.  The fill is done either by stacking the      */ 
/* columns to form the 1D output matrix or by stacking the rows.         */

         if (gI[8] > 0.5) { 
             F1 = cFb; 
             for (J = cBeg; J <= cEnd; ++J ) {
                Pix = (ByTe_4)(J * nYMesh + rBeg);
                F2 = rFb; 
                for (K = rBeg; K <= rEnd; ++K, ++Pix ) {
                   Frac = F1 * F2;
                   sprintf(Index, "%d", Pix);

                   rVO = Tcl_GetVar2Ex(tI, gN, Index, Flg);
	           Tcl_GetDoubleFromObj (tI, rVO, &gR);
                   rVO = Tcl_GetVar2Ex(tI, nN, Index, Flg);
	           Tcl_GetDoubleFromObj (tI, rVO, &nO);

                   gR += V[I] * Frac; 
	           rV1 = Tcl_DuplicateObj(rV1);
	           Tcl_SetDoubleObj (rV1, gR);
                   Tcl_SetVar2Ex(tI, gN, Index, rV1, Flg);

                   nO += Frac;
	           rV1 = Tcl_DuplicateObj(rV1);
	           Tcl_SetDoubleObj (rV1, nO);
                   Tcl_SetVar2Ex(tI, nN, Index, rV1, Flg);

                   F2 = (K == prEnd ) ? rFe : 1.0;
                }
                F1 = (J == pcEnd ) ? cFe : 1.0;
             }
          } else {
             F1 = rFb; 
             for (J = rBeg; J <= rEnd; ++J ) {
                Pix = (ByTe_4)(J * nXMesh + cBeg);
                F2 = cFb; 
                for (K = cBeg; K <= cEnd; ++K, ++Pix ) {
                   Frac = F1 * F2;
                   sprintf(Index, "%d", Pix);

                   rVO = Tcl_GetVar2Ex(tI, gN, Index, Flg);
	           Tcl_GetDoubleFromObj (tI, rVO, &gR);
                   rVO = Tcl_GetVar2Ex(tI, nN, Index, Flg);
	           Tcl_GetDoubleFromObj (tI, rVO, &nO);
                   
                   gR += V[I] * Frac; 
	           rV1 = Tcl_DuplicateObj(rV1);
	           Tcl_SetDoubleObj (rV1, gR);
                   Tcl_SetVar2Ex(tI, gN, Index, rV1, Flg);

                   nO += Frac;
	           rV1 = Tcl_DuplicateObj(rV1);
	           Tcl_SetDoubleObj (rV1, nO);
                   Tcl_SetVar2Ex(tI, nN, Index, rV1, Flg);

                   F2 = (K == pcEnd ) ? cFe : 1.0;
                }
                F1 = (J == prEnd ) ? rFe : 1.0;
             }
          }
      }
   }

/* NOW if we are closing things out then normalize the matrix             */

   if ((ActioN & 1) == 1)  { 
      for (I = 0; I < TotGrids; ++I ) {
         sprintf(Index, "%d", I);
         rVO = Tcl_GetVar2Ex(tI, gN, Index, Flg);
	 Tcl_GetDoubleFromObj (tI, rVO, &gR);
         rVO = Tcl_GetVar2Ex(tI, nN, Index, Flg);
	 Tcl_GetDoubleFromObj (tI, rVO, &nO);

         gR = ( nO > 0.0 ) ? gR / nO : gI[11];

	 rV1 = Tcl_DuplicateObj(rV1);
	 Tcl_SetDoubleObj (rV1, gR);
         Tcl_SetVar2Ex(tI, gN, Index, rV1, Flg);
      }
   }

   return 1;
}
