/*  The C equivalent of TUvecMath,  It consists of a TCL interface over     */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 10 OBJV elements.  These are:                                 */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  Input Array 1                                              */
/*     OBJV[2]:  Operation                                                  */
/*     OBJV[3]:  Input Array 2 (or Constant)                                */
/*     OBJV[4]:  Output Array                                               */
/*     OBJV[5]:  Offset into first vector                                   */
/*     OBJV[6]:  Offset into second vector                                  */
/*     OBJV[7]:  Offset into output vector array                            */
/*     OBJV[8]:  Length of first vector                                     */
/*     OBJV[9]:  Length of second vector                                    */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TutilAnsi.h"

int ArrayMathCmd (ClientData cD, Tcl_Interp *tI, 
                                int objc, Tcl_Obj *CONST *objv) 
{
   void *memPtr; 
   ReaL_8 *V, *W, *R, rV, Cnst;
   ByTe_4 NumA, Bytes, oL;
   ByTe_1 Op, *Oper, XFerR, IsArray, NeedsTwo;
   char  *aName;
   int Len, OffA = 0, OffB = 0, OffC = 0;
   int dA = 1, dB = 1, dC = 1;
   int Flg = TCL_LEAVE_ERR_MSG;
   Tcl_Obj *objPtr, *rVO;  

/* Make sure that all the IO parameters are present */

   if (objc < 6 ) {
     Tcl_WrongNumArgs(tI, 1, objv, "Usage: TUarrayMath A1 Op A2 A Len");
     return TCL_ERROR;
   }

/* Get the input parameters which may or may not be present                */
      
   Tcl_GetIntFromObj (tI, objv[5], &Len); 
   if (objc >= 7)  { Tcl_GetIntFromObj (tI, objv[6], &OffA); }
   if (objc >= 8)  { Tcl_GetIntFromObj (tI, objv[7], &OffB); }
   if (objc >= 9)  { Tcl_GetIntFromObj (tI, objv[8], &OffC); }
   if (objc >= 10) { Tcl_GetIntFromObj (tI, objv[9],  &dA); }
   if (objc >= 11) { Tcl_GetIntFromObj (tI, objv[10], &dB); }
   if (objc >= 12) { Tcl_GetIntFromObj (tI, objv[11], &dC); }
 
/* Get the Operation input parameter and then convert it to a single    */
/* character which will be passed to the C routine                      */

   Oper = Tcl_GetString(objv[2]);
   oL = strlen(Oper);
   switch (Oper[0]) {
      case 'A':                       /* AVG ALOG ALOG10 ATAN ATAND */
        if (oL < 4) {                 /* ACOS ACOSD                 */
            Op = Oper[0];
        } else {
            if (Oper[1] == 'C') {
               Op = (oL == 5) ? '1' : '0';
            } else if (Oper[1] == 'L') {
               Op = (oL == 4) ? 'd' : 'e';
	    } else { Op = (oL == 5) ? '3' : '2'; }
        }
      break;
      case 'C': 
        Op = (oL == 4) ? '6' : '7'; 
      break;
      case 'L': 
         Op = (oL == 4) ? '5' : '4';
      break;
      case 'S': 
        if (oL == 3) {
          Op = (Oper[1] == 'U') ? 'S' : '9'; 
        } else {
          Op = (Oper[1] == 'Q') ? 'c' : '8'; 
        }
      break;
      case 'T': 
        Op = (oL == 4) ? 'a' : 'b'; 
      break;
      default:  
         Op = Oper[0];
      break;
    }

/* Decide if there is a resultant array.  If so we will need to transfer it */
/* back into a Tcl array.  At the same time are there two inputs?           */

   switch (Op) {
      case 'A': 
      case 'S': 
         XFerR = 0;
         NeedsTwo = 0;
      break;
      case 'V': 
         XFerR = 0;
         NeedsTwo = 1;
      break;
      default:  
         XFerR = 1;
         NeedsTwo = 1;
      break;
    }

/* Check to see if the second array is really an array or just a constant */

   IsArray = 0;
   if (XFerR == 1) {
      aName = Tcl_GetString(objv[3]);
      if ( (rVO = Tcl_GetVar2Ex(tI, aName, "0", Flg)) != NULL) { 
          IsArray = 1; 
      }
   } 

/* If second input is not an array get the constant                       */

   if (!IsArray && NeedsTwo) { 
      aName = Tcl_GetString(objv[3]);
      if ( (rVO = Tcl_GetVar2Ex(tI, aName, (char *)0, Flg)) == NULL) { 
         Tcl_GetDouble (tI, aName, &Cnst);
      } else { Tcl_GetDoubleFromObj (tI, rVO, &Cnst); }
   }

/* Now lets to all the necessary mallocs at once                          */

   NumA = (XFerR == 0) ? 1 : (IsArray == 1) ? 3 : 2;
   Bytes = (NumA * Len) * sizeof(ReaL_8);
   if ((memPtr = malloc (Bytes)) == NULL)
     return TCL_ERROR;


/* OK set the array pointers and tranfer the Tcl arrays to the C arrays.  */
/* Only need to convert the data over the range which will be used by     */
/* the C routine.                                                         */

   V = (ReaL_8 *)memPtr; 

   if (TclArrayToC (tI, objv[1], (void *)V, Len, OffA, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }

   if (IsArray == 1) {
      W = (ReaL_8 *)memPtr + Len; 
      if (TclArrayToC (tI, objv[3], (void *)W, Len, OffB, 'D') == 0) {
         free (memPtr);
         return TCL_OK;
      }
   } 

/* Call the C routine                                                       */

   if (IsArray) {
      if (XFerR == 1 ) { R = (ReaL_8 *)memPtr + 2 * Len; } 
      rV = ArrayMath_C (V, Op, W, R, Len, dA, dB, dC);
   } else {
      if (XFerR == 1 ) { R = (ReaL_8 *)memPtr + Len; } 
      rV = ArrayMath_C (V, Op, &Cnst, R, Len, dA, 0, dC);
   }

/* Return the Resultant Array to a Tcl Array                                */

   if (XFerR) { CArrayToTcl (tI, objv[4], R, Len, OffC, 'D'); } 

/* Get rid of the temporary arrays and return                               */

   free (memPtr);

   objPtr = Tcl_GetObjResult(tI);
   Tcl_SetDoubleObj (objPtr, (double)rV);

   return TCL_OK;
}


/* THE C ROUTINE                                                            */
/*                                                                          */
/*  In the C routine operations have character representations.  Some of    */
/*  the reductions from the Tcl Operations include:                         */
/*     'A':  AVG                                                            */
/*     '0':  ACOS in radians                                                */
/*     '1':  ACOS in degrees                                                */
/*     '2':  ATAN2 in radians                                               */
/*     '3':  ATAN2 in degrees                                               */
/*     '4':  LOG10                                                          */
/*     '5':  LOGE                                                           */
/*     '6':  COSD                                                           */
/*     '7':  COS                                                            */
/*     '8':  SIND                                                           */
/*     '9':  SIN                                                            */
/*     'a':  TAND                                                           */
/*     'b':  TAN                                                            */
/*     'c':  SQRT                                                           */
/*     'd':  ALOG                                                           */
/*     'e':  ALOG10                                                         */
/*     'S':  SUM                                                            */
/*     'V':  VAR                                                            */

ReaL_8 ArrayMath_C (ReaL_8 *V, ByTe_1 oP, ReaL_8 *W, ReaL_8 *R, ByTe_4 Len,
                    ByTe_4 dA, ByTe_4 dB, ByTe_4 dC)
{
    ReaL_8 T = 0.0, RtoD, Q;
    register ByTe_4 I = 0, J = 0, K = 0;
    ByTe_4 Cnt;

    switch (oP) {
      case '0': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = acos(V[I]); 
         }
      break;
      case '1': 
         RtoD = 180.0 / 3.14159265358979323846;
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = RtoD * acos(V[I]); 
         }
      break;
      case '2': 
         for ( ; I < Len; I += dA, J += dB, K += dC ) { 
            R[K] = atan2(W[J], V[I]); 
         }
      break;
      case '3': 
         RtoD = 180.0 / 3.14159265358979323846;
         for ( ; I < Len; I += dA, J += dB, K += dC ) { 
            R[K] = RtoD * atan2(W[J], V[I]); 
         }
      break;
      case '4': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = (V[I] > 0.0 ) ? log10(V[I]) : *W; 
         }
      break;
      case '5': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = (V[I] > 0.0 ) ? log(V[I]) : *W; 
         }
      break;
      case '6': 
         RtoD = 180.0 / 3.14159265358979323846;
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = cos(V[I] / RtoD); 
         }
      break;
      case '7': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = cos(V[I]); 
         }
      break;
      case '8': 
         RtoD = 180.0 / 3.14159265358979323846;
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = sin(V[I] / RtoD); 
         }
      break;
      case '9': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = sin(V[I]); 
         }
      break;
      case 'a': 
         RtoD = 180.0 / 3.14159265358979323846;
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = tan(V[I] / RtoD); 
         }
      break;
      case 'b': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = tan(V[I]); 
         }
      break;
      case 'c': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = sqrt(V[I]); 
         }
      break;
      case 'd': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = exp(V[I]); 
         }
      break;
      case 'e': 
         for ( ; I < Len; I += dA, K += dC ) { 
            R[K] = pow(10.0, V[I]); 
         }
      break;
      case 'A': 
         for (Cnt = 0; I < Len; ++Cnt, I += dA ) { T += V[I]; }
         T /= (ReaL_8)Cnt;
      break;
      case 'S': 
         for ( ; I < Len; I += dA ) { T += V[I]; }
      break;
      case 'V': 
         for (Cnt = 0; I < Len; ++Cnt, I += dA ) { 
            Q = V[I] - *W;
            T += Q * Q;
         }
         T /= (ReaL_8)(Cnt - 1);
      break;
      case '*': 
         for ( ; I < Len; I += dA, J += dB, K += dC ) { 
            R[K] = V[I] * W[J];
         }
      break;
      case '/': 
         for ( ; I < Len; I += dA, J += dB, K += dC ) { 
            R[K] = V[I] / W[J];
         }
      break;
      case '+': 
         for ( ; I < Len; I += dA, J += dB, K += dC ) { 
            R[K] = V[I] + W[J];
         }
      break;
      case '-': 
         for ( ; I < Len; I += dA, J += dB, K += dC ) { 
            R[K] = V[I] - W[J];
         }
      break;
    }

    return T;
}
