# SWAP the data bytes from big endian to little endian or vice versa.
#
# INPUTS:
#   HdR:   UDF Header file
#   DaTa:  UDF Data file
#   vInfo: VIDF info neeeded in the swap process
#   ibF:   Input byte format
#          L - little endian      
#          B - big endian      
#   obF:   Input byte format
#          L - little endian      
#          B - big endian      
#
#  CHANGED: Aug 17 2005
#      To speed things up I changed the code to only go through the loop
#      to compute the number of bytes in each data section when either
#      TotC, NsmP, or NseN changed in the header read.  Also added a
#      condition to 0 out the general ancillary bytes if sS > 0
#  CHANGED: July 19 2007
#      Added old IDFS check to see if sensor size or sensor anc word lengths
#      need to be made equal
#  CHANGED: Jan 11 2010
#      Added check to see if this is a CLUSTER CPXP1L file and if the first
#      nSS value is good.  These are occasionally garbage.  This may occur
#      in CPX1L virtuals also and will check these also if so.

package provide TclUDFdB 1.0

proc DBswapData { HdR DaTa vInfo iBf oBf } {
   global env 

   upvar $vInfo vI

# SET the input and output write formats

   if { $iBf == "L" } {
      set iB4 i ; set iB2 s
   } else { set iB4 I ; set iB2 S }

   if { $oBf == "L" } {
      set oB4 i ; set oB2 s
   } else { set oB4 I ; set oB2 S }

# OPEN the header file as read and the data file as read/write.  The header
#   file has already been swapped so its already in output format.

   if [catch {open $HdR r} fh] {
      puts stderr "DBswapData: Can't Open $HdR"
      return -1
   }
   if [catch {open $DaTa r+} fd] {
      puts stderr "DBswapData: Can't Open $DaTa"
      return -1
   }
   fconfigure $fh -translation binary
   fconfigure $fd -translation binary

# GET the file size.  Need this to know when the file has been processed.

   set fS [file size $DaTa]

# DETERMINE the length of the SENSOR SET OFFSET ARRAY and the number of 
#   bytes in the data record up to the data array

   set N $vI(MaxNss)
   set B [expr 4 * ($N + 4)]

# INITIALIZE the current byte position in the file

   set cP 0

# CPXP1L check

   set nSS1 0
   set dN [file split $DaTa]
   set pR [lindex $dN [expr [llength $dN] - 5]]
   if [string match CLUSTERII $pR] {

      set fN [lindex $dN end]
      set fNSz [string length $fN]
      set vN [string range $fN 0 [expr $fNSz - 13]]
      set A1 [string match CPXP1L $vN] ; set A2 [string match CPX1L $vN]
      set A3 [string match CPXP1H $vN] ; set A4 [string match CPX1H $vN]
      if { $A1 || $A2 || $A3 || $A4 } {
         binary scan [read $fd $B] ${iB4}3${iB4}$N${iB4} TmP hO nSS1
         seek $fd $vI(DataLen) start
         binary scan [read $fd $B] ${iB4}3${iB4}$N${iB4} TmP hO nSS2
	 if { $nSS1 != $nSS2 } { 
	    puts stderr "$pR $vN : nSS mismatch $nSS1 $nSS2 - setting to $nSS2"
	    set nSS1 $nSS2 
	 }
         seek $fd $cP start
      }
   }

# LOOP over the the file data record by data record

   while { [tell $fd] < $fS } {

# CONVERT the data record info fields

      binary scan [read $fd $B] ${iB4}3${iB4}$N${iB4} TmP hO nSS
      if { $nSS1 != 0 } { set nSS $nSS1 ; set nSS1 0 }

      if $vI(First) { DBsasizeChk $nSS vI }

      seek $fd $cP start
#     if { $nSS < -32 } { set nSS -32 }
      puts -nonewline $fd [binary format ${oB4}3${oB4}$N$oB4 \
           $TmP $hO $nSS]
      incr cP $B
      seek $fd $cP start

# IF the number of sensor sets is negative make it positive.  Also set
#   the increment header variable flag to 0 since all sensor sets in 
#   this data record will have the same header

      if { $nSS < 0 } {
         set IncHdr 0
         set nSS [expr -$nSS]
      } else { set IncHdr 1 }

#  THIS is the number of bytes in the data array portion of the data
#    recrod

      set nB [expr $vI(DataLen) - $B]

#  LOOP over the defined sensor sets and process one by one
      
      set LastToHdR -1
      set LastNsmP  -1
      set LastNseN  -1
      set LastTotC  -1
      for { set sS 0 } { $sS < $nSS } { incr sS } {

#  READ the header record associated with this sensor set

         if { $IncHdr == 1 } {
            set ToHdR [lindex $hO $sS]
         } else { set ToHdR [lindex $hO 0] }

         set DoThis 0
         if { $ToHdR != $LastToHdR } {
            set LastToHdR $ToHdR 
            seek $fh [expr $ToHdR + 24] start
            binary scan [read $fh 6] $oB2$oB2$oB2 NseN NsmP TotC
            if { $NseN != $LastNseN } {
               set DoThis 1
            } elseif { $NsmP != $LastNsmP } {
               set DoThis 1
            } elseif { $TotC != $LastTotC } { set DoThis 1 }
            set LastNsmP  $NsmP
            set LastNseN  $NseN
            set LastTotC  $TotC
         }

# DETERMINE the number of bytes of ancillary data in each of the ancillary
#   data areas and the number of bytes of sensor data.  The preface ancillary 
#   data size is only determined for the first sensor set.  Its zero for all 
#   the rest.  The preface ancillary data only occurs once in the data array.
#   In the nBytes array:
#      0 = Preface Ancillary Data
#      1 = Sensor Data
#      2 = General Ancillary Data
#      3 = Sensor Ancillary Data

         if $DoThis {
            set tC $NseN
            if { $vI(SmpId) == 3 } { set tC [expr $tC * $TotC] }
            for { set I 0 } { $I < 4 } { incr I } { set nBytes($I) 0 }
            for { set I 0 } { $I < $vI(CalSets) } { incr I } {
               set cTar [lindex $vI(CalTarget) $I]

               set cUse [lindex $vI(CalUse) $I]

               if { ($sS > 0) && ($cTar == 3) } { continue }
               if { ($cUse == 0) || ($cTar == 3) } {
                  set Num 1 
               } else {
                 set Num [expr int($NsmP / $cUse)]
                 if { [expr $NsmP % $cUse] > 0 } { incr Num }
               }
   
               if { $cTar == 3 } { 
                  incr nBytes(0)
               } elseif { $cTar == 2 } { 
                  incr nBytes(2) $Num
               } else { incr nBytes(3) [expr $Num * $tC] }
            }

            set nBytes(0) [expr $nBytes(0) *  [lindex $vI(WdLen) 0]]
            set nBytes(1) [expr $tC * $NsmP * [lindex $vI(WdLen) 1]]
            set nBytes(2) [expr $nBytes(2) *  [lindex $vI(WdLen) 2]]
            set nBytes(3) [expr $nBytes(3) *  [lindex $vI(WdLen) 3]]
            set nBytes(4) $nBytes(0)
            set nBytes(5) [expr $nBytes(1) + $nBytes(2) + $nBytes(3)]
            set nBytes(6) [expr $nBytes(5) + $nBytes(0)]
         }

# PROCESS the SENSOR SET. This is done by looping over the four data areas.  
#  Only need to convert the data in a given data area if the wordlength is 
#  not 1.  Do need to remember to skip it though.

         if { $sS > 0 } {
            set nBytes(0) 0
            set tB $nBytes(5)
         } else {
            set nBytes(0) $nBytes(4)
            set tB $nBytes(6)
         }

         for { set I 0 } { $I < 4 } { incr I } { 
            if { $nBytes($I) > 0 } {
               if { [lindex $vI(WdLen) $I] == 2 } {
                  set nT [ expr int($nBytes($I) / 2)]
                  binary scan [read $fd $nBytes($I)] ${iB2}$nT TmP
                  seek $fd $cP start
                  puts -nonewline $fd [binary format ${oB2}$nT $TmP]
               } elseif { [lindex $vI(WdLen) $I] == 4 } {
                  set nT [ expr int($nBytes($I) / 4)]
                  binary scan [read $fd $nBytes($I)] ${iB4}$nT TmP
                  seek $fd $cP start
                  puts -nonewline $fd [binary format ${oB4}$nT $TmP]
               }
               incr cP $nBytes($I)
               seek $fd $cP start
            }
         }

#  This is the number of bytes left after processing this sensor set

         incr nB [expr -$tB]
      }

# IF there are padding bytes left over then skip them

      if { $nB > 0 } { 
         incr cP $nB
         seek $fd $cP start
      }
   }

   close $fh
   close $fd
}
