/*  The C equivalent of UDFFloat,  It consists of a TCL interface over      */
/*  a straight C subroutine                                                 */
/*                                                                          */
/*  There are 3 OBJV elements.  These are:                                  */
/*     OBJV[0]:  Routine name                                               */
/*     OBJV[1]:  UDF raw input data                                         */
/*     OBJV[2]:  Elements in array                                          */
/*     OBJV[3]:  The floating point format                                  */

#include <tcl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ByteDefs.h"
#include "TudfAnsi.h"
#include "TutilAnsi.h"

int ToFloat (ClientData cD, Tcl_Interp *tI, int objc, Tcl_Obj *CONST *objv) 
{
   void   *memPtr = NULL; 
   ByTe_4 Bytes;
   int    N, Fmt;
  
/* MAKE sure that all the IO parameters are present                      */

   if (objc != 4 ) {
     Tcl_WrongNumArgs(tI, 1, objv, "Usage: UDFFloat X N Fmt");
     return TCL_ERROR;
   }

/* GET the inputs                                                        */
 
   Tcl_GetIntFromObj (tI, objv[2], &N);
   Tcl_GetIntFromObj (tI, objv[3], &Fmt);

/* GET the work array memory and put the input data into it              */

   Bytes = sizeof(ReaL_8) * N;
   if ((memPtr = malloc (Bytes)) == NULL) { return TCL_ERROR; }

   if (TclArrayToC (tI, objv[1], memPtr, N, 0, 'D') == 0) {
      free (memPtr);
      return TCL_OK;
   }   

/* CALL the C routine                                                    */

   ToFloat_C (memPtr, N, Fmt);

/* PUT the data back into the data array                                 */

   CArrayToTcl (tI, objv[1], memPtr, N, 0, 'D'); 

/* GET rid of the temporary array and return                             */

   free (memPtr);
   return TCL_OK;

}

/* THE C ROUTINE                                                         */
/*                                                                       */
/*   This is the UDF float to native float conversion routine            */
/*                                                                       */
/*  FmT Definitions                                                      */
/*    2   Single Precicion 1 (32 bits)                                   */
/*    3   Double Precicion 1 (64 bits) (undefined UDF format)            */
/*    4   Half Precision 1   (16 bits)                                   */
/*    5   Half Precision 2   (16 bits) [CLUSTER PEACE MOMENTS]           */
/*    6   Half Precision 3   (16 bits) [INTERBALL SCA]                   */
/*    7   Single Precision 2 (32 bits) [IMAGE (IEEE?)]                   */
/*                                                                       */
/*  Variable Definitions                                                 */
/*    MSgn   Sn of the Mantissa                                          */
/*    MMag   Magnitude of the Mantissa                                   */
/*    ESgn   Sn of the Exponent                                          */
/*    EMag   Magnitude of the Exponent                                   */
/*    S      Exponent modifier                                           */
/*    E      Base to use when applyng exponent                           */
/*    N      Any applied normalization factor                            */
/*                                                                       */
/*  Special Definitions                                                  */
/*                                                                       */
/*   If both mantissa and exponent are 0 then                            */
/*      MSgn=0, ESgn=0: value is 0.0                                     */
/*      MSgn=1, ESgn=0: value is +infinity                               */
/*      MSgn=1, ESgn=1: value is -infinity                               */
/*      MSgn=0, ESgn=1: value is invalid                                 */

void ToFloat_C (void *Data, int nE, int Fmt) 
{
   register ByTe_4 I;
   register ReaL_8 *X;

   ReaL_8 S, E, N;
   u_ByTe_4 V, B1, B2;
   u_ByTe_4 MSgn, MMag, ESgn, EMag;
   ByTe_4 ManT, ExP;
   
   X = (ReaL_8 *)Data;
   switch (Fmt)
   {
      case 2:                                      /* single precision */
        S = 7.0;
        E = 10.0;
        for ( I = 0; I < nE; ++I) {
           V = (u_ByTe_4)X[I];
           MSgn = (V >> 31) & 0x00000001;
           MMag = (V >> 7) & 0x00ffffff;
           ESgn = (V >> 6) & 0x00000001;
           EMag = V & 0x0000003f;
           if ((MMag == 0) && (EMag == 0) ) {
              if ((MSgn == 0) && (ESgn == 0) ) {
                 X[I] = 0.0;
              } else if ( (MSgn == 1) && (ESgn == 0) ) {
                 X[I] =  3.402823e38;
              } else if ( (MSgn == 1) && (ESgn == 1) ) {
                 X[I] =  -3.402823e38;
              } else { X[I] = 0.0; }
           } else {
              ManT = ( MSgn == 1 ) ? -MMag : MMag; 
              ExP =  ( ESgn == 1 ) ? -EMag : EMag; 
              X[I] = ManT * pow(E, (ExP - S));
           }
        }
      break;
      case 3:                                      /* double precision */
        for ( I = 0; I < nE; ++I) {
            X[I] = 0.0;
        }
      break;
      case 4:                                      /* half precision 1 */
        S = 2.0;
        E = 10.0;
        for ( I = 0; I < nE; ++I) {
           V = (u_ByTe_4)X[I];
           MSgn = (V >> 15) & 0x00000001;
           MMag = (V >> 7) & 0x000000ff;
           ESgn = (V >> 6) & 0x00000001;
           EMag = V & 0x0000003f;
           if ((MMag == 0) && (EMag == 0) ) {
              if ((MSgn == 0) && (ESgn == 0) ) {
                 X[I] = 0.0;
              } else if ( (MSgn == 1) && (ESgn == 0) ) {
                 X[I] =  3.402823e38;
              } else if ( (MSgn == 1) && (ESgn == 1) ) {
                 X[I] =  -3.402823e38;
              } else { X[I] = 0.0; }
           } else {
              ManT = ( MSgn == 1 ) ? -MMag : MMag; 
              ExP =  ( ESgn == 1 ) ? -EMag : EMag; 
              X[I] = ManT * pow(E, (ExP - S));
           }
        }
      break;
      case 5:                                      /* half precision 2 */
        S = 128.0;
        E = 2.0;
        N = 256.0;
        ESgn = 0;
        for ( I = 0; I < nE; ++I) {
           V = (u_ByTe_4)X[I];
           MMag = (V != 0) ? ((V >> 8) & 0x0000007f) + 128 :  0;
           MSgn = (V >> 15) & 0x00000001;
           EMag = (V & 0x000000ff);
           if ((MMag == 0) && (EMag == 0) ) {
              if ((MSgn == 0) && (ESgn == 0) ) {
                 X[I] = 0.0;
              } else if ( (MSgn == 1) && (ESgn == 0) ) {
                 X[I] =  3.402823e38;
              } else if ( (MSgn == 1) && (ESgn == 1) ) {
                 X[I] =  -3.402823e38;
              } else { X[I] = 0.0; }
           } else {
              ManT = ( MSgn == 1 ) ? -MMag : MMag; 
              ExP = EMag;
              X[I] = ManT * pow(E, (ExP - S)) / N;
           }
        }
       break;
      case 6:                                      /* half precision 2 */
        E = 2.0;
        for ( I = 0; I < nE; ++I) {
           V = (u_ByTe_4)X[I];
           B1 = (V >> 8) & 0x000000ff;
           B2 = V  & 0x000000ff;
           ESgn = (B1 >> 7) & 0x01;
           EMag = B1 & 0x3f;
           MSgn = (B1 >> 6) & 0x01;
           if (EMag != 0) {
              MMag = B2 + 256; 
              N = 512.0;
           } else {
              MMag = B2;
              N = 1.0;
           }
           if ((MMag == 0) && (EMag == 0) ) {
              if ((MSgn == 0) && (ESgn == 0) ) {
                 X[I] = 0.0;
              } else if ( (MSgn == 1) && (ESgn == 0) ) {
                 X[I] =  3.402823e38;
              } else if ( (MSgn == 1) && (ESgn == 1) ) {
                 X[I] =  -3.402823e38;
              } else { X[I] = 0.0; }
           } else {
              ManT = ( MSgn == 1 ) ? -MMag : MMag; 
              ExP =  EMag; 
              X[I] = ManT * pow(E, ExP) / N;
           }
        }
      break;
      case 7:                                      /* IMAGE FP - IEEE? */
        E = 2.0;
        S = 127.0;
        N = 1 << 23;
        for ( I = 0; I < nE; ++I) {
           V = (u_ByTe_4)X[I];
           EMag = (V >>23) & 255;
           MMag = V & 0x07fffff;
           if (EMag == 255) {
              EMag = 0;
              MSgn = 1;
              ESgn = (MMag == 0) ? 0 : 1;
              MMag = 0;
           } else { 
              ESgn = 0.0;
              MSgn = (V >> 31) & 0x00000001;
	      MMag |= 0x800000;
           }
           if ((MMag == 0) && (EMag == 0) ) {
              if ((MSgn == 0) && (ESgn == 0) ) {
                 X[I] = 0.0;
              } else if ( (MSgn == 1) && (ESgn == 0) ) {
                 X[I] =  3.402823e38;
              } else if ( (MSgn == 1) && (ESgn == 1) ) {
                 X[I] =  -3.402823e38;
              } else { X[I] = 0.0; }
           } else {
              ManT = ( MSgn == 1 ) ? -MMag : MMag; 
              ExP =  EMag; 
              X[I] = ManT * pow(E, (ExP - S)) / N;
           }
        }
      break;
   }
}
