# THIS produce computes a set of power spectra which can be used in a dynamic
#    power spectra plot
#  
#  INPUT parameters:
#    _vI  : Input data array
#    _vO  : Output data. Must be 4 variables.
#           In order these are: 
#              1. the beg time variable, 
#              2. the ending time variable, 
#              3. the freq variable
#              4. the power spectrum variable.
#    _Met : Either MEM or FFT depending on what method to use to compute
#              the power spectra
#    _Len : The number of data points to use in the MEM or FFT
#    _Adv : The number of data points to advance between MEM or FFT
#    _nC  : The number of coefficients to use in MEM
#    _nF  : The number of frequencies to use in MEM
#    _bF  : The beginning frequency in the MEM spectra
#    _eF  : The ending frequency in the MEM spectra
#    _Sca : The scaling to use between frequency steps frequency steps
#
# GROUPS.  A group is a set multiple definitions from different input data
#   sets that are being output in a common output variable.  The method for
#   determining the spectra must be the same for each member in the group
#   and is taken from the first definition in the group.  The frequency
#   range must be identical also.  That means that for FFT the input data 
#   sets must all have the same length and dT.  MEM is more versitle and
#   allows for diverse input data sets.
#
#   SO, in any group the input variable is the only variable allowed to change.
#     All other variables are held as they are defined in the first definition
#     in the group.

package provide UDFAnalysisTh 1.0

proc APsolveDps { fD } {
   global apANS env Prefs

   APkeepTabs "STEP $fD : DYNAMIC-PS"

# THIS is the text window for this function definition

   set W .apFDEF$fD.body.list
   if ![winfo exists $W] {
      set GuI [lindex $apANS($apANS($fD,Func)) 0]
      eval $GuI $fD 1
   }

# THIS is the code which can be evoked when running a thread.  This is
#    a single procedure which handles both the FFT and MEM methods.

   set PgmA {
      source [ file join $env(TCLTOOLS_HOME) TclToolInits.tcl ]
      TclToolInits THREAD UTILS FFT
      lappend auto_path [file join $env(TCLTOOLS_HOME) TclAnalysisTh]
      package require UDFAnalysis
      package require UDFAnalysisTh

# THE procedure which produces the power spectra
 
      proc PwrS  { rN bC sT } {

# SET up initial parameters
 
         set _Len [tsv::get DpS _Len]
         set _nC  [tsv::get DpS _nC]
         set _nF  [tsv::get DpS _nF]

         set dT   [tsv::get DpS dT]
         set PaD  [tsv::get DpS PaD]
         set nE   [lindex [tsv::get _In Dim] 0]
         set N    [expr $_nF  * $bC]
 
# COPY data into the processing array and pad the array if necessary

         set K $sT
         for { set J 0 } { $J < $_Len } { incr J ; incr K } {
            set _T_($J) [tsv::get _In $K]
         }

# SOLVE for the power spectrum according to method requested

         if [string match FFT [tsv::get DpS _Met]] {

# PAD the data if necessary 

            for {  } { $J < $PaD } { incr J } { set _T_($J) 0.0 }

# PERFORM FFT
 
            RealFfT _T_ $PaD 1

# COMPUTE power spectra
 
	    set NrM [expr 1.0 / double($_Len)]
	    set K 0
	    for  { set J 0 } { $J < $_nF } { incr J ; incr K ; incr N } {
	       set sP [expr $_T_($K) * $_T_($K)] ; incr K
	       tsv::set _pS $N [expr ($sP + $_T_($K) * $_T_($K)) * $NrM] 
            }
         } else {
 
#  First get the MEM coefficients

	    set MsD [MaxEntCoef _T_ $_Len $_nC _CoeF]

# SOLVE for the power density at each frequency step

	    for  { set J 0 } { $J < $_nF } { incr J ; incr N } {
	       set FdT [expr [tsv::get _fA $J] * $dT]
	       tsv::set _pS $N [MaxEntPwr _CoeF $_nC $FdT $MsD]
	    }
         }
      }
   }

#XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
# FUNCTION starts here
#XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

# THIS is the number of function definitions 

   set nFd [$W index end]

# NO instances then return

   if { $nFd == 0 } { return }

# OPEN up the thread pool

   set tPoolID [THpoolOpen $PgmA]

# GROUP sets of definitions which use a common output

   set Starts ""
   for { set I 0 } { $I < $nFd } { incr I } {
      set LiNe [$W get $I]
      scan $LiNe "%s %s" _vI _vO
      if ![string match $apANS(EmptyVar)  $_vO] { lappend Starts $I }
   }

# LOOP over the defined groups
 
   set nG [expr [llength $Starts] - 1 ]
   for { set G 0 } { $G <= $nG } { incr G } {

# GET the beginning and ending definition in the group
 
      if { $G != $nG } {
         set gE [lindex $Starts [expr $G + 1]]
      } else { set gE $nFd }
      set bG [lindex $Starts $G]

# THERE are certain things that are only done once for a group.  These
#    include getting the output data set up and forming frequency
#    arrays.  Get information from the first definition in the group

      set LiNe [$W get $bG]
      scan $LiNe "%s %s %s %s %s %s %s %s %s %s" \
                     _vI _vO _Met _Len _Adv _nC _nF _bF _eF _Sca

# GET the output variables extablished. There should be 4 output varaiables. 
#   In order these are: 
#   1. the beg time variable, 
#   2. the ending time variable, 
#   3. the freq variable
#   4. the power spectrum variable.

      set oNames [lindex [APgetVNames $_vO] 0]
      set nP [llength $oNames]
      if { $nP < 4 } {
         puts stderr "DYNAMIC-PS: Insufficient output variables"
      }

      set vR [lindex $oNames 0] ; global [set vR ] ; upvar 0 [set vR] _tB
      set vR [lindex $oNames 1] ; global [set vR ] ; upvar 0 [set vR] _tE
      set vR [lindex $oNames 2] ; global [set vR ] ; upvar 0 [set vR] _fA
      set vR [lindex $oNames 3] ; global [set vR ] ; upvar 0 [set vR] _pS

# IF this uses the FFT algorithm then make sure that the data is a power
#   of 2 and if not then compute the length which will accomplish this.
#   Also set up the frequency arrays for both the MEM or FFT methods depending
#   on which is to be applied.  

      if [string match FFT $_Met] {

# FOR FFT we need the timing. Get that info from the first defined input
#    variable in the group
 
         set iNames [lindex [APgetVNames $_vI] 0]
         set vR [lindex $iNames 0] ; global [set vR ] ; upvar 0 [set vR] _In
	 set dT $_In(gIcZ)

# GET _Len to points
   
         set _Len [APtimeToPts  _In $_Len]
 
         set PaD 1
         while { $PaD < $_Len } { set PaD [expr $PaD << 1] }

         set _dF [expr 1.0 / $dT / double($PaD)]
         set _nF [expr $PaD / 2]
	 for { set I 0 } { $I < $_nF } { incr I } { 
	    set _fA($I) [expr $I * $_dF]
         }
	 set _fA(Dim) [list $_nF 1]

      } else { 
	 set PaD $_Len

         set _nFb [expr double($_nF) - 1.0] 
         if [string match LOG $_Sca] {   
            set _dF [expr (log10($_eF) - log10($_bF)) / $_nFb]
            set _lbF [expr log10($_bF)]
            for { set J 0 } { $J < $_nF } { incr J } {
               set _fA($J) [expr $_lbF + $J * $_dF]
               set _fA($J) [expr pow(10.0, $_fA($J))]
            }
         } else {
            set _dF [expr ($_eF - $_bF) / $_nFb]
            for { set J 0 } { $J < $_nF } { incr J } {
               set _fA($J) [expr $_bF + $J * $_dF]
            }
	 }

         set _fA(Dim) [list $_nF 1]

# Copy the MEM frequency array to tsv space.
 
         THarrayXfer TO _fA _fA
      }

      tsv::set DpS _nF $_nF
      tsv::set DpS _nC $_nC
      tsv::set DpS _Met $_Met
      tsv::set DpS PaD $PaD

# INITIALIZATIONS
#    bC is the block counter

     set bC 0

# Process each definition

      for { set I $bG } { $I < $gE } { incr I } {

# GET the first line
   
         set LiNe [$W get $I]

# BREAK it apart

         scan $LiNe "%s %s %s %s %s %s %s %s %s %s" \
                     _vI _vO _Met _Len _Adv _nC _tF _bF _eF _Sca

# GET the input variable.  Should only be one so take what you get

         set iNames [lindex [APgetVNames $_vI] 0]
         set nI [llength $iNames]
         set vR [lindex $iNames 0] ; global [set vR ] ; upvar 0 [set vR] _In

# GET _Len and _Adv to points
   
         set _Len [APtimeToPts  _In $_Len]
         set _Adv [APtimeToPts  _In $_Adv]
         tsv::set DpS _Len $_Len

# PUSH the input data over to tsv space
 
         THarrayXfer TO _In _In

# MAKE sure that we have enough data points to compute at least one power
#   spectrum. If not free up the used tsv space and return  
      
         set nE [lindex $_In(Dim) 0]
	 tsv::set DpS nE $nE
         if { $_Len > $nE } {
	    tsv::unset DpS
            if [string match MEM $_Met] { tsv::unset _fA }
	    return 
	 } 

# INPUT data must be in a time based grid.  Get the grid used.  If none
#   report error and exit.  

         if ![info exists _In(gInE)] { 
            puts stderr "Variable $vR has no associated grid information"
	    exit
         }
	 APxferGInfo _In gI DATA REVERSE 

# GET the center time per cell as well as the time spanned by a single cell.

         TUgridInfo 30 gI _tM
         set dT $gI(gIcZ)
	 tsv::set DpS dT $dT 

# WALK across the data set - picking up _Len points, processing, advancing
#   _Adv points and start the cycle again.
#
# BeG and EnD are the start and EnD positions of the current block in the data

         set BeG 0
         set EnD $_Len

# LOOP as long as there are data blocks to process

         set JoBs ""
	 set rN 0

         while { $EnD < $nE } {
            lappend JoBs [THschdTask $tPoolID [list PwrS $rN $bC $BeG]]
	    incr rN

# SET the beginning and ending time for this block

            set _tB($bC) $_tM($BeG)
            set _tE($bC) $_tM([expr $BeG + $_Adv])
            incr bC

# ADVANCE the beginning data location and make sure that there is
#   sufficient data for another block

            incr BeG $_Adv
            set EnD [expr $BeG + $_Len]
         }
	 THjobsDone $tPoolID $JoBs

# REMOVE the input data from tsv space
 
         tsv::unset _In
      }

# TRANSFER the power spectra data from tsv space to output variable and
#    remove the tsv data
 
      THarrayXfer FROM _pS _pS YES

# SET the array dimensions

      set _tB(Dim) [list $bC 1]
	      set _tE(Dim) [list $bC 1]
      set _pS(Dim) [list $bC $_nF]
      puts stderr $_pS(Dim)
   }

# FREE the tsv data used
 
   tsv::unset DpS
   if [string match MEM $_Met] { tsv::unset _fA }
}
