/*  ===========
 >>>FILE: pbr.c
 >  ===========
 >
 > Description --
 >   This routine returns postime data file names based upon the requested time
 > range in the local database.
 >
 > Synopsis --
 >   int pbr (btime, etime, src, type, exten, fname)
 >     Time_t btime, etime;
 >     StrHier src;
 >     int type;
 >     char  *exten, *fname;
 >     
 >   btime    the start time requested
 >   etime    the stop time requested
 >   src      node into the projects tree for the desired virtual instrument 
 >   type     file type: header, data, vidf, scf
 >   exten    filename extension for the data to be used
 >   fname    returned path and file of requestd data
 >
 > Return Values --
 >
 > Routines Used --
 >   int find_entry()       finds all database entries that cover the input
 >                          time range
 >   int OpenDbf()          open a database file ind initialize structure
 >   int OpenIndex()        open an index file
 >   FieldRecord_t *FIELD() return a pointer to a field within a database record
 >
 > Externals --
 >   dbfRecords_t *dbfRecords    database structures
 >   LinkList Projects           pointer to projects tree
 >
 > Internals --
 >   StrHier I, V         pointers to the instrument and virtual instrument
 >                        data within the project tree
 >   Time_t r_btime       begin time of in database record
 >   Time_t r_btime       end time of in database record
 >   int first            flag to indicate the first file found for the time
 >                        range given
 >   char str[10]         string to aid in converting the time from the
 >                        database file
 >   char *r_fname        file name from database record
 >   char file[30]        found data file to return
 >   char vname[10]       virtual instrument for database record
 >   char *r_vname        trimmed virtual instrument name
 >   char *path           path value from configuration file
*/ 

/* REWORKED Sept. 17 2010 by CAG                                           */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "OpSySInD.h"
#include "libCfg.h"
#include "Server.h"

#define FG(a,b,c) if (FieldGetN(a, b, c) != SUCCESS) return FAILURE

int pbr ( Time_t btime,  Time_t etime, StrHier src, int type, 
          char *exten, char *fname, ByTe_4 *Tm )
{
   ByTe_4 dT;
   ByTe_1 fChk;
   ByTe_1 file[30], *r_fname;
   ByTe_1 *path, DbfName[FILE_NAME_LEN], NdxName[FILE_NAME_LEN];
   ByTe_1 *Slash = PathSep;
   int  Dbf, Ndx, rC;
   int tG;
   Time_t r_btime, r_etime;

/*  IF we're looking for real-time data then return ther correct paths and */
/*  file names.  That's all we need to do                                  */

   if (btime.yr == -1 || btime.yr == -2) {
      if ((path = CfgPath(src, "RealTimePath", 4)) == NULL) {
         return CONFIG_FILE_ERROR;
      }
      switch(type) {
         case _HEADER:
            (void)sprintf(fname, "%s%s%sH", path, Slash, StrHierName(src));
            break;

         case _DATA:
            (void)sprintf(fname, "%s%s%sD", path, Slash, StrHierName(src));
            break;

         case _VIDF:
            (void)sprintf(fname, "%s%s%sI", path, Slash, StrHierName(src));
            break;
      }

      return ALL_OK;
   }


/*  A bit more complicated for realtime data.  We need to return the first */
/*  valid file name but we need to make sure that if the data starts       */
/*  before that file that routines higher up can ask for a promote to see  */
/*  if there is data for the earlier time.                                 */

/* INITIALIZE the return code to indicate that all is OK                   */

   rC = ALL_OK;

/*  NEED to open the database to search for the first file that contains   */
/*  the requested time                                                     */

/*  GET the database directory path and generate the database and index    */
/*  filenames                                                              */

   if ((path = DbfFile(src)) == NULL) { return CONFIG_FILE_ERROR; }

   memset(DbfName, 0, sizeof(DbfName));
   memset(NdxName, 0, sizeof(NdxName));
   switch(type) {
      case _HEADER:
      case _DATA:
         sprintf(DbfName, "%s.HD.DBF", path);
         sprintf(NdxName, "%s.HD.NDX", path);
         break;

      case _VIDF:
         sprintf(DbfName, "%s.I.DBF", path);
         sprintf(NdxName, "%s.I.NDX", path);
         break;
   }

/*  OPEN the database and index files                                      */

   if (OpenDbf(&Dbf, DbfName, O_RDONLY) != SUCCESS) { return DBF_OPEN_ERROR; }

   if (OpenNdx(Dbf, NdxName, &Ndx) != SUCCESS) {
      CloseDbf(Dbf);
      return INDEX_OPEN_ERROR;
   }

/*  PERFORM a search for the requeted data.  If there is nothing online   */
/*  matching the requested time return NO_DATA which will force a promote */

   if (!find_entry(Dbf, Ndx, StrHierName(src), btime, etime, 1)) {
      CloseNdx(Dbf, Ndx);
      CloseDbf(Dbf);
      return NO_DATA;
   }

/*  PULL the beginning time from the current database record               */

   FG(Dbf, "B_YR", &(r_btime.yr));
   FG(Dbf, "B_DAY", &(r_btime.day));
   FG(Dbf, "B_MSEC", &(r_btime.msec));

/*  CHECK to make sure that if the initial time is less than the start   */
/*  time of the first file that its not more UDF_TIME_GAP seconds early. */ 
/*  If is then force a promote to see if the server has earlier data     */
/*  If UDF_TIME_GAP doesn't exist then the time gap is set o 0 seconds.  */

   dT = r_btime.yr - btime.yr;
   if (dT < 0) {
      fChk = 0;
   } else if ( dT > 1 ) {
      fChk = 1;
   } else {
      if (getenv("UDF_TIME_GAP") != NULL) {
         sscanf(getenv("UDF_TIME_GAP"), "%d", &tG);
      } else { tG = 0; }

      dT *= ((((btime.yr % 4) == 0) && ((btime.yr % 100) != 0)) || ((btime.yr % 400) == 0)) ? 366 : 365;
      dT = dT + (r_btime.day - btime.day);
      dT = dT * 86400 + (r_btime.msec - btime.msec) / 1000;

      fChk = (dT > tG) ? 1 : 0;
   }
   if ( fChk == 1 ) { rC = NEED_PROMOTE; }

/*  GET the file name associated with the data.                            */
/*  TACIT assumption is that if the database points to a datafile that it  */
/*  is online and ready to use                                             */

/*  NEED to pull the ending time of the file out of the database so that   */
/*  can be passed back.  It's used to set the beginning of the next file   */
/*  if we need to get to it (user ending time is not in this file)         */

   FG(Dbf, "E_YR", &(r_etime.yr));
   FG(Dbf, "E_DAY", &(r_etime.day));
   FG(Dbf, "E_MSEC", &(r_etime.msec));
   Tm[0] = r_etime.yr;
   Tm[1] = r_etime.day;
   Tm[2] = r_etime.msec;

/* RETRIEVE the ending time of the file                                    */

   switch(type) {
      case _HEADER:
         (void)sprintf(file, "%s", MakeFileName(StrHierName(src),
                       r_btime, 'H'));
         break;
      case _DATA:
         (void)sprintf(file, "%s", MakeFileName(StrHierName(src),
                       r_btime, 'D'));
         break;
      case _VIDF:
         (void)sprintf(file, "%s", MakeFileName(StrHierName(src),
                       r_btime, 'I'));
         break;
   }
   r_fname = rtrim(file);

/*  IF the exten variable has something in it, use the user data path    */
/*  for data insted of the default one                                   */

   if (strlen(exten) != 0) {
      if (getenv("USER_DATA") == NULL) {
         (void)sprintf(fname, "%s%s%s%s",
                               getenv("HOME"), Slash, r_fname, exten);
      } else {
         (void)sprintf(fname, "%s%s%s%s",
                             getenv("USER_DATA"), Slash, r_fname, exten);
      }
   } else {
      (void)sprintf(fname, "%s%s%s", CfgPath(src,
                    "PostTimePath", 4), Slash, r_fname);
   }

/*  WE'RE' all done - close the database files                             */

   CloseNdx(Dbf, Ndx);
   CloseDbf(Dbf);

   return rC;
}
