/*  ===========
 >>>FILE: pbr.c
 >  ===========
 >
 > Description --
 >   This routine returns postime data file names based upon the requested time
 > range in the local database.
 >
 > Synopsis --
 >   int pbr (btime, etime, src, type, exten, fname)
 >     Time_t btime, etime;
 >     StrHier src;
 >     int type;
 >     char  *exten, *fname;
 >     
 >   btime    the start time requested
 >   etime    the stop time requested
 >   src      node into the projects tree for the desired virtual instrument 
 >   type     file type: header, data, vidf, scf
 >   exten    filename extension for the data to be used
 >   fname    returned path and file of requestd data
 >
 > Return Values --
 >
 > Routines Used --
 >   int find_entry()       finds all database entries that cover the input
 >                          time range
 >   int OpenDbf()          open a database file ind initialize structure
 >   int OpenIndex()        open an index file
 >   FieldRecord_t *FIELD() return a pointer to a field within a database record
 >
 > Externals --
 >   dbfRecords_t *dbfRecords    database structures
 >   LinkList Projects           pointer to projects tree
 >
 > Internals --
 >   StrHier I, V         pointers to the instrument and virtual instrument
 >                        data within the project tree
 >   Time_t r_btime       begin time of in database record
 >   Time_t r_btime       end time of in database record
 >   int first            flag to indicate the first file found for the time
 >                        range given
 >   char str[10]         string to aid in converting the time from the
 >                        database file
 >   char *r_fname        file name from database record
 >   char file[30]        found data file to return
 >   char vname[10]       virtual instrument for database record
 >   char *r_vname        trimmed virtual instrument name
 >   char *path           path value from configuration file
*/ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "OpSySInD.h"
#include "libCfg.h"
#include "Server.h"

#define FG(a,b,c) if (FieldGetN(a, b, c) != SUCCESS) return FAILURE

static ByTe_1 rcsid[] = "$Id: pbr.c,v 1.1 1999/11/12 20:36:21 chris.gurgiolo.b2r Stab chrisg $";

int pbr ( Time_t btime,  Time_t etime, StrHier src, int type, 
          char *exten, char *fname, ByTe_4 *Tm )
{
   ByTe_1 file[30], *r_fname, vname[10], *r_vname;
   ByTe_1 *path, DbfName[FILE_NAME_LEN], NdxName[FILE_NAME_LEN];
   ByTe_1 *Slash = PathSep;
   int  Dbf, Ndx, first;
   Time_t r_btime, r_etime;


   /*
    if we're looking for real-time data then use ther correct paths and
    file names
    */
   if (btime.yr == -1 || btime.yr == -2)
   {
      if ((path = CfgPath(src, "RealTimePath", 4)) == NULL)
      {
         return CONFIG_FILE_ERROR;
      }
      switch(type)
      {
         case _HEADER:
            (void)sprintf(fname, "%s%s%sH", path, Slash, StrHierName(src));
            break;

         case _DATA:
            (void)sprintf(fname, "%s%s%sD", path, Slash, StrHierName(src));
            break;

         case _VIDF:
            (void)sprintf(fname, "%s%s%sI", path, Slash, StrHierName(src));
            break;
      }

      return ALL_OK;
   }

   /*
    the request is for post-time data, need to open the database to search
    for the correct file that contains the request time and decoded key
    */

   /*
    get the database directory path and generate the database and index
    filenames
    */
   if ((path = DbfFile(src)) == NULL)
      return CONFIG_FILE_ERROR;

   memset(DbfName, 0, sizeof(DbfName));
   memset(NdxName, 0, sizeof(NdxName));
   switch(type)
   {
      case _HEADER:
      case _DATA:
         sprintf(DbfName, "%s.HD.DBF", path);
         sprintf(NdxName, "%s.HD.NDX", path);
         break;

      case _VIDF:
         sprintf(DbfName, "%s.I.DBF", path);
         sprintf(NdxName, "%s.I.NDX", path);
         break;
   }

   /*
    open up the database and index files
    */
   if (OpenDbf(&Dbf, DbfName, O_RDONLY) != SUCCESS)
   {
      return DBF_OPEN_ERROR;
   }
   if (OpenNdx(Dbf, NdxName, &Ndx) != SUCCESS)
   {
      CloseDbf(Dbf);
      return DBF_OPEN_ERROR;
   }

   /*
    preform a search for the requeted data
    */
   if (!find_entry(Dbf, Ndx, StrHierName(src), btime, etime, 1))
   {
      CloseNdx(Dbf, Ndx);
      CloseDbf(Dbf);
      return NO_DATA;
   }


   /*
    * pull the beginning time from the current database record
    */

   FG(Dbf, "B_YR", &(r_btime.yr));
   FG(Dbf, "B_DAY", &(r_btime.day));
   FG(Dbf, "B_MSEC", &(r_btime.msec));

   /*
    loop to make sure we get all the data requested and it's online, ready to
    use
    */
   first = 1;
   do
   {
      /*
       pull the ending time from the database record
       */
      FG(Dbf, "E_YR", &(r_etime.yr));
      FG(Dbf, "E_DAY", &(r_etime.day));
      FG(Dbf, "E_MSEC", &(r_etime.msec));
      Tm[0] = r_etime.yr;
      Tm[1] = r_etime.day;
      Tm[2] = r_etime.msec;

      if (first)
      {
         switch(type)
         {
            case _HEADER:
               (void)sprintf(file, "%s", MakeFileName(StrHierName(src),
                             r_btime, 'H'));
               break;
            case _DATA:
               (void)sprintf(file, "%s", MakeFileName(StrHierName(src),
                             r_btime, 'D'));
               break;
            case _VIDF:
               (void)sprintf(file, "%s", MakeFileName(StrHierName(src),
                             r_btime, 'I'));
               break;
         }
         r_fname = rtrim(file);

         /*
          if the exten variable has something in it, use the user data path
          for data insted of the default one
          */
         if (strlen(exten) != 0)
         {
            if (getenv("USER_DATA") == NULL)
            {
               (void)sprintf(fname, "%s%s%s%s",
                                     getenv("HOME"), Slash, r_fname, exten);
            }
            else
            {
               (void)sprintf(fname, "%s%s%s%s",
                                   getenv("USER_DATA"), Slash, r_fname, exten);
            }
         }
         else
         {
            (void)sprintf(fname, "%s%s%s", CfgPath(src,
                          "PostTimePath", 4), Slash, r_fname);
         }

         /*
          is this entry online ??
          */
         if (online(fname))
         {
            /*
             only want the first file name to pass back
             */
            first = 0;
            break;
         }
      }
      /*
       adjust the begin time to check to see if the next record is also within
       the requeted time range
       */
      btime = r_etime;
      btime.msec += 1000;

      /*
       get the next record in line
       */
      if (Skip(Dbf, Ndx, 1) != SUCCESS)
      {  
         break;
      }  

      /*
       get the virtual instrument name, check to see if we're still in the
       correct virtual instrument list
       */
      FG(Dbf, "V_INST", vname);
      FG(Dbf, "B_YR", &(r_btime.yr));
      FG(Dbf, "B_DAY", &(r_btime.day));
      FG(Dbf, "B_MSEC", &(r_btime.msec));
      r_vname = rtrim(vname);
      if(!r_vname) r_vname = vname;

   } while (strcmp(r_vname, StrHierName(src)) == 0 &&
             cmpTime(r_btime, etime) <= 0);

   /*
    we're all done - close the database files
    */
   CloseNdx(Dbf, Ndx);
   CloseDbf(Dbf);

   if (first == 0) 
   {
      /*
       found a file to plot
       */
      return ALL_OK;
   }
   else
   {
      return NO_DATA;
   }
}
