#include <math.h>
#include "util_str.h"
#include "idf_defs.h"
#include "ret_codes.h"
#include "user_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                         IR_GET_UNIT_NORMALS SUBROUTINE                     *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to retrieve the three components of the aperture *
 *  normal vector for each sensor from the parent instrument.  These values   *
 *  are defined as constants in the VIDF file.  Once the components have been *
 *  retrieved, the magnitude of the normal vector is computed and saved.      *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    None                                                                    *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_get_unit_normals ()                                              *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    ir_tento()                   returns power of 10 value to multiply by   *
 *                                 to get values to correct base (scaling)    *
 *    read_idf()                   reads information from the VIDF file       *
 *    sqrt()                       returns the square root of the argument    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure holding information concerning   *
 *                                 the experiment that is being processed     *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    register struct pitch_info   pointer to the structure that holds pitch  *
 *         *pa_ptr                 angle information                          *
 *    register double *dptr        pointer to the unit normal values          *
 *    register double *mag_ptr     pointer to magnitude values                *
 *    register double *vec_end     loop termination variable                  *
 *    register short i, j          looping variables                          *
 *    double sum1                  summation of the elements for the vector   *
 *    long ret_val                 holds value returned by called module      *
 *    long data_val                constant value read from VIDF file         *
 *    short index                  index into the 3 element normal vector     *
 *    short sensor                 the sensor being processed                 *
 *    char constant_id             the constant id value                      *
 *    char constant_sca            the scaling factor for the constant value  *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_4 ir_get_unit_normals (void)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   register struct pitch_info *pa_ptr;
   register ReaL_8 *dptr, *mag_ptr, *vec_end;
   register ByTe_2 i, j;
   ReaL_8 sum1;
   ByTe_4 ret_val, data_val;
   ByTe_2 index, sensor;
   ByTe_1 constant_id, constant_sca;

   /**************************************************************************/
   /*  Initialize the array that holds the unit normals to the detector      */
   /*  apertures.                                                            */
   /**************************************************************************/

   ex = ginfo.expt;
   pa_ptr = ex->pitch_angles;
   dptr = pa_ptr->unit_normals;
   vec_end = pa_ptr->unit_normals + (3 * ex->num_sensor);
   for (; dptr < vec_end;)
     *dptr++ = OUTSIDE_MIN;

   /**************************************************************************/
   /*  Retrieve the three unit normals.                                      */
   /**************************************************************************/

   for (i = 0; i < ex->num_consts; ++i)
    {
      ret_val = read_idf (ex->data_key, ex->exten, ex->version, &constant_id,
                          _ConstID, i, 0L, 1);
      if (ret_val < 0)
        return (ret_val);

      if (constant_id >= 6 && constant_id <= 8)
       {
         for (j = 0; j < ex->num_sensor; ++j)
          {
           /******************************************************************/
           /*  The order is x, y, z for id's 6, 7 and 8.                     */
           /******************************************************************/

            sensor = j;
            index = constant_id - 6;
            dptr = pa_ptr->unit_normals + (sensor * 3) + index;
 
            /*****************************************************************/
            /*  Retrieve the constant value, which is stored as a long.      */
            /*****************************************************************/

            ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                                (ByTe_1 *) &data_val, _ConsT, i, sensor, 1);
            if (ret_val < 0)
              return (ret_val);
 
            /*****************************************************************/
            /*  Retrieve the scaling factor, which is stored as a char, and  */
            /*  compute the unit normal for the individual component.        */
            /*****************************************************************/

            ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                                &constant_sca, _ConstScA, i, sensor, 1);
            if (ret_val < 0)
              return (ret_val);
    
            *dptr = data_val * ir_tento (constant_sca);
          }
       }
    }

   /*************************************************************************/
   /*  Make sure all three components of the aperture normal vector were    */
   /*  retrieved.  Compute the magnitude of the unit normals.               */
   /*************************************************************************/

   mag_ptr = pa_ptr->mag_normal;
   dptr = pa_ptr->unit_normals;
   for (i = 0; i < ex->num_sensor; ++i)
    {
      vec_end = dptr + 3;
      for (sum1 = 0.0; dptr < vec_end; ++dptr)
       {
         if (*dptr < VALID_MIN)
           return (PA_UNIT_NORMAL);
         sum1 += *dptr * *dptr;
       }
      *mag_ptr++ = sqrt (sum1);
    }

   return (ALL_OKAY);
}
