#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/*****************************************************************************
 *                                                                           *
 *                        IR_READ_IN_TBL  SUBROUTINE                         *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine is called to read the table values from the VIDF file     *
 *  for the table number being processed. (no need for construction)         *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    short sen                the sensor being processed                    *
 *    short tNum               the number of the table being processed       *
 *    float *tbl               pointer to the table                          *
 *    long tOff                variable which holds the offset into table    *
 *                             (-1 if need to read this info. from VIDF file)* 
 *    long tSize               the maximum number of values associated with  *
 *                             the given sensor                              *
 *                                                                           *
 *  USAGE                                                                    *
 *    x = ir_read_in_tbl (sen, &tbl, tOff, tSize, tNum)                      *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    sizeof ()                the size of the specified object in bytes     *
 *    read_idf()               reads information from the VIDF file          *
 *    ir_read_in_rval ()       returns an error code unique to this module   *
 *                             for each possible READ_IDF() error code       *
 *    ir_tento()               returns the power of 10 value to multiply by  *
 *                             to get values to the correct base (scaling)   *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info      structure that holds information concerning   *
 *        ginfo                the experiment that is being processed        *
 *    unsigned int Tmp_Bytes   number of bytes allocated for scratch space   *
 *    void *Tmp_Space          scratch space for various uses                *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct experiment_info   a pointer to the structure that holds         *
 *          *ex                specific experiment information               *
 *    register float *f1       pointer to data values (float values)         *
 *    register float power     the scaling to be applied to the table values *
 *    register long *l1        pointer to the values read                    *
 *    register char *c1        pointer to the scaling values                 *
 *    unsigned int bytes       the number of bytes to allocate space for     *
 *    long *lwk                memory that holds the values                  *
 *    long rV                  holds the value returned by called routine    *
 *    long offset              sensor offset into the table                  *
 *    long sca_size            the number of scaling factors                 *
 *    char *cwk                memory that holds the scaling values          *
 *    char sca                 scaling factor to be applied to all values    *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 *****************************************************************************/

ByTe_2 ir_read_in_tbl (ByTe_2 sen, ReaL_4 *tbl, ByTe_4 tOff, 
                       ByTe_4 tSize, ByTe_2 tNum)
{
   extern struct general_info ginfo;
   extern size_t Tmp_Bytes;
   extern void   *Tmp_Space;

   struct experiment_info *ex;
   register ReaL_8 power;
   register ReaL_4 *f1;
   register ByTe_4 *l1;
   register ByTe_1 *c1;
   size_t bytes;
   ByTe_4 *lwk, rV, offset, sca_size;
   ByTe_1 *cwk, sca;

 /**********************************************************************/
 /*  Allocate space to hold the values to be read from the VIDF file.  */
 /*  Use the provided external Tmp_Space and increase its size if      */
 /*  needed.                                                           */
 /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc        */
 /**********************************************************************/

   ex = ginfo.expt;
   bytes = (sizeof(ByTe_4) + sizeof(ByTe_1)) * tSize;

   if (bytes > Tmp_Bytes) {
      if (bytes <= 0) { return (READ_IN_MALLOC); }

      free(Tmp_Space);
      if ((Tmp_Space = malloc (bytes)) == NO_MEMORY)
         return (READ_IN_MALLOC);
      Tmp_Bytes = bytes;
   }

   lwk  = (ByTe_4 *) Tmp_Space;
   cwk = (ByTe_1 *) ((ByTe_1 *)Tmp_Space + sizeof(ByTe_4) * tSize);

  /**********************************************************************/
  /*  If the offset value passed into this routine is -1, read the      */
  /*  offset value into the table from the VIDF file; otherwise, use    */
  /*  the offset value passed into this routine.                        */
  /**********************************************************************/

   if (tOff == -1) {
      rV = read_idf (ex->data_key, ex->exten, ex->version,
                     (ByTe_1 *) &offset, _TblOfF, tNum, (ByTe_4) sen, 1);
      if (rV < 0) { return (ir_read_in_rval (rV)); }
   } else { offset = tOff; }

  /**********************************************************************/
  /*  Read the table values.                                            */
  /**********************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *) lwk, 
                       _TbL, tNum, offset, (ByTe_2)tSize);
   if (rV < 0) { return (ir_read_in_rval (rV)); }

  /**********************************************************************/
  /*  Get the number of the scaling values in the table                 */
  /**********************************************************************/

   rV = read_idf (ex->data_key, ex->exten, ex->version, 
                          (ByTe_1 *) &sca_size, _TblScaSZ, tNum, 0L, 1);
   if (rV < 0) { return (ir_read_in_rval (rV)); }

  /**********************************************************************/
  /*  There is one scaling value per value if number is greater than 0. */
  /**********************************************************************/

   if (sca_size > 0) {
      rV = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *) cwk,
                          _TblScA, tNum, offset, (ByTe_2) tSize);
      if (rV < 0) { return (ir_read_in_rval (rV)); }

      for (f1 = tbl, l1 = lwk, c1 = cwk; f1 < tbl + tSize;)
         *f1++ = *l1++ * ir_tento (*c1++);

   } else if (sca_size < 0) {

  /**********************************************************************/
  /*  There is a single scaling value for each sensor.                  */
  /**********************************************************************/

      rV = read_idf (ex->data_key, ex->exten, ex->version, (ByTe_1 *) &sca, 
                          _TblScA, tNum, (ByTe_4) sen, 1);
      if (rV < 0) { return (ir_read_in_rval (rV)); }

      power = ir_tento (sca);
      for (f1 = tbl, l1 = lwk;  f1 < tbl + tSize;)
          *f1++ = *l1++ * power;

   } else if (sca_size == 0) {

  /**********************************************************************/
  /*  The stored values are considered already scaled.                  */
  /**********************************************************************/

      for (f1 = tbl, l1 = lwk;  f1 < tbl + tSize;)
         *f1++ = *l1++;
   }

   return (ALL_OKAY);
}

/******************************************************************************
 *                                                                            *
 *                             IR_READ_IN_RVAL SUBROUTINE                     *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine returns an error code unique to the module READ_IN_TBL()   *
 *  based upon the error code returned by the call to READ_IDF().             *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    long rV                     the error code returned by READ_IDF()       *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_read_in_rval (rV)                                                *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_read_in_rval (ByTe_4 rV)
{
   switch (rV) {
      case IDF_NOT_FOUND:  return (READ_IN_IDF_NOT_FOUND);
      case IDF_MANY_BYTES: return (READ_IN_IDF_MANY_BYTES);
      case IDF_TBL_NUM:    return (READ_IN_IDF_TBL_NUM);
      case IDF_CON_NUM:    return (READ_IN_IDF_CON_NUM);
      case IDF_NO_ENTRY:    return (READ_IN_IDF_NO_ENTRY);
      default:              return ((ByTe_2) rV);
   }
}
