#include "libbase_udf.h"

/****************************************************************************
 *                                                                          *
 *                         IR_READ_DREC_TIMES SUBROUTINE                    *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This is the routine used to fill in the times in the structure        *
 *  returned by the read_drec() data access routine.                        *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    void *UDF                  memory location ptr for the structure that *
 *                               holds returned data values (read_drec)     *
 *    char full_swp              request 1 or all values in sensor set be   *
 *                               returned (for scalar parameters only)      *
 *                                                                          *
 *  USAGE                                                                   *
 *    x = ir_read_drec_times (UDF, full_swp)                                *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    ir_set_time()              determines the time associated with data   *
 *                               being returned                             *
 *    ir_sensor_set_time()       calculates the time associated with the    *
 *                               sensor set being processed                 *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info        structure holding information concerning   *
 *        ginfo                  the experiment that is being processed     *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data *ExDa structure that holds all of the currently       *
 *                               returned data values to be processed       *
 *    struct experiment_info     a pointer to the structure that holds      *
 *          *ex                  specific experiment information            *
 *    short days_in_yr           the number of days in the year             *
 *    char decrement_bday        flag indicating if start day needs to be   *
 *                               modified (if btime ends up negative)       *
 *    char decrement_eday        flag indicating if end day needs to be     *
 *                               modified (if etime ends up negative)       *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

void ir_read_drec_times (void *UDF, ByTe_1 full_swp)
{
  extern struct general_info ginfo;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  ByTe_2 days_in_yr;
  ByTe_1 decrement_bday, decrement_eday;

  /*************************************************************************/
  /*  Set the beginning time and ending time for this time sample.         */
  /*************************************************************************/
 
  ExDa = (struct idf_data *) UDF;
  ex = ginfo.expt;

  /*************************************************************************/
  /*  Was the beginning or ending time negative and was it corrected?      */
  /*************************************************************************/

  ir_set_time (full_swp, UDF, &decrement_bday, &decrement_eday);
  if (decrement_bday)
   {
     --ExDa->bday;

     /***************************************/
     /*  Check if previous year is crossed  */
     /***************************************/

     if (ExDa->bday < 1)
      {
        --ExDa->byear;

        /*********************************/
        /*  Take leap year into account  */
        /*********************************/

        if ((ExDa->byear % 4 == 0 && ExDa->byear % 100 != 0) ||
             ExDa->byear % 400 == 0)
           days_in_yr = 366;
        else
           days_in_yr = 365;
        ExDa->bday = days_in_yr;
      }
   }

  if (decrement_eday)
   {
     --ExDa->eday;

     /***************************************/
     /*  Check if previous year is crossed  */
     /***************************************/

     if (ExDa->eday < 1)
      {
        --ExDa->eyear;

        /*********************************/
        /*  Take leap year into account  */
        /*********************************/

        if ((ExDa->eyear % 4 == 0 && ExDa->eyear % 100 != 0) ||
             ExDa->eyear % 400 == 0)
           days_in_yr = 366;
        else
           days_in_yr = 365;
        ExDa->eday = days_in_yr;
      }
   }
 
  /*************************************************************************/
  /*  Adjust the time components for day boundary crossings.  The byear    */
  /*  and eyear elements are the same value BEFORE time adjustment.        */
  /*************************************************************************/

  if ((ExDa->byear % 4 == 0 && ExDa->byear % 100 != 0) || 
       ExDa->byear % 400 == 0)
     days_in_yr = 366;
  else
     days_in_yr = 365;

  ExDa->bday += ex->btime_ms / 86400000;
  ExDa->bmilli = ex->btime_ms % 86400000;
  ExDa->bnano = ex->btime_ns;
  if (ExDa->bday > days_in_yr)
   {
     ExDa->bday = 1;
     ExDa->byear += 1;
   }

  ExDa->eday += ex->etime_ms / 86400000;
  ExDa->emilli = ex->etime_ms % 86400000;
  ExDa->enano = ex->etime_ns;
  if (ExDa->eday > days_in_yr)
   {
     ExDa->eday = 1;
     ExDa->eyear += 1;
   }
}
