#include <stdio.h>
#include <math.h>
#include "libbase_udf.h"

/*
   integer floating point
      range: 10^-63 to 10^+63
      precision: 7 digits (0-16777215)
      mantissa * 10^exponent
      +-+-----------------------------------------------+-+-----------+
      |3|3|2|2|2|2|2|2|2|2|2|1|1|1|1|1|1|1|1|1|1|1| | | | | | | | | | |
      |1|0|9|8|7|6|5|4|3|2|1|0|9|8|7|6|5|4|3|2|1|0|9|8|7|6|5|4|3|2|1|0|
      +-+-----------------------------------------------+-+-----------+
      |s|         mantissa                              |s| exponent  |
      +-+-----------------------------------------------+-+-----------+
      Exponent:
         magnitude -> bits 0-5
         sign      -> bit 6
      Mantissa:
         magnitude -> bits 7-30
         sign      -> bit 31
*/

/*******************************************************************************
 *                                                                             *
 *                            IR_INTFP2FP  SUBROUTINE                          *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to convert the stored integer value back into     *
 *  floating point values.  Data is stored in integer form for IDFS files.     *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    unsigned long intfp     the integer value to convert to floating point   *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_intfp2fp (intfp)                                                  *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    exp10 ()                calculates 10 ** x                               *
 *    ir_tento ()             calculates 10 ** x (can use this or exp10)       *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    unsigned long m_sgn     the sign of the mantissa                         *
 *    unsigned long m_mag     the magnitude of the mantissa                    *
 *    unsigned long e_sgn     the sign of the exponent                         *
 *    unsigned long e_mag     the magnitude of the exponent                    *
 *    unsigned long intfp     the resultant integer value                      *
 *    union fp_u fp           holds the floating point and integer form of the *
 *                            value                                            *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ReaL_8 ir_intfp2fp (u_ByTe_4 intfp, ByTe_1 dtype)
{
   ReaL_8 sig, exp, mant; 
   u_ByTe_4 m_sgn,m_mag,e_sgn,e_mag;
   ReaL_8 fp, norm;
   ReaL_8 ebase;
   u_ByTe_1 byte1, byte2;
   

/**************************************************************************/
/*  The following are the definitions of the universal conversion factors */
/*  used:                                                                 */
/*                                                                        */
/*  sig    - the number of significant places in the mantissa.  It is     */
/*           to shift the decimal place of the exponent                   */
/*  ebase  - the exponent base                                            */
/*  norm   - any normalization factor supplied to the result              */
/**************************************************************************/


/**************************************************************************/
/*  Conversion of stored unsigned integer to floating point value         */
/*  according to the format.  This follows the following flow             */
/*                                                                        */
/*  1.  Get the sign of the mantissa                                      */
/*  2.  Get the mantissa itself                                           */
/*  3.  Get the sign of the exponent                                      */
/*  4.  Get the exponent itself                                           */
/*  5.  Set the number of significant digits in mantissa for this format  */
/*  6.  Set the exponent base                                             */
/*                                                                        */
/*  The schemes are                                                       */
/*     case 2:  IDFS 32 bit floating point                                */
/*     case 3:  IDFS 64 bit floating point                                */
/*     case 4:                                                            */
/*     case 5:  IDFS 16 bit floating point (PEACE MOMENTS TYPE)           */
/*     case 6:  IDFS 16 bit floating point (SCA INTERBALL TYPE)           */
/**************************************************************************/

   switch (dtype)
   {
      case 2:                                      /* single precision */
         m_sgn = (intfp >> 31) & 0x00000001;       /* extract mantissa */
         m_mag = (intfp >> 7) & 0x00ffffff;
   
         e_sgn = (intfp >> 6) & 0x00000001;        /* extract exponent */
         e_mag = intfp & 0x0000003f;

         sig = 7.0;
         ebase = 10.0;
         norm = 1.0;
   
         break;
      case 3:                                      /* double precision */
         m_mag = 0;
         e_mag = 0;
         m_sgn = 1;
         e_sgn = 0;
         sig   = 7;
         break;
      case 4:                                      /* half precision 1 */
         m_sgn = (intfp >> 15) & 0x00000001;       /* extract mantissa */
         m_mag = (intfp >> 7) & 0x000000ff;
   
         e_sgn = (intfp >> 6) & 0x00000001;        /* extract exponent */
         e_mag = intfp & 0x0000003f;

         sig = 2.0;
         ebase = 10.0;
         norm = 1.0;
   
         break;
      case 5:                                      /* half precision 2 */
         if (intfp != 0)
            m_mag = ((intfp >> 8) & 0x0000007f) + 128;
         else
            m_mag = 0;

         m_sgn = (intfp >> 15) & 0x00000001;    /* extract mantissa */
  
         e_sgn = 0;                             /* extract exponent */
         e_mag = (intfp & 0x000000ff);

         sig = 128;
         ebase = 2.0;

         /*  Although the diagram does not show this, MSSL verified that */
         /*  the mantissa must be divided by 256 since the mantissa is a */
         /*  fractional mantissa.                                        */

         norm = 256.0;
         break;

      case 6:                                      /* half precision 2 */
         byte1 = (intfp >> 8) & 0x000000ff;
         byte2 = intfp  & 0x000000ff;

         e_mag = byte1 & 0x3f;
         m_mag = (e_mag == 0) ? byte2 : byte2 + 256; 

         e_sgn = (byte1 >> 7) & 0x01;
         m_sgn = (byte1 >> 6) & 0x01;

         sig = 0.0;
         ebase = 2.0;
         norm = (e_mag == 0) ? 1 : 512.0;
   
         break;
      case 7:                                      /* IMAGE FP - IEEE? */
         m_sgn = (intfp >> 31) & 0x00000001;       /* extract mantissa */
         m_mag = intfp & 0x07fffff;
         e_sgn = 0;
         e_mag = (intfp >>23) & 255;
         if (e_mag == 255) {
           e_mag = 0;
           m_sgn = 1;
           e_sgn = (m_mag == 0) ? 0 : 1;
           m_mag = 0;
         } else { m_mag |= 0x800000; }

         sig = 127.0;
         ebase = 2.0;
         norm = 1 << 23;
      break;
      default:
         m_mag = 0;
         e_mag = 0;
         m_sgn = 1;
         e_sgn = 0;
         sig   = 7;
         break;
   }

/**************************************************************************/
/*  Form the floating point value                                         */
/*                                                                        */
/*  1.  If both mantissa and exponent are 0 then we have to check for     */
/*      special situations indicated by the sign bits                     */
/*      1a  MS=0, ES=0: value is 0.0                                      */
/*      1b  MS=1, ES=0: value is +infinity                                */
/*      1b  MS=1, ES=1: value is -infinity                                */
/*      1c  MS=0, ES=1: value is invalid                                  */
/*  2.  Otherwise                                                         */
/*  3.  Correct the manitssa for the sign                                 */
/*  4.  Correct the exponent for the sign                                 */
/*  5.  Convert number to value.  Mantissa is always assumed to have the  */
/*      decimal point to the left of the most significant digit.  This is */
/*      taken into account by shifting exponent to left by the maximum    */
/*      number of significant digits the format allows.                   */
/**************************************************************************/

   if ((m_mag == 0) && (e_mag == 0))            /* Special Values */
   {
      if ((m_sgn == 0) && (e_sgn == 0))         /* +0 or -0 */
         fp = 0.0;
      else if ((m_sgn == 1) && (e_sgn == 0))    /* +Infinity */
         fp = 3.402823e38;
      else if ((m_sgn == 1) && (e_sgn == 1))    /* -Infinity */
         fp = -3.402823e38;
      else                                      /* Not a Number */
         fp = 0.0;
   }
   else
   {
      mant = m_mag;
      exp  = e_mag;
      if (m_sgn == 1)
         mant = -mant;
      if (e_sgn == 1)
         exp = -exp;

      fp = mant * pow(ebase, -sig + exp) / norm;
   }
   return (fp);
}

/*******************************************************************************
 *                                                                             *
 *                             IR_TENTO  SUBROUTINE                            *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine returns the power of 10 value associated with the           *
 *  exponential value passed into this routine as a parameter.                 *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    char cvar                the power of 10 desired (the exponent value)    *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_tento (cvar)                                                      *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    float ten                the value 10                                    *
 *    float res                the result (passed back to calling routine)     *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ReaL_8 ir_tento (ByTe_1 cvar)
{
   ReaL_8 ten = 10.0;
   ReaL_8 res =  1.0;

   switch (cvar) {
      case  15: res = 1.0e15; break;
      case  14: res = 1.0e14; break;
      case  13: res = 1.0e13; break;
      case  12: res = 1.0e12; break;
      case  11: res = 1.0e11; break;
      case  10: res = 1.0e10; break;
      case   9: res =  1.0e9; break;
      case   8: res =  1.0e8; break;
      case   7: res =  1.0e7; break;
      case   6: res =  1.0e6; break;
      case   5: res =  1.0e5; break;
      case   4: res =  1.0e4; break;
      case   3: res =  1.0e3; break;
      case   2: res =  1.0e2; break;
      case   1: res =  1.0e1; break;
      case   0: res =  1.0e0; break;
      case  -1: res =  1.0e-1; break;
      case  -2: res =  1.0e-2; break;
      case  -3: res =  1.0e-3; break;
      case  -4: res =  1.0e-4; break;
      case  -5: res =  1.0e-5; break;
      case  -6: res =  1.0e-6; break;
      case  -7: res =  1.0e-7; break;
      case  -8: res =  1.0e-8; break;
      case  -9: res =  1.0e-9; break;
      case -10: res =  1.0e-10; break;
      case -11: res =  1.0e-11; break;
      case -12: res =  1.0e-12; break;
      case -13: res =  1.0e-13; break;
      case -14: res =  1.0e-14; break;
      case -15: res =  1.0e-15; break;
      default:
         if (cvar <= 0) {
           while (cvar++ < 0)
              res /= ten;
         } else {
            while (cvar-- > 0)
              res *= ten;
         }
      break;
   }
   return (res);
}
