#include <stdlib.h>
#include <stdio.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_INIT_SENSOR_PTR SUBROUTINE                    *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to initialize the array of sensor_table          *
 *  structures which hold information pertinent to the sensor combinations    *
 *  found.  The table information, critical status byte, critical status      *
 *  operation, and array of critical actions are retrieved for each unique    *
 *  sensor combination found.                                                 *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    short num_combo          the number of unique table combinations needed *
 *                             for all sensors for the virtual instrument     *
 *                             being processed                                *
 *    char *crB                an array holding the critical action bytes     *
 *                             for table                                      *
 *    char FirsT               flag indicating the first time this module is  *
 *                             called for the combination being processed     *
 *    char *cInfo              an array of 1 flag per table which indicates   *
 *                             if there are critical action values present    *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_init_sensor_ptr (num_combo, crB, FirsT, cInfo)                   *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    ir_get_tbl_info ()       retrieves the various table(s) information for *
 *                             the sensor being processed                     *
 *    ir_get_critical_info()   retrieves the critical status operation and    *
 *                             critical action array for applicable sensors   *
 *    sizeof ()                the size of the specified object in bytes      *
 *    malloc()                 allocates memory                               *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info   a pointer to the structure that holds specific *
 *          *ex                experiment information                         *
 *    struct sensor_tables     a pointer to the structure which holds the     *
 *          *sptr              various information for the tables utilized by *
 *                             sensor-table combination being processed       *
 *    struct crit_action_info  a pointer to the structure that holds info.    *
 *       *cr_act_ptr           pertinent to table(s) that are flagged as      *
 *                             critical action table(s)                       *
 *    register short i, j      looping variables                              *
 *    register short aIndx     index into the array of structures that hold   *
 *                             critical action information                    *
 *    unsigned int B           the number of bytes to allocate                *
 *    short rV                 holds the value returned by the called routine *
 *    short cstat              the critical status byte of interest           *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_init_sensor_ptr (ByTe_2 num_combo, ByTe_1 *crB, ByTe_1 FirsT, 
                                          ByTe_1 *cInfo)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct sensor_tables *sptr;
   struct crit_action_info *cr_act_ptr;
   register ByTe_4 i, j;
   register ByTe_2 *s1;
   register ByTe_1 *c1, *c2;

   ByTe_2  sen, aIndx;
   size_t B;
   ByTe_4 nSen, nTbl;
   ByTe_2 rV, cstat;

   ex = ginfo.expt;
   nSen = ex->num_sensor;
   nTbl = ex->num_tbls;

  /*************************************************************************/
  /*  Initialize the structure(s) holding the unique combination offsets.  */
  /*  First find the largest TDW_LEN for each combination being processed. */
  /*************************************************************************/

   for (i = 0; i < num_combo; ++i) {
      sptr = ex->sen_tbl_ptr + i;
      sptr->max_tdw_len = 0;

  /*************************************************************************/
  /*  Loop over all sensors for this virtual instrument and look only at   */
  /*  those sensors which use this set of table combinations.              */
  /*************************************************************************/

      s1 = ex->index_sen_tbl;
      c1 = ex->sen_tdw_len;
      for (j = 0; j < nSen; ++j, ++s1, ++c1) {
         if (*s1 == i) {
            if (*c1 > sptr->max_tdw_len) { sptr->max_tdw_len = *c1; }
         }
      }
   }

  /************************************************************************/
  /*  Loop over each combination being processed.                         */
  /************************************************************************/

   for (i = 0; i < num_combo; ++i) {
      sptr = ex->sen_tbl_ptr + i;

  /************************************************************************/
  /*  If there are no tables in this virtual which have critical status   */
  /*  bytes assocated with them then we need no memory to hold any of     */
  /*  that information.                                                   */
  /************************************************************************/

      if (ex->crit_action == 0) {
         sptr->base_crit_action = NO_MEMORY;
         sptr->act_ptr = NO_MEMORY;
       } else {

  /************************************************************************/
  /*  At least one table has a critical action dependency.  Allocate      */
  /*  structures to hold the critical action info for each table that     */
  /*  UTILIZES critical action.  Allocate each time called since memory   */
  /*  may have been freed.                                                */
  /************************************************************************/

         B = ex->crit_action * sizeof(struct crit_action_info);
         if ((sptr->base_crit_action = malloc (B)) == NO_MEMORY)
            return (CRIT_ACT_MALLOC);
         sptr->act_ptr = (struct crit_action_info *)sptr->base_crit_action;
         cr_act_ptr = sptr->act_ptr;

         c1 = cInfo;
         for (j = 0; j < nTbl; ++j, ++c1) {
            if (*c1 != 0) {
               cr_act_ptr->crit_tbl = j;
               if (FirsT) {
                  cr_act_ptr->base_action = NO_MEMORY;
                  cr_act_ptr->crit_action = NO_MEMORY;
               }
               ++cr_act_ptr;
            }
         }
      }

  /************************************************************************/
  /*  Find the first sensor that utilizes this combination.               */
  /************************************************************************/

/*
      s1 = ex->index_sen_tbl;
      for (sen = 0; sen < nSen; ++sen, ++s1) {
         if (*s1 == sen) { break; }
      }
*/

      s1 = ex->index_sen_tbl + i;
      for (sen = 0; sen < nSen; ++sen) {
         if (*s1 == sen) { break; }
      }

  /************************************************************************/
  /*  Get the various table information for this sensor.                  */
  /************************************************************************/

      rV = ir_get_tbl_info (sen, i);
      if (rV != ALL_OKAY) { return (rV); }

  /************************************************************************/
  /*  Get the critical status operation and actions if there are any      */
  /*  critical status bytes defined.  The crit_flag is set to -1 so that  */
  /*  the first time a header record is read, the correct tables will be  */
  /*  read, based on the mode of the instrument.                          */
  /************************************************************************/

      if (sptr->act_ptr != NO_MEMORY) {
         cr_act_ptr = sptr->act_ptr;
         aIndx = 0;
	 c1 = crB;
	 c2 = cInfo;
         for (j = 0; j < nTbl; ++j, ++c2, c1 += nSen) {
            if (*c2 != 0) {
               cr_act_ptr->crit_flag = -1;
               cr_act_ptr->crit_offset = -1;
               cr_act_ptr->crit_status = *(c1 + sen);
               cstat = *(c1 + sen);

               if (cstat != -1) {
                  rV = ir_get_critical_info (i, sen, (ByTe_1) j, aIndx);
                  if (rV != ALL_OKAY) { return (rV); }
               } else {

  /************************************************************************/
  /*  The critical status value for this sensor is -1, which means no     */
  /*  action is necessary for this status byte.                           */
  /************************************************************************/

                  cr_act_ptr->base_action = NO_MEMORY;
                  cr_act_ptr->crit_action = NO_MEMORY;
               }
               ++cr_act_ptr;
               ++aIndx;
            }
         }
      }
   }

  return (ALL_OKAY);
}
