#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                           IR_GET_MODE_TBL  SUBROUTINE                       *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called in order to retrieve the table information for    *
 *  all integer mode tables.  Ascii tables are not considered in this module;  *
 *  the ascii information is kept in a different structure if it is needed.    *
 *  Since the number of tables vary from virtual instrument to virtual         *
 *  instrument, the space to hold the table information must be allocated      *
 *  as the information becomes known.  The way modes work is like this: there  *
 *  is one MODE_TABLES structure for each mode.  Each MODE_TABLES structure    *
 *  contains one TBL_FMT, TBL_VAR, TBL_EXPAND, TBL_NUM and TBL_SIZE for each   *
 *  mode table defined.  The table is the table for the mode in question.      *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    short mode               the mode being processed                        *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_get_mode_tbl (mode)                                               *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    sizeof ()                the size of the specified object in bytes       *
 *    free ()                  frees allocated memory                          *
 *    malloc()                 allocates memory                                *
 *    read_idf()               reads information from the IDF file             *
 *    ir_mode_tbl_size ()      determines the total number of bytes needed to  *
 *                             hold all integer table values for all tables    *
 *    ir_get_new_tbl()         reads the requested table for the mode from     *
 *                             the VIDF file                                   *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info      structure that holds information concerning     *
 *        ginfo                the experiment that is being processed          *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info   a pointer to the structure that holds           *
 *          *ex                specific experiment information                 *
 *    struct mode_tables       a pointer to the structure which holds the      *
 *         *mptr               mode-dependent table information                *
 *    register short i         looping variable                                *
 *    unsigned int bytes       the number of bytes to allocate                 *
 *    long ret_val             the value returned by the called routine        *
 *    long offset              indexing variable                               *
 *    short rval               holds the value returned by called routine      *
 *    char tbl_ind             index to get to correct mode_tables structure   *
 *    char which_tbl           the table number being processed                *
 *    void *tmp_ptr            pointer which holds address passed back by the  *
 *                             call to the MALLOC routine                      *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_get_mode_tbl (ByTe_2 mode)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct mode_tables *mptr;
   register ByTe_2 i;
   size_t bytes;
   ByTe_4 ret_val, offset;
   ByTe_2 rval;
   ByTe_1 tbl_ind, which_tbl;
   void *tmp_ptr;

   /*************************************************************************/
   /*  Point to the correct mode_tables structure.                          */
   /*************************************************************************/

   ex = ginfo.expt;
   mptr = ex->mode_tbl_ptr + mode;
   mptr->base_tbls = NO_MEMORY;
   mptr->base_misc_tbls = NO_MEMORY;
   mptr->num_tbls = 0;

   /************************************************************************/
   /*  Include only non-ascii tables when counting the number of tables to */
   /*  be applied.                                                         */
   /************************************************************************/

   for (i = 0; i < ex->num_tbls; ++i)
    {
      if (*(ex->tbl_type + i) == 1)
        continue;

      if (*(ex->tbl_var + i) == 4 || *(ex->tbl_var + i) == 5)
        ++mptr->num_tbls;
    }

   /************************************************************************/
   /*  Pick up all INTEGER tables.                                         */
   /************************************************************************/

   if (mptr->num_tbls != 0)
    {
      /***********************************************************************/
      /*  Allocate the space to hold the var, expand, fmt, size, tbl_num and */
      /*  pointer to the table values for each integer table.                */
      /***********************************************************************/
   
      if (mptr->base_misc_tbls != NO_MEMORY)
       {
         free (mptr->base_misc_tbls);
         mptr->base_misc_tbls = NO_MEMORY;
       }
      bytes = (sizeof (ByTe_4) + sizeof (ReaL_4 *) + sizeof (ByTe_1) + 
          sizeof (ByTe_1) + sizeof (ByTe_1) + sizeof (ByTe_1)) * mptr->num_tbls;
      if (bytes <= 0)
        return (MODE_TBL_MISC_MALLOC);
  
      if ((tmp_ptr = malloc (bytes)) == NO_MEMORY)
        return (MODE_TBL_MISC_MALLOC);
      mptr->base_misc_tbls = tmp_ptr;
   
      /***********************************************************************/
      /*  Set various pointers to the space allocated to hold table info.    */
      /***********************************************************************/
   
      mptr->tbl_size = (ByTe_4 *) mptr->base_misc_tbls;
      offset = mptr->num_tbls * sizeof (ByTe_4); 
      mptr->tbl_ptr = (float **) ((ByTe_1 *)mptr->base_misc_tbls + offset);
      offset += mptr->num_tbls * sizeof (float *); 
      mptr->tbl_var = (ByTe_1 *) ((ByTe_1 *)mptr->base_misc_tbls + offset);
      offset += mptr->num_tbls * sizeof (ByTe_1);
      mptr->tbl_expand = (ByTe_1 *) ((ByTe_1 *)mptr->base_misc_tbls + offset);
      offset += mptr->num_tbls * sizeof (ByTe_1);
      mptr->tbl_fmt = (ByTe_1 *) ((ByTe_1 *)mptr->base_misc_tbls + offset);
      offset += mptr->num_tbls * sizeof (ByTe_1);
      mptr->tbl_num = (ByTe_1 *) ((ByTe_1 *)mptr->base_misc_tbls + offset);
   
      /***********************************************************************/
      /*  Loop over all defined tables for this virtual instrument.          */
      /***********************************************************************/
   
      for (i = 0, tbl_ind = 0; i < ex->num_tbls; ++i)
       {
         /********************************************************************/
         /*  Integer information is processed, not ASCII information.        */
         /********************************************************************/
   
         if (*(ex->tbl_type + i) == 1)
           continue;

         if (*(ex->tbl_var + i) == 4 || *(ex->tbl_var + i) == 5)
          {
            *(mptr->tbl_num + tbl_ind) = (ByTe_1) i;
            *(mptr->tbl_var + tbl_ind) = *(ex->tbl_var + i);
   
            ret_val = read_idf (ex->data_key, ex->exten, ex->version, 
                                (ByTe_1 *)(mptr->tbl_fmt + tbl_ind),
                                _TblFmT, i, (ByTe_4) mode, 1);
            if (ret_val < 0)
             {
               switch (ret_val)
                {
                   case IDF_NOT_FOUND:
                     return (MODE_TBL_IDF_NOT_FOUND);
                   case IDF_MANY_BYTES:
                     return (MODE_TBL_IDF_MANY_BYTES);
                   case IDF_TBL_NUM:
                     return (MODE_TBL_IDF_TBL_NUM);
                   case IDF_CON_NUM:
                     return (MODE_TBL_IDF_CON_NUM);
                   case IDF_NO_ENTRY:
                     return (MODE_TBL_IDF_NO_ENTRY);
                   default:
                     return ((ByTe_2) ret_val);
                }
             }

            /******************************************************************/
            /*  New error checks added.  If tbl_fmt = 0, only raw data can be */
            /*  used as look up values into the table.                        */
            /******************************************************************/

            if (*(mptr->tbl_fmt + tbl_ind) == 0)
             {
               /****************************************************************/
               /*  Only raw calibration data allowed for sweep length dependent*/
               /*  tables.                                                     */
               /****************************************************************/  

               if (*(ex->tbl_type + i) == 2)
                 return (MODE_TBL_VAR_NOT_CAL);

               /****************************************************************/
               /*  TBL_VAR = 5 means processed data, which is not allowed, must*/
               /*  be raw.                                                     */
               /****************************************************************/  
               else if (*(ex->tbl_type + i) == 0 && *(ex->tbl_var + i) == 5)
                 return (MODE_TBL_VAR_NOT_RAW);
             }
   
            /*****************************************************************/
            /*  The TBL_EXPAND flag pertains to polynomial coefficients only.*/
            /*  Always set the flag to a 1 for lookup tables.  For the tables*/
            /*  defined for each step, do not expand coefficients for now    */
            /*  for sake of size.                                            */
            /*****************************************************************/
   
            if (*(mptr->tbl_fmt + tbl_ind) > 0)
              *(mptr->tbl_expand + tbl_ind) = (*(ex->tbl_type + i) == 2) ? 
                                                   0 : *(ex->tbl_expand + i);
            else
              *(mptr->tbl_expand + tbl_ind) = 1;

            ++tbl_ind;
          }
       }
    
      /*********************************************************************/
      /*  Determine how much space to allocate for INTEGER table values.   */
      /*  Ascii strings are saved in a different structure.                */
      /*********************************************************************/
   
      rval = ir_mode_tbl_size (mode, &bytes);
      if (rval != ALL_OKAY)
        return (rval);

      /***********************************************************************/
      /*  If all TBL_FMT values are -1, no space needs to be allocated to    */
      /*  hold values and the pointers can be set to indicate no memory.     */
      /***********************************************************************/
   
      if (bytes == 0)
       {
         mptr->base_tbls = NO_MEMORY;
         mptr->tbls = NO_MEMORY;
       }
      
      /*********************************************************************/
      /*  Allocate the space to hold the different table values.           */
      /*********************************************************************/
   
      else
       {
         if (mptr->base_tbls != NO_MEMORY)
          {
            free (mptr->base_tbls);
            mptr->base_tbls = NO_MEMORY;
          }
         if (bytes <= 0)
           return (MODE_TBL_MALLOC);

         if ((tmp_ptr = malloc (bytes)) == NO_MEMORY)
           return (MODE_TBL_MALLOC);
         mptr->base_tbls = tmp_ptr;
         mptr->tbls = (float *) mptr->base_tbls;
   
         /********************************************************************/
         /*  Loop over all tables for this mode.                             */
         /********************************************************************/
   
         offset = 0;
         for (i = 0; i < mptr->num_tbls; ++i)
          {
            which_tbl = *(mptr->tbl_num + i);

            /************************************************************/
            /*  If the format is negative, this table is not used.      */
            /*  Otherwise, retrieve the table values.                   */
            /************************************************************/

            if (*(mptr->tbl_fmt + i) >= 0)
             {
               *(mptr->tbl_ptr + i) = mptr->tbls + offset;
               rval = ir_get_new_tbl (mode, which_tbl, *(mptr->tbl_expand + i), 
                                  *(mptr->tbl_fmt + i), *(mptr->tbl_ptr + i), 
                                  *(mptr->tbl_size + i), (ByTe_4) -1);

               if (rval != ALL_OKAY)
                 return (rval);

               offset += *(mptr->tbl_size + i);
             }
          }
       }
    }

   return (ALL_OKAY);
}
