#include <string.h>
#include "util_str.h"
#include "user_defs.h"
#include "libbase_udf.h"
#include "ret_codes.h"
#include "OpSySInD.h"

/****************************************************************************
 *                                                                          *
 *                       IR_FILL_PITCH_ANGLES SUBROUTINE                    *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine is called in order to calculate the pitch angles for     *
 *  a given sensor.  One value is returned for each element of the sweep in *
 *  the parent data source.  The time range for each sweep step is computed *
 *  and data from the IDFS pitch angle source is acquired for each time     *
 *  range.  To be consistent with read_drec in regards to time advancement, *
 *  the pitch angle data is retrieved when the time advancement flag says   *
 *  to advance in time.  New data will be read the NEXT time this routine   *
 *  is called, except for the very first call, just like read_drec works.   *
 *                                                                          *
 *  INPUT VARIABLES                                                         *
 *    char fwd                flag that indicates when to advance to the    *
 *                            time sample                                   *
 *    char full_swp           flag that indicates if 1 value is being       *
 *                            requested or all values for the record (for   *
 *                            a scalar parameter only)                      *
 *    void *UDF               ptr to the memory location for the structure  *
 *                            that holds returned data values (read_drec)   *
 *                                                                          *
 *  USAGE                                                                   *
 *    x = ir_fill_pitch_angles (fwd, full_swp, UDF)                         *
 *                                                                          *
 *  NECESSARY SUBPROGRAMS                                                   *
 *    adjust_time ()          corrects time elements if too many seconds    *
 *                            (day crossing) or too many days(year crossing)*
 *    strcpy()                copies a string to another string variable    *
 *    ir_locate_ex()          determines if the requested combination has   *
 *                            already been processed and points to the      *
 *                            correct structure allocated for the combo     *
 *    ir_pitch_angle_sweep_times()  determines the time of each element in  *
 *                            the sweep                                     *
 *    ir_acquire_pitch_data ()   acquires the data from the 3 components of *
 *                            pitch angle source                            *
 *    ir_compute_pitch_angle ()  computes pitch angle for each sweep step   *
 *                                                                          *
 *  EXTERNAL VARIABLES                                                      *
 *    struct general_info     structure that holds information concerning   *
 *        ginfo               the experiment that is being processed        *
 *                                                                          *
 *  INTERNAL VARIABLES                                                      *
 *    struct idf_data         structure that holds all of the currently     *
 *       *ExDa                returned data values to be processed          *
 *    struct experiment_info  a pointer to the structure that holds         *
 *       *ex, *ex_new         specific experiment information               *
 *    struct ptr_rec *ptr     a pointer to the structure which holds all    *
 *                            pointers to the header and data for the       *
 *                            experiment of interest                        *
 *    struct pitch_info       a pointer to the structure that holds pitch   *
 *        *pa_ptr             angle information                             *
 *     struct pitch_times     structure that holds the time period for the  *
 *         src_time           current delta-t being processed               *
 *    reg long *btime_ptr,    pointers to the start time / end time memory  *
 *       *etime_ptr           locations                                     *
 *    reg long *time_end      loop termination variable                     *
 *    register double *dptr   ptr to the data for the 3 pitch components    *
 *    reg double *dptr_end    loop termination variable                     *
 *    register float *fptr    ptr to normalization factors                  *
 *    unsigned long old_dkey  data key for parent source                    *
 *    unsigned short max_ele  the number of elements in the sweep           *
 *    uns short old_version   version number for parent source              *
 *    uns short swp_step      current step of the sweep being processed     *
 *    short ret_val           holds value returned by called module         *
 *    char use_end_yr_day     flag indicating if end year/day values are to *
 *                            be used                                       *
 *    char old_exten[]        filename extension for parent source          *
 *    char same_times         flag indicating if the same time range is     *
 *                            applicable to all steps of the vector         *
 *                                                                          *
 *  SUBSYSTEM                                                               *
 *    Display Level                                                         *
 *                                                                          *
 ***************************************************************************/

ByTe_2 ir_fill_pitch_angles (ByTe_1 fwd, ByTe_1 full_swp, void *UDF)
{
  extern struct general_info ginfo;

  struct idf_data *ExDa;
  struct experiment_info *ex, *ex_new;
  struct ptr_rec *ptr;
  struct pitch_info *pa_ptr;
  struct pitch_times src_time;
  register ByTe_4 *btime_ptr, *etime_ptr, *time_end;
  register ReaL_8 *dptr, *dptr_end;
  register ReaL_4 *fptr;
  u_ByTe_4 old_dkey;
  u_ByTe_2 max_ele, old_version, swp_step;
  ByTe_2 ret_val;
  ByTe_1 use_end_yr_day, old_exten[3], same_times;

  /***********************************************************************/
  /* Set a pointer to the structure which holds all pointers for header  */
  /* and data information for thxperiment currently being processed.     */
  /***********************************************************************/

  ex = ginfo.expt;
  ptr = ex->info_ptr;
  pa_ptr = ex->pitch_angles;
  ExDa = (struct idf_data *) UDF;
  old_dkey = ex->data_key;
  old_version = ex->version;
  strcpy (old_exten, ex->exten);

  /***************************************************************************/
  /*  Since read_drec is called when positioning a file, calculate pitch     */
  /*  angles ONLY if the pitch angle IDFS source has been positioned.        */
  /***************************************************************************/

  ret_val = ir_locate_ex (pa_ptr->data_key, pa_ptr->exten, pa_ptr->version, 0);
  ex_new = ginfo.expt;
  if (pa_ptr->file_status != ALL_OKAY || !ex_new->fnext)
   {
     ExDa->num_pitch = 0;
     ret_val = ir_locate_ex (old_dkey, old_exten, old_version, 0);
     return (ALL_OKAY);
   }

  /************************************************************************/
  /* For a full swp, max_ele is not just set to N_SAMPLE because the user */
  /* could have asked for one at a time and then changed to a full swp to */
  /* get the remainder of the elements.                                   */
  /************************************************************************/

  max_ele = (ex->smp_id == 2 && !full_swp) ? 1 : *ptr->N_SAMPLE - ptr->time_row;

  /************************************************************************/
  /*  Does the pitch angle data need to be retrieved?  This is necessary  */
  /*  due to time advancement on the parent data set (fwd flag sent to    */
  /*  read_drec).                                                         */
  /************************************************************************/

  if (pa_ptr->get_pa_data)
   {
     pa_ptr->get_pa_data = 0;
     ir_pitch_angle_sweep_times (ex, ptr, pa_ptr, max_ele, UDF,
                                 &use_end_yr_day, &same_times);

     /************************************************************************/
     /*  Initialize data values and normalization factors.                   */
     /************************************************************************/

     fptr = pa_ptr->time_frac;
     dptr = pa_ptr->data_val;
     dptr_end = pa_ptr->data_val + ex->swp_len * 3;
     for (; dptr < dptr_end; )
      {
        *dptr++ = OUTSIDE_MIN;
        *fptr++ = -1.0;
      }

     /************************************************************************/
     /*  Determine the time of each element in the sweep and retrieve the    */
     /*  pitch angle data for that time component.                           */
     /************************************************************************/
   
     swp_step = 0;
     etime_ptr = pa_ptr->etime_ms;
     btime_ptr = pa_ptr->btime_ms;
     time_end = pa_ptr->btime_ms + max_ele;
     for (; btime_ptr < time_end; ++btime_ptr, ++etime_ptr, ++swp_step)
      {
        /*********************************************************************/
        /*  Since the time values are computed in terms of milliseconds, set */
        /*  the nanosecond component to zero.                                */
        /*********************************************************************/
   
        src_time.byear = ExDa->byear;
        src_time.bday = ExDa->bday;
        src_time.bmilli = *btime_ptr;
        src_time.bnano = 0;
        adjust_time (&src_time.byear, &src_time.bday, &src_time.bmilli, 2);
   
        src_time.eyear = (use_end_yr_day) ? ExDa->eyear : ExDa->byear;
        src_time.eday = (use_end_yr_day) ? ExDa->eday: ExDa->bday;
        src_time.emilli = *etime_ptr;
        src_time.enano = 0;
        adjust_time (&src_time.eyear, &src_time.eday, &src_time.emilli, 2);
   
        if (src_time.emilli < src_time.bmilli && src_time.eday == src_time.bday) 
          return (PA_BAD_TIMES);

        ret_val = ir_acquire_pitch_data (src_time, pa_ptr, swp_step, same_times);
        if (ret_val != ALL_OKAY)
         {
           ExDa->num_pitch = 0;
           ret_val = ir_locate_ex (old_dkey, old_exten, old_version, 0);
           return (ALL_OKAY);
         }
      }
   }

  /***************************************************************************/
  /*  Compute the pitch angle for each element of the sweep.                 */
  /***************************************************************************/

  ir_compute_pitch_angle (pa_ptr, UDF, ExDa->sensor, max_ele);
  ExDa->num_pitch = max_ele;
  ret_val = ir_locate_ex (old_dkey, old_exten, old_version, 0);

  /***************************************************************************/
  /*  If read_drec was called with the time advancement flag set, reset the  */
  /*  pitch angle data retrieval flag for NEXT call.  In this way, it is     */
  /*  consistent with the read_drec time advancement philosophy.             */
  /***************************************************************************/

  if (fwd)
    pa_ptr->get_pa_data = 1;
  return (ALL_OKAY);
}
