#include "util_str.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                         IR_COPY_PITCH_ANGLE SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to process pitch angle data when the time range   *
 *  for each element of the sweep are the same (parallel acquisition).  In     *
 *  this case, element zero of the pitch angle array is computed, while all    *
 *  other elements are simply re-computed using the data for the three         *
 *  components that was acquired for element zero.                             *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    struct pitch_info *pa_ptr    a pointer to the structure that holds pitch *
 *                                 angle information                           *
 *    unsigned short swp_step      current step of the sweep being processed   *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_copy_pitch_angle (pa_ptr, swp_step)                               *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    None                                                                     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    None                                                                     *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    register short i             looping variable                            *
 *    register double *dptr        pointer to data for the three components    *
 *                                 for the step being processed                *
 *    register double *base_dptr   pointer to data for the three components    *
 *                                 for element zero                            *
 *    register float *fptr         pointer to the normalization factors for    *
 *                                 the step being processed                    *
 *    register float *base_fptr    pointer to the normalization factors for    *
 *                                 element zero                                *
 *    float time_frac              the percentage of the data to be included   *
 *                                 for the delta-t being processed             *
 *    double dval[3]               the data values to be copied into the step  *
 *                                 being processed                             *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

void ir_copy_pitch_angle (struct pitch_info *pa_ptr, u_ByTe_2 swp_step)
{
   register ByTe_2 i;
   register ReaL_8 *dptr, *base_dptr;
   register ReaL_4 *fptr, *base_fptr;
   ReaL_8 dval[3];
   ReaL_4 time_frac = 0.0;

   /*************************************************************************/
   /*  Always point to element zero of the memory location and retrieve the */
   /*  data.  Set frac to one since the value in location zero has already  */
   /*  been multiplied by frac.                                             */
   /*************************************************************************/

   base_dptr = pa_ptr->data_val;
   base_fptr = pa_ptr->time_frac;
   for (i = 0; i < 3; ++i) 
    {
      dval[i] = *(base_dptr + i);

      if (*(base_fptr + i) < 0.0)
        continue;
      dval[i] /= *(base_fptr + i);
    }
   time_frac = 1.0;

   /***************************************************************************/
   /*  Copy the values from element zero to the current step being processed. */
   /***************************************************************************/

   dptr = pa_ptr->data_val + (3 * swp_step);
   fptr = pa_ptr->time_frac + (3 * swp_step);
   for (i = 0; i < 3; ++i)
    {
      *(dptr + i) = dval[i];
      *(fptr + i) = time_frac;
    }
}
