#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "user_defs.h"
#include "libbase_udf.h"
#include "OpSySInD.h"

/*****************************************************************************
 *                                                                           *
 *                        IR_CONVERT_SENSOR_DATA SUBROUTINE                  *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine converts sensor data into physical units by applying the  *
 *  specified tables and the table operations in the specified order to      *
 *  achieve the desired output.  The order is implied by the table           *
 *  specification.  If the chk_fill flag is set to 1, the data values will   *
 *  be compared against the specified FILL value.  If the data value is      *
 *  identified as a fill value, the value OUTSIDE_MIN will be returned and   *
 *  it is up to the user to decide what is to be done with this value.  The  *
 *  FILL value must be specified in terms of raw telemetry.                  *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    char num_tbls                number of tables to apply                 *
 *    char *tbls_to_apply          array of tables to be applied             *
 *    short *tbl_oper              array of operations to be applied to      *
 *                                 tables                                    *
 *    char chk_fill                data to be checked for FILL value flag    *
 *    float *DataOut               array holding the returned data in        *
 *                                 converted units                           *
 *    long fill_value              the value identified as the FILL value    *
 *    void *UDF                    ptr to memory location for structure that *
 *                                 holds returned data values (read_drec)    *
 *                                                                           *
 *  USAGE                                                                    *
 *    ir_convert_sensor_data (num_tbls, tbls_to_apply, tbl_oper,             *
 *                                  &DataOut, chk_fill, fill_value, UDF)     *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    ir_transfer_data ()          converts raw integer telemetry to the     *
 *                                 raw units (integer or float)              *
 *    ir_idf_ex_func ()            performs higher order table operations    *
 *    abs()                        returns the absolute value of a number    *
 *    IDF_OPER[]._math_drv ()      performs arithmetic operation defined     *
 *                                 for the table being processed             *
 *    CombineOp()                  performs defined arithmetic operations    *
 *                                 to combine the permanent and temporary    *
 *                                 accumulators                              * 
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info ginfo    structure holding information concerning  *
 *                                 the experiment that is being processed    *
 *    struct def_oper IDF_OPER[]   array of pointers to the routines that    *
 *                                 perform arithmetic operation defined for  *
 *                                 the sensor, sweep, and cal. data tables   *
 *    void *Tmp_Space              scratch space for various uses            *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct idf_data *ExDa        structure holding all of the currently    *
 *                                 returned data values to be processed      *
 *    struct experiment_info *ex   a pointer to the structure holding        *
 *                                 specific experiment information           *
 *    struct sensor_tables *sptr   a pointer to the structure holding the    *
 *                                 various information for tables utilized   *
 *                                 by current sensor-table combination       *
 *    register unsigned short *s1  pointer to number in each cal set info.   *
 *    register char i              looping variable                          *
 *    void *Scratch                scratch space for various uses            *
 *    long data_start              offset into the data at which to start    *
 *                                 picking up data values (used for cal.)    *
 *    long num_data                the number of data values being returned  *
 *    long hold_val                number of successive data values within   *
 *                                 a data set that one calibration value is  *
 *                                 to be applied to                          *
 *    long num_ele_in_tbl          no. of elements in table being applied    *
 *    unsigned short *s_end        loop termination variable                 *
 *    ByTe_2 BasOp                 basic table operation                     *
 *    ByTe_2 ExOp                  extended (after table) operation          *
 *    ByTe_2 BufOp                 buffer operation modifier                 * 
 *    ByTe_2 PosOp                 positive operator                         * 
 *    char *sen                    pointer to sensor data from READ_DREC()   *
 *    char *swp                    pointer to sweep data from READ_DREC()    *
 *    char *cal                    pointer to cal. data from READ_DREC()     *
 *    char dtype                   format of the data - uns int, int, float  *
 *    char fn_of_raw               flag indicating if table is a function    *
 *                                 of raw data or processed data             *
 *    char *DFrom                  pointer to data values which the table    *
 *                                 values are to be applied to               *
 *    char which_cal               the calibration set being processed       *
 *    char *mode                   pointer to mode data from READ_DREC()     *
 *    float *DataFrom              this is the input data array              *
 *    float *DataTo                this is the output data array             *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 ****************************************************************************/

ByTe_2 ir_convert_sensor_data (ByTe_1 num_tbls, ByTe_1 *tbls_to_apply, 
                             ByTe_2 *tbl_oper, ReaL_4 *DataOut, ByTe_1 IFmt, 
                             ByTe_1 chk_fill, ByTe_4 fill_value, void *UDF)
{
  extern struct general_info ginfo;
  extern struct def_oper IDF_OPER[];
  extern void   *Tmp_Space;
  extern size_t Tmp_Bytes;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  struct sensor_tables *sptr;
  register ReaL_8   *d1;
  register u_ByTe_2 *s1;
  register ByTe_2   *OP, I;
  register ByTe_1   *TbL;
  void *Scratch;
  ReaL_8 *RetD, *RetB, *dEnD;
  ReaL_8 *DataFrom, *DataTo; 
  ReaL_8 SenNum[2];
  ReaL_4 *TblVals, *f1;

  ByTe_4 *l1, data_start, num_data, hold_val, num_ele_in_tbl;
  ByTe_4 T, F;
  size_t bytes;
  u_ByTe_2 *s_end;
  ByTe_2 BasOp, ExOp, BufOp, PosOp;
  ByTe_2 A, C, R;
  ByTe_1 *sen, *swp, *cal, dtype, fn_of_raw;
  ByTe_1 *DFrom, which_cal, *mode;
  ByTe_1 tV;

  /**********************************************************************/
  /*  Set pointers to sensor, sweep step, calibration and status (mode) */
  /*  data returned by the READ_DREC() routine.                         */
  /**********************************************************************/

  ExDa = (struct idf_data *) UDF;
  sen = (ByTe_1 *) ExDa->sen_data;
  swp = (ByTe_1 *) ExDa->swp_data;
  cal = (ByTe_1 *) ExDa->cal_data;
  mode = (ByTe_1 *) ExDa->mode;

  /**********************************************************************/
  /*  Set pointers to the current sensor being processed.               */
  /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc        */
  /**********************************************************************/

  ex = ginfo.expt;
  sptr = ex->sen_tbl_ptr + *(ex->index_sen_tbl + ExDa->sensor);
  num_data  = ExDa->num_sample;

  bytes = sizeof (ReaL_8) * ExDa->num_sample;
  if (bytes <= 0) { return (CAL_DATA_MALLOC); }
  if ((Scratch = malloc (bytes)) == NO_MEMORY) { return (CAL_DATA_MALLOC); }
  RetD = (ReaL_8 *) Scratch;

  if (bytes > Tmp_Bytes) {
     free(Tmp_Space);
     if ((Tmp_Space = malloc( bytes)) == NO_MEMORY) {
        free (Scratch);
        return (CAL_DATA_MALLOC);
     }
     Tmp_Bytes = bytes;
  }
  RetB = (ReaL_8 *) Tmp_Space;
  DataTo = RetD;

  /**********************************************************************/
  /* Transfer the integer raw values into the data array to be returned */
  /* to the user.                                                       */
  /**********************************************************************/

  ir_transfer_data (RetD, sen, num_data, *(ex->d_type + ExDa->sensor), 0L);

  /*********************************************************************/
  /*  If tables are to be applied, convert to the data accordingly.    */
  /*********************************************************************/

  OP = tbl_oper;
  TbL = tbls_to_apply;
  I = 0;
  while ( I++ < num_tbls) {

     /*******************************************************************/
     /*  Check for combination of accumulators since operations are NOT */
     /*  table specific.                                                */
     /*                                                                 */
     /*  Buffer Operations  BufOp = Operation / 1000                    */
     /*  Buffer 0 == DataOut                                            */
     /*  Buffers 1, 2, 3, 4, 5, 6 == Tmp_Space                          */
     /*                                                                 */
     /*        FROM            TO          BufOp                        */
     /*       ------          ----         -----                        */
     /*         V              0              0                         */
     /*         V              1              1                         */
     /*         V              2             20                         */
     /*         V              3             21                         */
     /*         V              4             22                         */
     /*         V              5             23                         */
     /*         V              6             24                         */
     /*                                                                 */
     /*         1              0              2                         */
     /*         0              1              3                         */
     /*         1              2              4                         */
     /*         0              3              5                         */
     /*         2              0              6                         */
     /*         2              1              7                         */
     /*         3              2              8                         */
     /*         1              3              9                         */
     /*         3              0             10                         */
     /*         3              1             11                         */
     /*         0              2             12                         */
     /*         2              3             13                         */
     /*         0              4             14                         */
     /*         0              5             15                         */
     /*         0              6             16                         */
     /*         1              4             17                         */
     /*         2              5             18                         */
     /*         3              6             19                         */
     /*******************************************************************/

     tV = *(sptr->tbl_var + *TbL);                  /* table variable        */
     PosOp = (*OP < 0 ) ? -(*OP): *OP;              /* positive operator     */
     BufOp = PosOp / 1000;                          /* buffer specific op    */
     ExOp = (PosOp % 1000) / 10;                    /* extended operation    */
     BasOp = (*OP % 10);                            /* basic operation       */
     if (BasOp < 0)                                 /* negative base oper?   */
        BasOp = 9 - BasOp;                          /* make it positive      */
     
     if (BufOp > 1 && BufOp < 20) {                 /* buf-buf operation     */
        R = BufOp - 2;                              /* reduced buffer OP     */
        A = R % 4;                                  /* seq ops 0-3,0-3,0-3   */
        C = R / 4;                                  /* oper block            */
        if (C < 3) {                                /* 2-way buf operations  */
           T = A;                                   /* finding TO buffer     */
           F = (A == 0) ? C + 1 : (A == 3) ? C :    /* finding FROM buffer   */
                        (C < 2) ? R/2 : 3*(A % 2);  /* offset                */
        } else {                                    /* END 2-WAY OPS         */
           T = (R % 3) + 4;                         /* finding TO buffer     */
           F = (R > 16) ? (R % 16) : 0;             /* finding FROM buffer   */
        }                                           /* END 1-WAY OPS         */

        F = (F - 1) * ex->swp_len;                  /* From byte offset      */
        T = (T - 1) * ex->swp_len;                  /* To byte offset        */

        DataTo = (T < 0) ? RetD : RetB + T;         /* set the DataTo buffer */
        DataFrom = (F < 0) ? RetD : RetB + F;       /* set DataFrom buffer   */

        CombineOp (DataTo, DataFrom, num_data, BasOp); /* do basic operation */

        /************************************************************/
        /*  Perform the higher order operations.                    */
        /************************************************************/

        if (ExOp > 0)
           ir_idf_ex_func (ExOp, DataTo, num_data, UDF);
        ++OP;
	++TbL;
        continue;
     }                                              /* END BUFFER OPERATION  */

     /********************************************************************/
     /*  A check is made upfront for ASCII tables and tables associated  */
     /*  with status mode information.                                   */
     /********************************************************************/

     if (*(ex->tbl_type + *TbL) == 1) {
        ++OP;
	++TbL;
        continue;
     } else if (*(ex->tbl_var + *TbL) == 4 || *(ex->tbl_var + *TbL) == 5) {
        ++OP;
	++TbL;
        continue;
     }

     /********************************************************************/
     /*  Is this table to be applied for the conversion of the data?     */
     /*  A size of 0 indicates that no table is defined (tbl_fmt < 0);   */
     /*  therefore, this table is bypassed for data conversion.          */
     /********************************************************************/

     if (*(sptr->tbl_size + *TbL) != 0) {
        fn_of_raw = (tV == 1 || tV >= 7) ? 0 : 1;
        T = ((BufOp % 18) - 1) * ex->swp_len;
        DataTo = (T < 0) ? RetD : RetB + T;

        /*****************************************************************/
        /* This table is a function of a calibration set.  Determine the */
        /* index value to get to the calibration values in question.     */
        /*****************************************************************/

        data_start = 0;
        if (tV < 0) {
           which_cal = abs (tV) - 1;

           s1 = ExDa->cset_num;
           s_end = s1 + which_cal;
           for ( ; s1 < s_end; )
              data_start += *s1++;

           /***************************************************************/
           /*  Determine the number of data elements each cal. value is   */
           /*  to be applied to.  A single cal. value is to be applied to */
           /*  each data element if CAL_USE = 0.                          */
           /***************************************************************/

           if (*(ex->CAL_USE + which_cal) == 0)
             hold_val = ExDa->num_sample;
           else
             hold_val = *(ex->CAL_USE + which_cal);
         }

        /***************************************************************/
        /* Start at the beginning of the data array and there is a one-*/
        /* to-one mapping between table elements and data elements.    */
        /***************************************************************/

        else
           hold_val = (tV != 7) ? 1 : ex->num_sample;

        /************************************************************/
        /*  Get the data type and input data array.                 */
        /************************************************************/

        switch (tV) 
        {
            case 0:                                /* RAW SENSOR DATA */
               DFrom = sen;
               dtype = *(ex->d_type + ExDa->sensor);
            break;
            case 2:                                /* RAW SCAN DATA   */
              DFrom = swp;
              dtype = 0;
            break;
            case 4:                                /* RAW MODE DATA   */
               DFrom = mode;
               dtype = 0;
            break;
            case 5:                                /* PROC MODE DATA  */
            case 6:                                /* RAW QUAL DATA   */
            break;
            case 1:                                /* PROCESSED DATA */
               DFrom = (ByTe_1 *)DataTo;
               dtype = 2;
            break;
            case 7:                                /* SENSOR NUMBER  */
            case 8:                                /* MATRIX COLUMN NUMBER  */
               SenNum[0] = (tV == 7) ? ExDa->sensor : ExDa->column;
               DFrom = (ByTe_1 *)SenNum;
               dtype = 2;
            break;
            default:                               /* RAW ANC DATA   */
              DFrom = cal;
              dtype = (*(ex->cal_wlen + which_cal) < 0) ? 1 : 0;
            break;
        }

        /****************************************************************/
        /*  Perform the operation defined for the table.                */
        /****************************************************************/

        if (*(ex->tbl_type + *TbL) == 2 || *(ex->tbl_type + *TbL) == 4)
          num_ele_in_tbl = *(sptr->tbl_size + *TbL) / ex->swp_len;
        else
          num_ele_in_tbl = *(sptr->tbl_size + *TbL);

        TblVals = *(sptr->tbl_ptr + *TbL);
        if (*(ex->tbl_type + *TbL) == 5) 
           TblVals += ExDa->column * ex->swp_len;

        IDF_OPER[BasOp]._math_drv (DataTo, DFrom, data_start, TblVals,  
                    num_ele_in_tbl, num_data, *(sptr->tbl_expand + *TbL), 
                    dtype, hold_val, fn_of_raw, *(ex->tbl_type + *TbL), 
                    UDF, ex->swp_len);

        /************************************************************/
        /*  Perform the higher order operations.                    */
        /************************************************************/

        if (ExOp > 0)
           ir_idf_ex_func (ExOp, DataTo, num_data, UDF);
     }

     ++OP;
     ++TbL;
   }

  /************************************************************************/
  /*  Check the data for FILL values.  If the data is found to be fill    */
  /*  data, the value returned is set to OUTSIDE_MIN.                     */
  /************************************************************************/

   dEnD = RetD + ExDa->num_sample;
   if (chk_fill) {
      l1 = ExDa->sen_data;
      for ( ; RetD < dEnD; ++RetD) {
         if (*l1++ == fill_value) { *RetD = OUTSIDE_MIN; }
      }
   }

  /************************************************************************/
  /* Transfer to output array                                             */
  /************************************************************************/

   RetD = DataTo;
   if (IFmt == 1) {
      d1 = (ReaL_8 *)DataOut;
      for ( ; RetD < dEnD; ) { *d1++ = *RetD++; }
   } else {
      f1 = DataOut;
      for ( ; RetD < dEnD; ) { *f1++ = *RetD++; }
   }

  free (Scratch); 
  return (ALL_OKAY);
}
