#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/*****************************************************************************
 *                                                                           *
 *                          IR_ASSIGN_POINTERS SUBROUTINE                    *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine is called to set up the pointers to the header and data   *
 *  record elements.  These pointers are set to point to elements that are   *
 *  single variables, in other words, within the header or data record       *
 *  structure, these elements do not come after a variable-length array. For *
 *  elements that do come after a variable-length array, the routine         *
 *  ir_fix_pointers is called to assign the pointers.                        *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    char header_ptrs             flag which indicates that the header      *
 *                                 pointers need to be reset                 *
 *    char data_ptrs               flag which indicates that the data        *
 *                                 pointers need to be reset                 *
 *    void *UDF                    ptr to memory location for structure that *
 *                                 holds returned data values (read_drec)    *
 *                                                                           *
 *  USAGE                                                                    *
 *    x = ir_assign_pointers (header_ptrs, data_ptrs, UDF)                   *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    sizeof ()                    the size of the specified object in bytes *
 *    ir_fix_pointers ()           assigns the pointers for those elements   *
 *                                 whose size may vary from header to header *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info ginfo    structure holding information concerning  *
 *                                 the experiment that is being processed    *
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct experiment_info *ex   a pointer to the structure that holds     *
 *                                 specific experiment information           *
 *    struct ptr_rec *ptr          a pointer to the structure which holds    *
 *                                 all pointers to the header and data for   *
 *                                 the experiment of interest                *
 *    long offset                  index into the memory allocated for the   *
 *                                 header and data information               *
 *    short rV                     the value returned by the called routine  *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 *****************************************************************************/

ByTe_2 ir_assign_pointers (ByTe_1 Header, ByTe_1 Data, void *UDF)
{
   extern struct general_info ginfo;

   register struct experiment_info *ex;
   register struct ptr_rec *ptr; 
   register ByTe_1 *Ct, *EnD;
   ByTe_4 offset;
   ByTe_2 rV;

  /*************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header   */
  /*  and data information for the experiment currently being processed.   */
  /*************************************************************************/

   ex = ginfo.expt;
   ptr = ex->info_ptr;

  /*************************************************************************/
  /*  Assign the pointers for the header information.  Year starts at an   */
  /*  offset since header length is first.                                 */
  /*************************************************************************/

   if (Header) {
      offset = sizeof(u_ByTe_2);
      ptr->YEAR = (ByTe_2 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_2);
      ptr->DAYOFYEAR = (ByTe_2 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_2);
      ptr->TIME_UNITS = (ByTe_1 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_1);
      ptr->I_MODE = (u_ByTe_1 *)(ex->HEADER_MEM + offset);
      offset += sizeof (u_ByTe_1);
      ptr->DATA_ACCUM = (ByTe_4 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_4);
      ptr->DATA_LAT = (ByTe_4 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_4);
      ptr->SWP_RESET = (ByTe_4 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_4);
      ptr->SEN_RESET = (ByTe_4 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_4);
      ptr->N_SEN = (ByTe_2 *)(ex->HEADER_MEM + offset);
      offset += sizeof (ByTe_2);
      ptr->N_SAMPLE = (u_ByTe_2 *)(ex->HEADER_MEM + offset);
      offset += sizeof (u_ByTe_2);

      if (ex->smp_id == 3) {
         ptr->N_COLS = (u_ByTe_2 *)(ex->HEADER_MEM + offset);
         offset += sizeof (u_ByTe_2);
         ptr->LEAD_COL = (u_ByTe_2 *)(ex->HEADER_MEM + offset);
         offset += sizeof (u_ByTe_2);
         ptr->TOT_COLS = (u_ByTe_2 *)(ex->HEADER_MEM + offset);
         offset += sizeof (u_ByTe_2);
      }

      ptr->SAMP_INDEX = (ByTe_2 *)(ex->HEADER_MEM + offset);
   }
   
  /*************************************************************************/
  /*  Assign the pointers for the data information.                        */
  /*************************************************************************/

   if (Data) {
      ptr->TIME = (ByTe_4 *) (ex->DATA_MEM);
      offset = sizeof (ByTe_4);
      ptr->SPIN = (ByTe_4 *) (ex->DATA_MEM + offset);
      offset += sizeof (ByTe_4);
      ptr->SUN_SEN = (ByTe_4 *) (ex->DATA_MEM + offset);
      offset += sizeof (ByTe_4);
      ptr->HDR_OFF = (ByTe_4 *) (ex->DATA_MEM + offset);
   
  /*************************************************************************/
  /*  HDR_OFF is an array of longs whose dimension is max_nss.             */
  /*************************************************************************/
   
      offset += sizeof (ByTe_4) * ex->max_nss;
      ptr->NSS = (ByTe_4 *) (ex->DATA_MEM + offset);
      offset += sizeof (ByTe_4);

  /*************************************************************************/
  /*  PREFACE CALS if any come before the DATA.                            */
  /*************************************************************************/
   
      EnD = ex->cal_target + ex->cal_sets;
      for (Ct = ex->cal_target, ptr->NumPreAnc = 0; Ct < EnD; ++Ct) {
         if (*Ct == 3) { ptr->NumPreAnc += 1; }
      }

      if (ptr->NumPreAnc > 0) {
         ptr->CAL_PREFACE = (u_ByTe_1 *) (ex->DATA_MEM + offset);
         offset += ex->AncPBytes * ptr->NumPreAnc;
         ptr->DATA = (u_ByTe_1 *) (ex->DATA_MEM + offset);
      } else { ptr->DATA = (u_ByTe_1 *) (ex->DATA_MEM + offset); }
   }

  /*************************************************************************/
  /*  Set pointers that depend upon information read from a header record. */
  /*************************************************************************/

   if (Header) {
      rV = ir_fix_pointers (UDF);
      if (rV != ALL_OKAY) { return (rV); }
   }

   return (ALL_OKAY);
}
