#include <stdio.h>
#include "util_str.h"
#include "libbase_udf.h"
#include "user_defs.h"
#include "math.h"

/****************************************************************************
 *                                                                          *
 *                             ThetaConing                                  *
 *                                                                          *
 *  DESCRIPTION                                                             *
 *    This routine fills in the theta angles for an instrument which is     *
 *  coning about the satellite spin vector.                                 * 
 *                                                                          *
 ***************************************************************************/

void ThetaConing (void *UDF)
{
   extern struct general_info ginfo;

   struct idf_data *ExDa;
   struct experiment_info *ex;
   register ReaL_4 *f1, *f2, *f3, *f4, *fEnD;

   ReaL_4 cP, sP, cT, sT, cW, sW, cS, sS;
   ReaL_4 Phi, Theta, Omega;
   ReaL_4 A11, A12, A13;
   ReaL_4 A31, A32, A33;
   ReaL_4 B11, B12, B13;
   ReaL_4 B21, B22, B23;
   ReaL_4 B31, B32, B33;
   ReaL_4 T1, T2;
   ReaL_4 X, Y, Z;
   ReaL_4 X1, Y1, Z1;
   ReaL_4 X2, Y2, Z2;
   ReaL_4 Tmp, SenAng;
   ByTe_4 ProcFlag;
 
   /***********************************************************************/
   /* Set a pointer to the structure which holds all pointers for header  */
   /* and data information for thxperiment currently being processed.     */
   /***********************************************************************/

   ex  = ginfo.expt;
   ExDa = (struct idf_data *) UDF;

   /***********************************************************************/
   /* IF not contants then do a generic theta load and leave              */
   /***********************************************************************/

   if ( ex->num_consts <= 0 ) {
      f3 = ExDa->start_theta;
      f4 = ExDa->stop_theta;
      fEnD = f3 + ExDa->num_angle;
      for ( ; f3 < fEnD; ) {
         *f3++ = -500.0; 
         *f4++ = -500.0; 
      }
      return;
   }
   
   /***********************************************************************/
   /* Do the nominal theta load                                           */
   /***********************************************************************/

   T1 =  *(ex->constants + (2 * ex->num_sensor) + ExDa->sensor);
   T2 =  *(ex->constants + (3 * ex->num_sensor) + ExDa->sensor);
   f3 = ExDa->start_theta;
   f4 = ExDa->stop_theta;
   fEnD = f3 + ExDa->num_angle;
   for ( ; f3 < fEnD; )
   {
      *f3++ = T1; 
      *f4++ = T2; 
   }
   
   /***********************************************************************/
   /* No Coning if Theta is < -490 (no constant values)                   */
   /***********************************************************************/

   Theta = *(ex->constants + (6 * ex->num_sensor) + ExDa->sensor);
   if (Theta < -400.0)
      return;

   /***********************************************************************/
   /* Use constant id 29 for sensor azimuthal offset if available,        */
   /* otherwise default to the azimuthal offset used for returning az     */
   /***********************************************************************/

   Phi = *(ex->constants + (5 * ex->num_sensor) + ExDa->sensor);
   SenAng = *(ex->constants + (4 * ex->num_sensor) + ExDa->sensor);
   if (SenAng < -490.0)
      SenAng = *(ex->constants + ExDa->sensor);

   ProcFlag = (Theta > 0.0) ? 1 : 0;
  
   if (Phi < 0.0)
      Phi = 0.0;

   f1 = ExDa->start_az;
   f2 = ExDa->stop_az;
   f3 = ExDa->start_theta;
   f4 = ExDa->stop_theta;
   fEnD = f3 + ExDa->num_angle;
   if (ProcFlag == 0)
   {
      Phi = Phi - 270.0;
      Phi *= TORAD;
      Theta *= -TORAD;

      cS = cos (SenAng * TORAD);
      sS = sin (SenAng * TORAD);
      cP = cos (Phi);
      sP = sin (Phi);
      cT = cos (Theta);
      sT = sin (Theta);
      B11 = cP;
      B12 = sP;
      B13 = 0.0;
      B21 = -cT * sP;
      B22 =  cT * cP;
      B23 = sT;
      B31 = sT * sP;
      B32 = -sT * cP;
      B33 = cT;

      for ( ; f3 < fEnD; )
      {
         Omega = (*f2++ + *f1) / 2.0;
         Tmp = Omega - *f1++;
         if (Tmp > 60.0 || Tmp < -60.0)
            Omega += 180.0;

         Tmp = (Omega - SenAng) * TORAD;
         cW = cos (Tmp);
         sW = sin (Tmp);

         A11 = cW * cP - cT * sP * sW;
         A12 = -sW * cP - cT * sP * cW;
         A13 = sT * sP;
         A31 = sT * sW;
         A32 = sT * cW;
         A33 = cT;

         X = sin (*f3 * TORAD);
         Y = 0.0;
         Z = cos (*f3 * TORAD);
         X1 = A11 * X + A31 * Z;
         Y1 = A12 * X + A32 * Z;
         Z1 = A13 * X + A33 * Z;
         X2 = B11 * X1 + B21 * Y1 + B31 * Z1;
         Y2 = B12 * X1 + B22 * Y1 + B32 * Z1;
         Z2 = B13 * X1 + B23 * Y1 + B33 * Z1;

         *f3++ = atan2(sqrt (X2 * X2 + Y2 * Y2), Z2) / TORAD;

         X = sin (*f4 * TORAD);
         Y = 0.0;
         Z = cos (*f4 * TORAD);
         X1 = A11 * X + A31 * Z;
         Y1 = A12 * X + A32 * Z;
         Z1 = A13 * X + A33 * Z;
         X2 = B11 * X1 + B21 * Y1 + B31 * Z1;
         Y2 = B12 * X1 + B22 * Y1 + B32 * Z1;
         Z2 = B13 * X1 + B23 * Y1 + B33 * Z1;

         *f4++ = atan2(sqrt (X2 * X2 + Y2 * Y2), Z2) / TORAD;
      }
   }
   else
   {
      SenAng *= TORAD;
      sS = Theta * sin (SenAng);
      for ( ; f3 < fEnD; )
      {
          *f3++ += sS; 
          *f4++ += sS; 
      }
   }
}
