#include <math.h>
#include "util_str.h"
#include "user_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                             PAFORMAT0                                      *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine computes the pitch angle for each time period associated   *
 *  with each element of the sweep.  The dot product of the unit normal to    *
 *  the detector aperture with the local magnetic field is computed.  The     *
 *  value computed is in  radians and is is converted to degrees before it    *
 *  is returned.                                                              *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    struct pitch_info *pA      a pointer to the structure that holds pitch  *
 *                               angle information                            *
 *                               are returned                                 *
 *    void *UDF                  pointer to the data structure                *
 *    short SeN                  the sensor for which data is requested       *
 *    unsigned short nElem       the number of elements in the sweep          *
 *                                                                            *
 *  USAGE                                                                     *
 *    PaFormat0 (pA, pData, SeN, nElem)                                       *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sqrt ()                     returns the square root of the argument     *
 *    acos ()                     returns arc cosine of input                 *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    None                                                                    *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    reg unsigned short Step     the sweep step being processed              *
 *    register double *nVec       ptr to the unit normal vector               *
 *    register double *mVec       ptr to the magnetic field vector            *
 *    register double *mEnd       loop termination variable                   *
 *    register ReaL_4 *pD         ptr to pitch angle data                     *
 *    register ReaL_4 *pDEnd      ptr to end of pitch angle data              *
 *    double Dot                  result of calculation for the numerator     *
 *    double Mag                  summation of the magnetic field elements    *
 *    double DotProd              computed dot product value                  *
 *    double rData                computed pitch angle value                  *
 *    int offset                  offset into allocated memory                *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void PaFormat0 (struct pitch_info *pA, void *UDF, ByTe_2 SeN, 
                u_ByTe_2 nElem)
{
   struct idf_data *ExDa;
   register ReaL_8 *nVec, *mVec, *mEnd;
   register ReaL_4 *pD, *pDEnd;
   ReaL_8 Dot, Mag, DotProd, rData;

   /*********************************************************************/
   /*  Process each element of the sweep.                               */
   /*********************************************************************/

   ExDa = (struct idf_data *) UDF;
   
   pD = ExDa->pitch_angles;
   pDEnd = pD + nElem;
   mVec = pA->data_val;
   for ( ; pD < pDEnd; ) {

      /*********************************************************************/
      /*  Do a summation from 1 to N for the numerator and denominator.    */
      /*  Since the unit normal are constants, the magnitude was computed  */
      /*  when the three components were retrieved.                        */
      /*********************************************************************/
      
      mEnd = mVec + 3;
      Mag = Dot = 0.0;
      nVec = pA->unit_normals + (SeN * 3);
      for ( ; mVec < mEnd; ++mVec) {
         Dot += *nVec++ * *mVec;
         Mag += *mVec * *mVec;
      }
      
      /*********************************************************************/
      /*  Solve for the denominator and compute the dot product.           */
      /*********************************************************************/
      
      Mag = sqrt (Mag);
      DotProd = (-1.0 * Dot) / Mag / pA->mag_normal[SeN];
      rData = acos (DotProd);
      *pD++ = (ReaL_4) (rData / TORAD);
   }
}
