#include "util_str.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                             IR_SENSOR_SET_TIME SUBROUTINE                  *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine returns the start and end time of the current sensor set.  *
 *  The time is returned in two parts, the time in milliseconds and for       *
 *  finer resolution, the number of nanoseconds left over after the time in   *
 *  milliseconds has been determined.  The algorithm used to determine the    *
 *  time is taken directly from the VIDF definition document.                 *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    long *btime_ms             start time (msec) of the sensor set          *
 *    long *btime_ns             start time (nanoseconds) of the sensor set   *
 *    long *etime_ms             end time (msec) of the sensor set            *
 *    long *etime_ns             end time (nanoseconds) of the sensor set     *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_sensor_set_time (&btime_ms, &btime_ns, &etime_ms, &etime_ns)         *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    None                                                                    *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    unsigned short n_sample    number of samples returned for each sensor   *
 *    ByTe_4 TotCol              number of data columns in sensor set         *
 *    ByTe_4 SsT                 equivalent sensor set reset time component   *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void ir_sensor_set_time (ByTe_4 *btime_ms, ByTe_4 *btime_ns, ByTe_4 *etime_ms, 
                         ByTe_4 *etime_ns)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_4 SsT, TotCol;
  u_ByTe_2 n_sample;

  /****************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header and  */
  /*  data information for the experiment currently being processed.          */
  /****************************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  n_sample = *ptr->N_SAMPLE - 1;
  TotCol = (ex->smp_id != 3) ? *ptr->N_SEN - 1 : 
                               (*ptr->N_SEN - 1) * *(ptr->N_COLS);

  /***************************************************************************/
  /*  Determine the beginning time of the current sensor set.  PTR->TIME is  */
  /*  not added to the nanoseconds element since it is returned in msec.     */
  /***************************************************************************/

  *btime_ms = *ptr->TIME + ex->accum_ss_ms; 
  *btime_ns = ex->accum_ss_ns;

  /***************************************************************************/
  /*  If the nanosecond time element can be converted into milliseconds, do  */
  /*  so and add this to the millisecond time element and keep the leftover  */
  /*  nanoseconds.  1000000 nanoseconds = 1 millisecond.                     */ 
  /***************************************************************************/

  *btime_ms += *btime_ns / 1000000;
  *btime_ns = *btime_ns % 1000000;

  /***************************************************************************/
  /*  Determine the ending time for the current sensor set.                  */
  /*                                                                         */
  /*                    CHANGE - 8/19/98 (CAG)                               */
  /*  You can essentially use a negative sensor reset time to keep time from */
  /*  from advancing from sensor set to sensor set.  A negative sensor set   */
  /*  reset time should not be used here though but just the total accum.    */
  /***************************************************************************/

  SsT = (ex->sen_ms < 0) ? 0 : ex->sen_ms;
  *etime_ms = *btime_ms + TotCol * ex->time_row_ms + 
              n_sample * ex->time_col_ms + ex->time_sm_ms + ex->swp_ms + SsT;

  SsT = (ex->sen_ns < 0) ? 0 : ex->sen_ns;
  *etime_ns = *btime_ns + TotCol * ex->time_row_ns + 
               n_sample * ex->time_col_ns + ex->time_sm_ms + ex->swp_ns + SsT;

  /***************************************************************************/
  /*  If the nanosecond time element can be converted into milliseconds, do  */
  /*  so and add this to the millisecond time element and keep the leftover  */
  /*  nanoseconds.  1000000 nanoseconds = 1 millisecond.                     */ 
  /***************************************************************************/

  *etime_ms += *etime_ns / 1000000;
  *etime_ns = *etime_ns % 1000000;
}
