#include <stdlib.h>
#include "util_str.h"
#include "libbase_udf.h"
#include "OpSySInD.h"

/******************************************************************************
 *                                                                            *
 *                             IR_RESET_SEN_SET SUBROUTINE                    *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to adjust the cumulative sensor set time values. *
 *  Time was advanced and in doing so, a new sensor set was encountered.      *
 *  These cumulative sensor set values are used in the equation defined in    *
 *  the VIDF document for computing the time of a given sample.               *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    None                                                                    *
 *                                                                            *
 *  USAGE                                                                     *
 *    ir_reset_sen_set_time ()                                                *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    abs()                      returns the absolute value of a number       *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info        structure that holds information concerning  *
 *        ginfo                  the experiment that is being processed       *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info     a pointer to the structure that holds        *
 *          *ex                  specific experiment information              *
 *    struct ptr_rec *ptr        a pointer to the structure which holds all   *
 *                               pointers to the header and data for the      *
 *                               experiment of interest                       *
 *    long num_bits              number of bits to bypass the sensor set(s)   *
 *    long num_bytes             number of bytes to bypass the sensor set(s)  *
 *    unsigned short n_sample    number of samples returned for each sensor   *
 *    unsigned short set_sample  number of samples to use to determine time   *
 *    short TotCol               number data columns in current sen set       *
 *    short base_swp_off         sweep step offset of 1st element in column   *
 *    short swp_off              sweep step offset for last element in column *
 *    short SBits                how many bits are used per data element      *
 *                               within each byte (necessary due to padding)  *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

void ir_reset_sen_set_time (void)
{
  extern struct general_info ginfo;

  struct experiment_info *ex;
  struct ptr_rec *ptr;
  ByTe_4 num_bits, num_bytes, TotCol;
  u_ByTe_2 n_sample, set_sample;
  ByTe_2 base_swp_off, swp_off, SBits;

  /****************************************************************************/
  /*  Set a pointer to the structure which holds all pointers for header and  */
  /*  data information for the experiment currently being processed and point */
  /*  to the sensor structure for the sensor of interest.                     */
  /****************************************************************************/

  ex  = ginfo.expt;
  ptr = ex->info_ptr;
  n_sample = *ptr->N_SAMPLE;
  TotCol = (ex->smp_id != 3) ? *ptr->N_SEN : *ptr->N_SEN * *ptr->N_COLS;

  /***********************************************************/
  /*  Instrument could be in a mode where a single step is   */
  /*  being returned.                                        */
  /***********************************************************/

  if (n_sample == 1)
     set_sample = 1;
  else if (ex->smp_id == 2)
  {
    set_sample = n_sample;
  }
  else
   {
     if (ex->da_method == 3)
      {

        /*************************************************************/
        /*  Determine how many steps are missing and then multiply   */
        /*  by the number of samples returned to get the length of   */
        /*  the sweep.  The assumption is that the number of missing */
        /*  steps is the same from one scan_index value to the next, */
        /*  so just use 0 and 1.                                     */
        /*************************************************************/

        base_swp_off = *ptr->SAMP_INDEX;
        swp_off = *(ptr->SAMP_INDEX + 1);
        set_sample = abs (swp_off - base_swp_off) * n_sample;
      }
     else
      {
        base_swp_off = *ptr->SAMP_INDEX;
        swp_off = *(ptr->SAMP_INDEX + (n_sample - 1));
        set_sample = (ex->da_method == 0) ? n_sample : (ex->da_method == 1) ?
                      ex->swp_len : abs (swp_off - base_swp_off);
      }
   }

  /*************************************************************************/
  /*  Determine the time between 2 successive sensor sets of data and add  */
  /*  this value to the cumulative sensor set time values.                 */
  /*************************************************************************/

  ex->accum_ss_ms += ex->time_row_ms * (TotCol - 1) + 
                     ex->time_col_ms * (set_sample - 1) + ex->swp_ms + 
                     ex->sen_ms + ex->time_sm_ms;
  ex->accum_ss_ns += ex->time_row_ns * (TotCol - 1) + 
                     ex->time_col_ns * (set_sample - 1) + ex->swp_ns + 
                     ex->sen_ns + ex->time_sm_ns;

  /****************************************************************************/
  /*  Byte_size indicates how many bits are used per data element within each */
  /*  byte.  For 5, 6 and 7 bit data, there are pad bits so that there is     */
  /*  only one sample per byte (takes 8 bits).  For 3 bit data, use 4 since   */
  /*  the data is laid down as a 4-bit value with the most significant bit    */
  /*  set to zero.  The rest of the cases divide evenly into the byte.        */
  /****************************************************************************/

  if (ex->tdw_len < 8)
     SBits = (ex->tdw_len >= 5) ? 8 : (ex->tdw_len == 3) ? 4 : ex->tdw_len;
  else
     SBits = (ex->tdw_len == 8) ? 8 :
                (ex->tdw_len > 8 && ex->tdw_len <= 16) ? 16 : 32;

  /**************************************************************************/
  /*  The no. of bits needed for the old sensor set just changed.  Since    */
  /*  the next sensor set starts at the next byte, make sure number of bits */
  /*  gets adjusted to next byte.                                           */
  /**************************************************************************/

  num_bits = n_sample * TotCol * SBits;
  num_bits += ptr->NumSenAnc * ex->AncSBits;
  num_bits += ex->AncGBytes * ptr->NumGenAnc * 8;

  num_bytes = num_bits / 8;
  if (num_bits % 8 != 0) 
    ++num_bytes;
  num_bits = num_bytes * 8;
  
  ex->accum_ss_sz += num_bits;
}
