#include <stdio.h>
#include <string.h>
#include "idf_defs.h"
#include "local_defs.h"
#include "util_str.h"
#include "ret_codes.h"
#include "libbase_udf.h"
#include "OpSySInD.h"


/*******************************************************************************
 *                                                                             *
 *                        IR_READ_IDF_MEMORY SUBROUTINE                        *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to retrieve information from the VIDF definition  *
 *  file, which is held in memory, for the data set specified.                 *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    char *var                   output value(s) associated with selected     *
 *                                field                                        *
 *    short quan                  specified field in VIDF file                 *
 *    short num                   the table or constant definition from        *
 *                                which the required field is to be retrieved  *
 *    short len                   if the selected field is an array, this is   *
 *                                the number of elements to be returned        *
 *    long start                  if the selected field is an array, this is   *
 *                                the starting position in the array           *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_read_idf_memory (&var, quan, num, start, len)                     *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    memcpy ()                   copies bytes from one memory area to another *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info         structure that holds information concerning  *
 *         ginfo                  the experiment that is being processed       *
 *    short size[]                the size of each field (1byte, 2bytes, etc)  *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info      a pointer to the structure that holds        *
 *            *ex                 specific experiment information              *
 *    register long *l1           pointer to offset values                     *
 *    long bytes                  the number of bytes to access for the field  *
 *    long avail_bytes            the number of bytes that can be accessed     *
 *                                for the field in question                    *
 *    long upper                  used to determine the number of bytes to     *
 *                                process                                      *
 *    long to_data                offset into the VIDF data array              *
 *    long Telem                  offset to get to table elements              *
 *    short lsize                 the size of the element being requested      *
 *    char *data                  pointer to the VIDF data held in memory      *
 *    char type                   flag indicating if the field is identified   *
 *                                with a table, a constant or neither          *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_4 ir_read_idf_memory (ByTe_1 *var, ByTe_2 quan, ByTe_2 num, ByTe_4 start, 
                           ByTe_2 len)
{
   extern struct general_info ginfo;
   extern ByTe_2 SiZe[];

   struct experiment_info *ex;
   register ByTe_4 *l1;
   ByTe_4 bytes, avail_bytes, upper, to_data, Telem;
   ByTe_4 qN, qS;
   ByTe_2 lsize;
   ByTe_1 *data, type;

   /**************************************************************************/
   /*  The ex structure of interest has been found by READ_IDF already.      */
   /**************************************************************************/

   ex = ginfo.expt;
   Telem = _ConstID - _TblScaSZ;
   type = (quan < _TblScaSZ) ? 0 : (quan < _ConstID) ? 1 : 2;
   qS = (quan == _PaBxbybZ) ? 2 : SiZe[quan];

   switch (type)
   {
      case 0:
         upper = ex->VIDF_INFO.offsets[quan+1];
         avail_bytes = upper - ex->VIDF_INFO.offsets[quan];
         if (len > 0)
         {
            bytes = SiZe[quan] * len;

            if (avail_bytes < bytes)
               return (IDF_MANY_BYTES);
         }
         else
         {
            bytes = avail_bytes;
            if (bytes == 0)
               return (IDF_NO_ENTRY);
         }
         to_data = ex->VIDF_INFO.offsets[quan] + SiZe[quan] * start;
         data = (ByTe_1 *) ex->VIDF_INFO.vidf_data + to_data;
         memcpy (var, data, bytes);
      break;
      case 1:
         if (num > ex->VIDF_INFO.TABLES - 1)
            return (IDF_TBL_NUM);

         if (quan == _TbL)
         {
            l1 = (ByTe_4 *)ex->VIDF_INFO.tbl_offs + 
                                       Telem * num + _TblTypE - _TblScaSZ;
            data = (ByTe_1 *) ex->VIDF_INFO.vidf_data + *l1;
            lsize = (*data == 1) ? 21 : SiZe[quan];
         }
         else
            lsize = SiZe[quan];

         l1 = (ByTe_4 *)ex->VIDF_INFO.tbl_offs + 
                                        Telem * num + quan - _TblScaSZ;
         upper = *(l1 + 1);
         avail_bytes = upper - *l1;
         if (len > 0)
         {
            bytes = lsize * len;
            if (avail_bytes < bytes)
               return (IDF_MANY_BYTES);
         }
         else
         {
            bytes = avail_bytes;
            if (bytes == 0)
               return (IDF_NO_ENTRY);
         }

         to_data = *l1 + lsize * start;
         data = (ByTe_1 *) ex->VIDF_INFO.vidf_data + to_data;
         memcpy (var, data, bytes);
      break;
      case 2:
         if (num > ex->VIDF_INFO.CONSTS - 1)
            return (IDF_CON_NUM);

         l1 = (ByTe_4 *)ex->VIDF_INFO.const_offs + 5 * num + quan - _ConstID;
         upper = *(l1 + 1);
         avail_bytes = upper - *l1;
         if (len > 0)
         {
            bytes = SiZe[quan] * len;
            if (avail_bytes < bytes)
               return (IDF_MANY_BYTES);
         }
         else
         {
            bytes = avail_bytes;
            if (bytes == 0)
               return (IDF_NO_ENTRY);
         }

         to_data = *l1 + SiZe[quan] * start;
         data = (ByTe_1 *) ex->VIDF_INFO.vidf_data + to_data;
         memcpy (var, data, bytes);
      break;
   }

   if (ex->BswaP && (qS ==2 || qS == 4) ) {
       qN = bytes / qS;
       ReOrderBytes (qS, (u_ByTe_1 *)var, qN); 
   }

   return (ALL_OKAY);
}
