#include "util_str.h"
#include "ret_codes.h"
#include "user_defs.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                        IR_PROCESS_PITCH_DATA SUBROUTINE                     *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to retrieve the data from the idfs data set that  *
 *  is used to compute pitch angles.  The data is retrieved one component at   *
 *  a time.  The percentage of the data sample to be included in the current   *
 *  delta-t time period is determined and the data value is multiplied by that *
 *  percentage before being added to the memory location for the component     *
 *  being processed.  The percentage or frac value is also saved so that the   *
 *  data value can be normalized before it is used in the calculation of the   *
 *  pitch angles.  It has been dictated that the IDFS source must be a scalar  *
 *  instrument, so only one data value is retrieved.                           *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    struct pitch_info *pa_ptr    a pointer to the structure that holds pitch *
 *                                 angle information                           *
 *     struct pitch_times          structure that holds the time period for    *
 *         src_time                the current delta-t being processed         *
 *    short comp                   index specifying which of the 3 components  *
 *    unsigned short swp_step      current step of the sweep being processed   *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_process_pitch_data (&pa_ptr, src_time, comp, swp_step)            *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    convert_to_units()           converts the specified data type into the   *
 *                                 data format requested                       *
 *    ir_pitch_time_frac ()        determines the percentage of the data sample*
 *                                 to be used for the delta-t being processed  *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info ginfo    structure that holds information concerning *
 *                                 the experiment that is being processed      *
 *    float conv_fill[]            temporary array that holds the values       *
 *                                 returned by convert_to_units()              *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info *ex   a pointer to the structure that holds       *
 *                                 specific experiment information             *
 *    struct idf_data *PA_DATA     structure holding the data for the pitch    *
 *                                 angle component being processed             *
 *    double *dptr                 ptr to the memory that holds the data value *
 *                                 to be used for the current component        *
 *    float time_frac              the percentage of the data to be included   *
 *                                 for the delta-t being processed             *
 *    float *fptr                  ptr to the normalization factor (frac) for  *
 *                                 the data for the current component          *
 *    short ret_val                holds value returned by called module       *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_process_pitch_data (struct pitch_info *pa_ptr, 
                              struct pitch_times src_time, 
                              ByTe_2 comp, u_ByTe_2 swp_step)
{
   extern struct general_info ginfo;
   extern ReaL_4 conv_fill[];

   struct experiment_info *ex;
   struct idf_data *PA_DATA;
   ReaL_8 *dptr;
   ReaL_4 time_frac = 0.0, *fptr;
   ByTe_2 ret_val;

   /**********************************************************************/
   /*  Convert data into units defined for the pitch components.  The    */
   /*  experiment_info structure should be the pitch angle IDFS source.  */
   /*  Data is from scalar source so only one value.                     */
   /**********************************************************************/
 
   ex = ginfo.expt;
   PA_DATA = (struct idf_data *) pa_ptr->idf_data_ptr[comp];
   if (PA_DATA->filled_data)
    {
      ret_val = convert_to_units (pa_ptr->data_key, pa_ptr->exten,
                                  pa_ptr->version, pa_ptr->idf_data_ptr[comp],
                                  SENSOR, 0,
                                  pa_ptr->num_tbls, pa_ptr->tbl_apply,
                                  pa_ptr->tbl_oper, conv_fill, ex->fill_flag,
                                  ex->fill_value);
      if (ret_val != ALL_OKAY)
        return (ret_val);
    }
   else
     conv_fill[0] = OUTSIDE_MIN;

   /*************************************************************************/
   /*  Determine the percentage of the data value to be included in the     */
   /*  current delta-t time period.  The frac value should normally be      */
   /*  between 0.0 and 1.0; however, if a gap was encountered on the data   */
   /*  set that called the pitch angle calculation, the frac for the pitch  */
   /*  angle components could be negative, indicating that the sample is    */
   /*  somewhere before the gap or in the gap region.  A negative frac      */
   /*  could also happen if a gap is encountered for a non-time controlling */
   /*  source and the start of the data after the gap is beyond the end     */
   /*  time of the delta-t period.  For both cases, the data is not         */
   /*  included in the current iteration.                                   */
   /*************************************************************************/

   time_frac = ir_pitch_time_frac (pa_ptr, src_time, comp);
   if (time_frac > 1.0)
     return (PA_BAD_FRAC);
 
   if (time_frac < 0.0)
     return (ALL_OKAY);

   /*****************************************************************/
   /* The assumption is that the frac value can never be negative.  */
   /* It is only set to a negative value as an initialization value.*/
   /*****************************************************************/

   dptr = pa_ptr->data_val + (3 * swp_step) + comp;
   fptr = pa_ptr->time_frac + (3 * swp_step) + comp;
   if (*fptr < 0.0)
    {
      *dptr = conv_fill[0] * time_frac;
      *fptr = time_frac;
    }
   else
    {
      *dptr += conv_fill[0] * time_frac;
      *fptr += time_frac;
    }

   return (ALL_OKAY);
}
