#include "util_str.h"
#include "idf_defs.h"
#include "ret_codes.h"
#include "libbase_udf.h"

/*******************************************************************************
 *                                                                             *
 *                       IR_PITCH_ANGLE_DATA_SRC SUBROUTINE                    *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to retrieve information from the VIDF file that   *
 *  specifies the IDFS data source to use to compute the pitch angles.         *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    None                                                                     *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_pitch_angle_data_src ()                                           *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    read_idf()                   reads information from the VIDF file        *
 *    get_data_key ()              returns the data key for the IDFS source    *
 *    ir_get_pinfo_rval ()         returns error code unique to GET_PITCH_INFO *
 *                                 for each possible READ_IDF() error code     *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info ginfo    structure that holds information concerning *
 *                                 the experiment that is being processed      *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info *ex   a pointer to the structure that holds       *
 *                                 specific experiment information             *
 *    register struct pitch_info   a pointer to the structure that holds pitch *
 *         *pa_ptr                 angle information                           *
 *    register char *str_ptr       pointer to the string being parsed          *
 *    long ret_val                 holds value returned by called module       *
 *    short rval                   holds value returned by called module       *
 *    char proj[]                  name of the project for pitch angle source  *
 *    char mission[]               name of the mission for pitch angle source  *
 *    char exper[]                 name of experiment for pitch angle source   *
 *    char inst[]                  name of instrument for pitch angle source   *
 *    char vinst[]                 name of virtual inst for pitch angle source *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_pitch_angle_data_src (void)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   register struct pitch_info *pa_ptr;
   register ByTe_1 *str_ptr;
   ByTe_4 ret_val;
   ByTe_2 rval;
   ByTe_1 proj[30], mission[30], exper[30], inst[30], vinst[30];

   /**************************************************************************/
   /*  Retrieve pertinent information from the VIDF file for pitch angle     */
   /*  data source.                                                          */
   /**************************************************************************/

   ex = ginfo.expt;
   pa_ptr = ex->pitch_angles;

   ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) proj, _PaProjecT, 0, 0L, 1);
   if (ret_val < 0)
     return (ir_get_pinfo_rval (ret_val));

   ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) mission, _PaMissioN, 0, 0L, 1);
   if (ret_val < 0)
     return (ir_get_pinfo_rval (ret_val));

   ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) exper, _PaExpeR, 0, 0L, 1);
   if (ret_val < 0)
     return (ir_get_pinfo_rval (ret_val));

   ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) inst, _PaInsT, 0, 0L, 1);
   if (ret_val < 0)
     return (ir_get_pinfo_rval (ret_val));

   ret_val = read_idf (ex->data_key, ex->exten, ex->version,
                       (ByTe_1 *) vinst, _PaVinsT, 0, 0L, 1);
   if (ret_val < 0)
     return (ir_get_pinfo_rval (ret_val));

   /**************************************************************************/
   /*  Make sure string is null terminated for comparison reasons.           */
   /**************************************************************************/

   str_ptr = proj;
   while (*str_ptr != ' ' && *str_ptr != 0)
     ++str_ptr;
   if (*str_ptr == ' ')
     *str_ptr = '\0';

   str_ptr = mission;
   while (*str_ptr != ' ' && *str_ptr != 0)
     ++str_ptr;
   if (*str_ptr == ' ')
     *str_ptr = '\0';

   str_ptr = exper;
   while (*str_ptr != ' ' && *str_ptr != 0)
     ++str_ptr;
   if (*str_ptr == ' ')
     *str_ptr = '\0';

   str_ptr = inst;
   while (*str_ptr != ' ' && *str_ptr != 0)
     ++str_ptr;
   if (*str_ptr == ' ')
     *str_ptr = '\0';

   str_ptr = vinst;
   while (*str_ptr != ' ' && *str_ptr != 0)
     ++str_ptr;
   if (*str_ptr == ' ')
     *str_ptr = '\0';

   rval = get_data_key (proj, mission, exper, inst, vinst, &pa_ptr->data_key);
   if (rval != ALL_OKAY)
     return (rval);

   return (ALL_OKAY);
}
