#include <stdio.h>
#include <stdlib.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                           IR_GET_TBL_INFO  SUBROUTINE                      *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called in order to retrieve the table information for   *
 *  all integer tables.  Ascii tables are not considered in this module; the  *
 *  ascii information is kept in a different structure if it is needed.       *
 *  Since the number of tables vary from virtual instrument to virtual        *
 *  instrument, the space to hold the table information must be allocated     *
 *  as the information becomes known.                                         *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    short sen                the sensor being processed                     *
 *    short sIndx              index into the SPTR structure to point to the  *
 *                             correct sensor combination structure           *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_get_tbl_info (sen, sIndx)                                        *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                the size of the specified object in bytes      *
 *    free ()                  frees allocated memory                         *
 *    malloc()                 allocates memory                               *
 *    read_idf()               reads information from the IDF file            *
 *    ir_tbl_size ()           determines the total number of bytes needed to *
 *                             hold all integer table values for all tables   *
 *    ir_get_new_tbl()         reads the requested table for the sensor from  *
 *                             the VIDF file                                  *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info      structure that holds information concerning    *
 *        ginfo                the experiment that is being processed         *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info   a pointer to the structure that holds          *
 *          *ex                specific experiment information                *
 *    struct sensor_tables     a pointer to the structure which holds the     *
 *         *sptr               various information for the tables utilized    *
 *                             by sensor-table combination being processed    *
 *    register short i         looping variable                               *
 *    long rV                  the value returned by the called routine       *
 *    long oFF                 indexing variable                              *
 *    long half_tbl            offset used to position table[0] location at   *
 *                             the middle of the table (for signed int)       *
 *    unsigned int B           the number of bytes to allocate                *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_get_tbl_info (ByTe_2 sen, ByTe_2 sIndx)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   struct sensor_tables *sptr;
   register ByTe_2 i;
   register ByTe_1 *c1, *c2;
   ByTe_4 rV, oFF, half_tbl;
   ByTe_4 sFp, sLp, sCp;
   size_t B;

   sLp = sizeof (ByTe_4 *);
   sFp = sizeof (ReaL_4 *);
   sCp = sizeof (ByTe_1 *); 

  /*************************************************************************/
  /*  Point to the correct sensor_tables structure.                        */
  /*************************************************************************/

   ex = ginfo.expt;
   sptr = ex->sen_tbl_ptr + sIndx;
   sptr->base_tbls = NO_MEMORY;
   sptr->base_misc_tbls = NO_MEMORY;
   sptr->num_tbls = 0;

  /************************************************************************/
  /*  Include only non-ascii tables when counting the number of tables to */
  /*  be applied.  Do not include mode-dependent tables.                  */
  /************************************************************************/

   c1 = ex->tbl_var;
   c2 = ex->tbl_type;
   for (i = 0; i < ex->num_tbls; ++i, ++c1, ++c2) {
      if ((*c1 == 4) || (*c1 == 5)) { continue; }
      if (*c2 != 1) { ++sptr->num_tbls; }
   }

  /************************************************************************/
  /* IF no tables then we are through here                                */
  /************************************************************************/

   if (sptr->num_tbls == 0) { return (ALL_OKAY); }

  /************************************************************************/
  /*  Pick up all INTEGER tables and allocate the space to hold the var,  */
  /*  expand, fmt, size, and pointer to the table values for each table.  */
  /************************************************************************/
   
   if (sptr->base_misc_tbls != NO_MEMORY) {
      free (sptr->base_misc_tbls);
      sptr->base_misc_tbls = NO_MEMORY;
   }
      
  /************************************************************************/
  /* REMEMBER - that each array of table information is valid only for    */
  /*  a specific sensor.                                                  */
  /*                                                                      */
  /*  Long pointer is to an array of table offsets, float pointer is a to */
  /*  an array of pointers to the tables, and the 3 character pointers    */
  /*  to arrays of the the table format, expand, and variable fields      */
  /************************************************************************/

   B = (sLp + sFp + 3 * sCp ) * sptr->num_tbls;
   if (B <= 0) { return (TBL_MISC_MALLOC); }

   sptr->base_misc_tbls = realloc (sptr->base_misc_tbls, B);
   if (sptr->base_misc_tbls == NO_MEMORY) { return (TBL_MISC_MALLOC); }
   
  /***********************************************************************/
  /*  Set various pointers to the space allocated to hold table info.    */
  /***********************************************************************/
   
   sptr->tbl_ptr = (ReaL_4 **) sptr->base_misc_tbls;
   oFF = sptr->num_tbls * sFp; 
   sptr->tbl_size = (ByTe_4 *) ((ByTe_1 *)sptr->base_misc_tbls + oFF);
   oFF += sptr->num_tbls * sLp; 
   sptr->tbl_var = (ByTe_1 *) ((ByTe_1 *)sptr->base_misc_tbls + oFF);
   oFF += sptr->num_tbls * sCp;
   sptr->tbl_expand = (ByTe_1 *) ((ByTe_1 *)sptr->base_misc_tbls + oFF);
   oFF += sptr->num_tbls * sCp;
   sptr->tbl_fmt = (ByTe_1 *) ((ByTe_1 *)sptr->base_misc_tbls + oFF);
   
  /***********************************************************************/
  /*  Loop over all defined tables for this virtual instrument.          */
  /***********************************************************************/
   
   c1 = ex->tbl_var;
   c2 = ex->tbl_type;
   for (i = 0; i < ex->num_tbls; ++i, ++c1, ++c2) {

  /***********************************************************************/
  /*  Exclude mode-dependent tables.                                     */
  /***********************************************************************/

      if ((*c1 == 4) || (*c1 == 5)) { continue; }

  /***********************************************************************/
  /*  Integer information is processed, not ASCII information.  We can   */
  /*  directly use index i even though only sptr->num_tbls is used to    */
  /*  allocate space since we added the restiction that all ASCII/MODE   */
  /*  tables be defined AFTER all other tables.                          */
  /***********************************************************************/
   
      if (*c2 != 1) {
         *(sptr->tbl_var + i) = *c1;
   
         rV = read_idf (ex->data_key, ex->exten, ex->version, 
                       (ByTe_1 *)(sptr->tbl_fmt + i), _TblFmT, i, 
                       (ByTe_4) sen, 1);
         if (rV < 0) {
            switch (rV) {
               case IDF_NOT_FOUND:  return (TBL_IDF_NOT_FOUND);
               case IDF_MANY_BYTES: return (TBL_IDF_MANY_BYTES);
               case IDF_TBL_NUM:    return (TBL_IDF_TBL_NUM);
               case IDF_CON_NUM:    return (TBL_IDF_CON_NUM);
               case IDF_NO_ENTRY:   return (TBL_IDF_NO_ENTRY);
               default:             return ((ByTe_2) rV);
            }
         }

  /***********************************************************************/
  /*  New error checks added.  If tbl_fmt = 0, only raw data can be used */
  /*  as look up values into the table.                                  */
  /***********************************************************************/

         if (*(sptr->tbl_fmt + i) == 0) {

  /***********************************************************************/
  /*  Only raw cal data allowed for sweep length dependent tables        */
  /***********************************************************************/

           if (*c2 == 2 && *(sptr->tbl_var + i) >= 0)
              return (TBL_VAR_NOT_CAL);

  /***********************************************************************/
  /*  TBL_VAR = 1 means processed data, which is not allowed, must be    */ 
  /*  raw.  Condition removed by CAG 04/02/98                            */
  /*                                                                     */
  /*  else if (*(ex->tbl_type+i)==0 && *(sptr->tbl_var+i)==1)            */
  /*  return (TBL_VAR_NOT_RAW);                                          */
  /***********************************************************************/

         }
   
  /***********************************************************************/
  /*  The TBL_EXPAND flag pertains to polynomial coefficients only.      */
  /*  Always set the flag to a 1 for lookup tables.  For the tables      */
  /*  defined for each step, do not expand coefficients for now          */
  /*  for sake of size.                                                  */
  /***********************************************************************/
  /***********************************************************************/
  /*  Mod CAG 6/29/99 - modified else condition. We need to leave        */
  /*  tbl_expand as is for procesed data since we can now use            */
  /*  lookup tables with processed data                                  */
  /***********************************************************************/
  
         if (*(sptr->tbl_fmt + i) > 0)
            *(sptr->tbl_expand + i) = (*c2 == 2 || *c2 == 4) ? 
	                                       0 : *(ex->tbl_expand + i);
         else
            *(sptr->tbl_expand + i) = ((*c1 != 1) && (*c1 != 5)) 
	                                    ? 1 : *(ex->tbl_expand + i);
      }
   }
    
  /***********************************************************************/
  /*  Determine how much space to allocate for INTEGER table values.     */
  /*  Ascii strings are saved in a different structure.                  */
  /***********************************************************************/
   
   ir_tbl_size (sIndx, &B);

  /***********************************************************************/
  /*  If all tables are ASCII tables, no space needs to be allocated to  */
  /*  hold INTEGER values and the pointers can be set to indicate no     */
  /*  memory for integer values.                                         */
  /***********************************************************************/
   
   if (B == 0) {
      sptr->base_tbls = NO_MEMORY;
      sptr->tbls = NO_MEMORY;
   } else {
      
  /***********************************************************************/
  /*  Allocate the space to hold the different table values.             */
  /***********************************************************************/
   
      if (sptr->base_tbls != NO_MEMORY) {
         free (sptr->base_tbls);
         sptr->base_tbls = NO_MEMORY;
      }
      if (B <= 0) { return (TBL_MALLOC); }

      if ((sptr->base_tbls = malloc (B)) == NO_MEMORY) { return (TBL_MALLOC); }
      sptr->tbls = (ReaL_4 *) sptr->base_tbls;
   
  /***********************************************************************/
  /*  Loop over all defined tables for this virtual instrument.          */
  /***********************************************************************/
   
      oFF = 0;
      c1 = ex->tbl_var;
      c2 = ex->tbl_type;
      for (i = 0; i < ex->num_tbls; ++i, ++c1, ++c2) {

  /***********************************************************************/
  /*  Exclude mode-dependent tables.                                     */
  /***********************************************************************/

         if ((*c1 == 4) || (*c1 == 5)) { continue; }

  /***********************************************************************/
  /*  Retrieve all integer (non-ascii) tables.                           */
  /***********************************************************************/
   
         if (*c2 != 1) {

  /***********************************************************************/
  /*  If the format is negative, this table is not used. Otherwise,      */
  /*  retrieve the table values.                                         */
  /***********************************************************************/
   
            if (*(sptr->tbl_fmt + i) >= 0) {
               *(sptr->tbl_ptr + i) = sptr->tbls + oFF;
               rV = ir_get_new_tbl (sen, i, *(sptr->tbl_expand + i), 
                                 *(sptr->tbl_fmt + i), *(sptr->tbl_ptr + i), 
                                 *(sptr->tbl_size + i), (ByTe_4) -1);
   
               if (rV != ALL_OKAY) { return ((ByTe_2) rV); }
   
               oFF += *(sptr->tbl_size + i);
   
  /***********************************************************************/
  /*  If the injest data is a signed binary quantity, the table is a fn  */
  /*  of raw sensor data which contains coefficients, and the table has  */
  /*  been expanded, move the table pointer to the middle of the table   */
  /*  so that indexing can be done for negative values (index by -n).    */
  /***********************************************************************/
   
               if (*(ex->d_type + sen) == 1 && *(sptr->tbl_var + i) == 0 && 
                   *(sptr->tbl_fmt + i) > 0 && *(sptr->tbl_expand + i) == 1)
               {
                  half_tbl = *(sptr->tbl_size + i) / 2;
                  *(sptr->tbl_ptr + i) += half_tbl;
               }
            }
         }
      }
   }

   return (ALL_OKAY);
}
