#include <stdlib.h>
#include "util_str.h"
#include "user_defs.h"
#include "ret_codes.h"
#include "libbase_udf.h"
#include "gen_defs.h"

/*****************************************************************************
 *                                                                           *
 *                         CONVERT_TO_UNITS SUBROUTINE                       *
 *                                                                           *
 *  DESCRIPTION                                                              *
 *    This routine is called in order to convert the raw telemetry values    *
 *  read by the READ_DREC() routine into the units specified by the calling  *
 *  routine.  This routine will convert sensor data, sweep data, calibration *
 *  and status (mode) data into the units specified by applying the tables   *
 *  specified by the calling routine.  It was decided that there would be no *
 *  tables to be applied to the data quality flag.  It is assumed that the   *
 *  calling routine has allocated enough space to hold all values being      *
 *  returned. Angular information has been added, but there are no tables to *
 *  be applied. Later versions of software will allow for table definition   *
 *  to convert from degrees (what is always returned from read_drec) to      *
 *  radians.                                                                 *
 *                                                                           *
 *  INPUT VARIABLES                                                          *
 *    unsigned long data_key     key which uniquely identifies the data set  *
 *                               being processed                             *
 *    char *exten                filename extension for the data to be used  *
 *    unsigned short vnum        version number to be associated with this   *
 *                               combination (allows for multiple opens)     *
 *    void *UDF                  ptr to memory location for the structure    *
 *                               that holds returned data values (read_drec) *
 *    short CoL                  the matrix column being processed           *
 *    char DType                 the type of data being processed - sensor,  *
 *                               sweep step, cal, mode, or d_qual            *
 *    char cal_set               calibration set number                      *
 *    char num_tbls              number of tables to apply                   *
 *    char *tbls_to_apply        array of tables to be applied               *
 *    short *tbl_oper            array of operations to be applied to tables *
 *    float *DataOut             array in which the data is returned in the  *
 *                               converted units                             *
 *    char chk_fill              flag indicating if data is to be checked    *
 *                               for FILL values                             *
 *    long fill_value            the value identified as the FILL value      *
 *                                                                           *
 *  USAGE                                                                    *
 *    x = convert_to_units (data_key, exten, vnum, UDF,       ,              *
 *                          DType, cal_set, num_tbls, tbls_to_apply,         *
 *                           tbl_oper, &DataOut, chk_fill, fill_value)       *
 *                                                                           *
 *  NECESSARY SUBPROGRAMS                                                    *
 *    sizeof ()                    the size of the specified object in bytes *
 *    ir_locate_ex()               determines if requested combination has   *
 *                                 already been processed and points to the  *
 *                                 correct structure allocated for the combo *
 *    ir_convert_sensor_data ()    converts sensor data into requested units *
 *    ir_convert_sweep_data ()     converts sweep data into requested units  *
 *    ir_convert_cal_data ()       converts cal. data into requested units   *
 *    ir_convert_mode_data ()      converts mode data into requested units   *
 *                                                                           *
 *  EXTERNAL VARIABLES                                                       *
 *    struct general_info ginfo    structure holding information concerning  *
 *                                 the experiment that is being processed    *
 *    unsigned int Tmp_Bytes       num of bytes allocated for scratch space  *
 *    void *Tmp_Space              scratch space for various uses            * 
 *                                                                           *
 *  INTERNAL VARIABLES                                                       *
 *    struct idf_data *ExDa        structure that holds all of the currently *
 *                                 returned data values to be processed      *
 *    struct experiment_info       a pointer to the structure that holds     *
 *          *ex                    specific experiment information           * 
 *    register float *f1           ptr to data values passed back to user    *
 *    register short i             looping variable                          *
 *    unsigned int bytes           the number of bytes to allocate space for *
 *    short rV                     holds value returned by called routine    *
 *    short tmp_only               number of tables that use intermediate    *
 *                                 accumulator                               *
 *    short tmp_final              num of times the primary and intermediate *
 *                                 accumulator are combined                  *
 *    char reset_called            flag indicating if LOCATE_EX was called   *
 *    void *tmp_ptr                ptr which holds address passed back by    *
 *                                 the call to the MALLOC routine            *
 *                                                                           *
 *  SUBSYSTEM                                                                *
 *    Display Level                                                          *
 *                                                                           *
 *   MODS:                                                                   *
 *                                                                           *
 *   CAG: 7/98 - Changes buffer check to any buffer used.                    *
 *               Always malloc for 3 buffers.                                *
 *               X000 operations is not a restriction any longer             *
 ****************************************************************************/

ByTe_2 convert_to_units (u_ByTe_4 data_key, ByTe_1 *exten, u_ByTe_2 vnum, 
                         void *UDF, ByTe_1 DType, ByTe_1 CalOrMode, 
                         ByTe_1 num_tbls, ByTe_1 *tbls_to_apply,
                         ByTe_2 *tbl_oper, ReaL_4 *DataOut, ByTe_1 chk_fill, 
                         ByTe_4 fill_value)
{
  extern struct general_info ginfo;
  extern size_t Tmp_Bytes;
  extern void   *Tmp_Space;

  struct idf_data *ExDa;
  struct experiment_info *ex;
  register ReaL_4 *f1, *f2, *fEnD;
  register ByTe_2 *OP, I;
  register ByTe_1 *TbL;
  size_t bytes;
  ByTe_2 rV;
  ByTe_1 reset_called, BufUsed = 0, IFmt;

  /***************************************************************************/
  /*  Check to see if the combination being processed has been processed     */
  /*  before.  If not, an error condition - probably didn't call FILE_OPEN.  */
  /*  Since ir_locate_ex() is called by more than one routine, return an     */
  /*  error code that indicates which calling routine resulted in the error. */
  /*  Since a 0 is passed for the last parameter, the only possible error is */
  /*  that the requested combination was not found among processed combos.   */
  /***************************************************************************/
 
  ExDa = (struct idf_data *) UDF;
  if (!ginfo.called_locate)
   {
     rV = ir_locate_ex (data_key, exten, vnum, 0);
     if (rV != ALL_OKAY)
       return (CNVT_NOT_FOUND);

     ginfo.called_locate = 1;
     reset_called = 1;
   }
  else
    reset_called = 0;
  rV = ALL_OKAY;
  ex = ginfo.expt;

  /***************************************************************************/
  /*  Check for 64 bit return data.  Set the input format and then set the   */
  /*  data dtype back to the normal positive value.                          */
  /***************************************************************************/

  if (DType < 0)
  {
     IFmt = 1;
     DType = -DType;
  }
  else
     IFmt = 0;


  /***************************************************************************/
  /*  Bad data type?                                                         */
  /***************************************************************************/

  if (DType > STOP_AZ_ANGLE)
  {
     if (reset_called)
        ginfo.called_locate = 0;
      return (CNVT_BAD_DTYPE);
  }

  /***************************************************************************/
  /*  Quick table check to see if tables requested in operation are not      */
  /*  greater than the total number of tables and if any buffers used        */ 
  /***************************************************************************/

   I = 0;
   OP = tbl_oper;
   TbL = tbls_to_apply;
   while ( I++ < num_tbls ) {
      if (*OP >= 1000) {
         if ((*OP > 13000 && *OP < 20000) || *OP >= 22000) {
            BufUsed = 2;
         } else {
            if (BufUsed == 0) { BufUsed = 1; }
         }
      }

      if (*TbL  >= ex->num_tbls) {
         if (reset_called)
            ginfo.called_locate = 0;
         return (CNVT_BAD_TBL_NUM);
      }
      ++OP;
      ++TbL;
   }

   if (BufUsed) {
      bytes = sizeof (ReaL_8) * 3 * ex->swp_len;

      if (bytes > Tmp_Bytes) {
         if (bytes <= 0) {
            if (reset_called) { ginfo.called_locate = 0; }
            return (CNVT_TMP_MALLOC);
         }

  /***********************************************************************/
  /*  CAG 12/16/08 - removed a realloc in favor of a free/malloc         */
  /***********************************************************************/
   
	 free(Tmp_Space);
         if ((Tmp_Space = malloc(bytes)) == NO_MEMORY) {
            if (reset_called)
               ginfo.called_locate = 0;
            return (CNVT_TMP_MALLOC);
         }
         Tmp_Bytes = bytes;
      }

    /***********************************************************************/
    /*  Initialize temporary accumulator for help with debugging.          */
    /***********************************************************************/

       f1 = (ReaL_4 *) Tmp_Space;
       fEnD = f1 + 3 * ex->swp_len;
       for ( ; f1 < fEnD; )
         *f1++ = OUTSIDE_MAX;
   }

  /***************************************************************************/
  /*  Convert the sensor data.                                               */
  /***************************************************************************/

  if (DType == SENSOR)
     rV = ir_convert_sensor_data (num_tbls, tbls_to_apply, tbl_oper, 
                             DataOut, IFmt,  chk_fill, fill_value, UDF);

  /***************************************************************************/
  /*  Convert the sweep step data.                                           */
  /***************************************************************************/

  else if (DType == SCAN_INDEX)
     rV = ir_convert_sweep_data (num_tbls, tbls_to_apply, tbl_oper, 
                            DataOut, IFmt, UDF);

  /***************************************************************************/
  /*  Convert the calibration data.                                          */
  /***************************************************************************/

  else if (DType == ANCILLARY)
     rV = ir_convert_cal_data (CalOrMode, num_tbls, tbls_to_apply, 
                                 tbl_oper, DataOut, IFmt, chk_fill, 
                                 fill_value, UDF);

  /***************************************************************************/
  /*  Convert the status (mode) data.                                        */
  /***************************************************************************/

  else if (DType == MODE)
     rV = ir_convert_mode_data (CalOrMode, num_tbls, tbls_to_apply, 
                                  tbl_oper, DataOut, IFmt, UDF);

  /***************************************************************************/
  /*  Simply copy the data quality value into the data array being returned. */
  /*  It is already the correct one for the sensor of interest, assuming     */
  /*  read_drec has been called.                                             */
  /***************************************************************************/

  else if (DType == D_QUAL)
   {
     f1 = DataOut;
     *f1 = (ReaL_4) ExDa->d_qual;
   }

  /***************************************************************************/
  /*  All angle values are in degrees.  Must define a table to convert to    */
  /*  radians.                                                               */
  /***************************************************************************/

  else if (DType == PITCH_ANGLE)
   {
     /***********************************************************************/
     /*  If no pitch angles were computed, set the data that is returned to */
     /*  a value that indicates a problem with pitch angles.  Since there   */
     /*  one value per sweep step (per data element), use num_sample.       */
     /***********************************************************************/

     f1 = DataOut;
     if (ExDa->num_pitch == 0)
      {
        fEnD = f1 + ExDa->num_sample;
        for (; f1 < fEnD;)
          *f1++ = OUTSIDE_MIN;
      }
     else
      {
        f2 = ExDa->pitch_angles;
        fEnD = f2 + ExDa->num_pitch;
        for (; f2 < fEnD;)
         *f1++ = *f2++;
      }
   }

  else if (DType == START_AZ_ANGLE)
   {
     f1 = DataOut;
     f2 = ExDa->start_az;
     fEnD = f2 + ExDa->num_angle;
     for (; f2 < fEnD;)
      *f1++ = *f2++;
   }

  else if (DType == STOP_AZ_ANGLE)
   {
     f1 = DataOut;
     f2 = ExDa->stop_az;
     fEnD = f2 + ExDa->num_angle;
     for (; f2 < fEnD;)
      *f1++ = *f2++;
   }

  if (reset_called)
    ginfo.called_locate = 0;
  return (rV);
}
