#include <stdlib.h>
#include <string.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "libbase_udf.h"
#include "OpSySInD.h"

/*******************************************************************************
 *                                                                             *
 *                         IR_ALLOC_PITCH_INFO SUBROUTINE                      *
 *                                                                             *
 *  DESCRIPTION                                                                *
 *    This routine is called to allocate the structure that holds pitch angle  *
 *  information.  Those data elements not retrieved from the VIDF file are     *
 *  initialized.                                                               *
 *                                                                             *
 *  INPUT VARIABLES                                                            *
 *    short etime_yr               the stop time requested (year component)    *
 *    short etime_day              the stop time requested (day component)     *
 *    long etime_sec               the stop time requested (seconds component) *
 *    long etime_nsec              the stop time requested (nanoseconds)       *
 *                                                                             *
 *  USAGE                                                                      *
 *    x = ir_alloc_pitch_info (etime_yr, etime_day, etime_sec, etime_nsec)     *
 *                                                                             *
 *  NECESSARY SUBPROGRAMS                                                      *
 *    sizeof ()                    the size of the specified object in bytes   *
 *    malloc()                     allocates memory                            *
 *    strcpy()                     copies a string to another string variable  *
 *                                                                             *
 *  EXTERNAL VARIABLES                                                         *
 *    struct general_info ginfo    structure that holds information concerning *
 *                                 the experiment that is being processed      *
 *                                                                             *
 *  INTERNAL VARIABLES                                                         *
 *    struct experiment_info *ex   a pointer to the structure that holds       *
 *                                 specific experiment information             *
 *    register struct pitch_info   a pointer to the structure that holds pitch *
 *         *pa_ptr                 angle information                           *
 *    register long i              looping variable                            *
 *    unsigned int bytes           the number of bytes to allocate             *
 *                                                                             *
 *  SUBSYSTEM                                                                  *
 *    Display Level                                                            *
 *                                                                             *
 ******************************************************************************/

ByTe_2 ir_alloc_pitch_info (ByTe_2 etime_yr, ByTe_2 etime_day, 
                            ByTe_4 etime_sec, ByTe_4 etime_nsec)
{
   extern struct general_info ginfo;

   struct experiment_info *ex;
   register struct pitch_info *pa_ptr;
   register ByTe_4 i;
   size_t   bytes;

   /**************************************************************************/
   /*  Allocate the structure that holds the pitch angle information.        */
   /**************************************************************************/

   ex = ginfo.expt;
   bytes = sizeof (struct pitch_info);
   if (bytes <= 0)
     return (PITCH_MALLOC);
   if ((ex->bmem.base_pitch_info = malloc (bytes)) == NO_MEMORY)
     return (PITCH_MALLOC);
   ex->pitch_angles = (struct pitch_info *) ex->bmem.base_pitch_info;
   pa_ptr = ex->pitch_angles;
   strcpy (pa_ptr->exten, ex->exten);
   pa_ptr->get_pa_data = 1;
   pa_ptr->file_status = 0;

   /**************************************************************************/
   /*  Save time passed to FILE_OPEN for parent data set.                    */
   /**************************************************************************/

   pa_ptr->stop_year = etime_yr;
   pa_ptr->stop_day = etime_day;
   pa_ptr->stop_milli = etime_sec;
   pa_ptr->stop_nano = etime_nsec;

   /**************************************************************************/
   /*  Initialize those values not read from the VIDF file.                  */
   /**************************************************************************/

   for (i = 0; i < 3; ++i)
    {
      pa_ptr->next_data[i] = 1;
      pa_ptr->all_done[i] = 0;
      pa_ptr->idf_data_ptr[i] = NO_MEMORY;
    }

   pa_ptr->base_tbls = NO_MEMORY;
   pa_ptr->tbl_apply = NO_MEMORY;
   pa_ptr->tbl_oper = NO_MEMORY;
   pa_ptr->base_time = NO_MEMORY;
   pa_ptr->btime_ms = NO_MEMORY;
   pa_ptr->etime_ms = NO_MEMORY;
   pa_ptr->base_normal = NO_MEMORY;
   pa_ptr->unit_normals = NO_MEMORY;
   pa_ptr->mag_normal = NO_MEMORY;
   pa_ptr->base_data = NO_MEMORY;
   pa_ptr->data_val = NO_MEMORY;
   pa_ptr->time_frac = NO_MEMORY;

   /***************************************************************************/
   /*  Since the data set for the three sensors could have different TIME_OFF */
   /*  values, make sure data from the data set returns one consistent time.  */
   /***************************************************************************/

   pa_ptr->ptimes.byear = 0;
   pa_ptr->ptimes.bday = 0;
   pa_ptr->ptimes.bmilli = 0;
   pa_ptr->ptimes.bnano = 0;
   pa_ptr->ptimes.eyear = 0;
   pa_ptr->ptimes.eday = 0;
   pa_ptr->ptimes.emilli = 0;
   pa_ptr->ptimes.enano = 0;

   /***************************************************************************/
   /*  Allocate space to hold the unit normals for each sensor defined in the */
   /*  parent data set.  The 4 represents the 3 components of the unit vector */
   /*  plus 1 value for the computed magnitude.                               */
   /***************************************************************************/
   /***************************************************************************/
   /*  Allocate space to hold the data values and normalization factors for   */
   /*  the three components.  Allocate one triplet per element of the parent  */
   /*  sweep.  Use SWP_LEN since maximum length of parent sweep.              */
   /***************************************************************************/
   /***************************************************************************/
   /*  Allocate space to save the milliseconds start/stop time component for  */
   /*  each sample.  Cannot assume that the start of B = the end of A since   */
   /*  timing may be parallel down the column so need two arrays.             */
   /***************************************************************************/

   if (ex->swp_len <= 0 || ex->num_sensor <= 0)
     return (PA_TIME_MALLOC);

   bytes = 2 * sizeof (ByTe_4) * ex->swp_len;                      /* time  */
   bytes += (sizeof (ReaL_8) + sizeof (ReaL_4)) * 3 * ex->swp_len;
   bytes += sizeof (ReaL_8) * 4 * ex->num_sensor;

   if ((pa_ptr->base_data = malloc (bytes)) == NO_MEMORY)
     return (PA_TIME_MALLOC);
   pa_ptr->base_time = NO_MEMORY;
   pa_ptr->base_normal = NO_MEMORY;

   pa_ptr->unit_normals = (ReaL_8 *) pa_ptr->base_data;
   bytes = ex->num_sensor * 3 * sizeof (ReaL_8);
   pa_ptr->mag_normal = (ReaL_8 *) ((ByTe_1 *)pa_ptr->base_data + bytes);
   bytes += ex->num_sensor * sizeof (ReaL_8);
   pa_ptr->data_val = (ReaL_8 *) ((ByTe_1 *)pa_ptr->base_data + bytes);
   bytes += ex->swp_len * 3 * sizeof (ReaL_8);
   pa_ptr->time_frac = (ReaL_4 *) ((ByTe_1 *)pa_ptr->base_data + bytes);
   bytes += ex->swp_len * 3 * sizeof (ReaL_4);
   pa_ptr->btime_ms = (ByTe_4 *) ((ByTe_1 *)pa_ptr->base_data + bytes);
   bytes += ex->swp_len * sizeof (ByTe_4);
   pa_ptr->etime_ms = (ByTe_4 *) ((ByTe_1 *)pa_ptr->base_data + bytes);

   return (ALL_OKAY);
}
