#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "util_str.h"
#include "ret_codes.h"
#include "gen_defs.h"
#include "idf_defs.h"
#include "libbase_udf.h"

/******************************************************************************
 *                                                                            *
 *                          IR_ALLOC_EXP_ONCE SUBROUTINE                      *
 *                                                                            *
 *  DESCRIPTION                                                               *
 *    This routine is called to allocate memory for an experiment_info        *
 *  structure which holds information read from the header and data records   *
 *  associated with the data set being processed.  One structure is allocated *
 *  for each of the different data sets.  Once memory has been allocated,     *
 *  pointers to the header and data elements are assigned.                    *
 *                                                                            *
 *  INPUT VARIABLES                                                           *
 *    void *idf_data_ptr           ptr to memory location for the structure   *
 *                                 holding returned data values (read_drec)   *
 *                                                                            *
 *  USAGE                                                                     *
 *    x = ir_alloc_exp_once (idf_data_ptr)                                    *
 *                                                                            *
 *  NECESSARY SUBPROGRAMS                                                     *
 *    sizeof ()                    the size of the specified object in bytes  *
 *    malloc()                     allocates memory                           *
 *    free ()                      frees previously allocated memory          *
 *    ir_read_header()             reads data from the header file            *
 *    ir_sensor_combo()            determines how many unique combinations    *
 *                                 exist among the tables used by all sensors *
 *    ir_get_mode_info()           retrieves any mode-dependent tables        *
 *    read_idf()                   reads information from the VIDF file       *
 *    ir_alloc_timing ()           allocates and initializes some timing      *
 *                                 structures that are needed by FILL_DATA()  *
 *                                                                            *
 *  EXTERNAL VARIABLES                                                        *
 *    struct general_info ginfo    structure holding information concerning   *
 *                                 the experiment that is being processed     *
 *                                                                            *
 *  INTERNAL VARIABLES                                                        *
 *    struct experiment_info *ex   a pointer to the structure that holds      *
 *                                 specific experiment information            *
 *    size_t B                     the number of bytes to allocate            *
 *    long rval                    the value returned by the called routine   *
 *    short rV                     holds value returned by called routine     *
 *                                                                            *
 *  SUBSYSTEM                                                                 *
 *    Display Level                                                           *
 *                                                                            *
 *****************************************************************************/

ByTe_2 ir_alloc_exp_once (void *idf_data_ptr)
{
   extern struct general_info ginfo;

   register struct experiment_info *ex;
   ByTe_2 rV;

  /************************************************************************/
  /*  If the memory for the experiment_info structure associated with the */
  /*  experiment/virtual/extension combination has already been allocated */
  /*  simply return.                                                      */
  /************************************************************************/

   ex = ginfo.expt;
   if (ex->info_ptr->h_size != -1) { return (ALL_OKAY); }

  /************************************************************************/
  /*  Get the different table combinations used by the various sensors.   */
  /************************************************************************/

   rV = ir_sensor_combo ();
   if (rV != ALL_OKAY) { return (rV); }

  /************************************************************************/
  /*  Get the table(s) for the modes.                                     */
  /************************************************************************/

   rV = ir_get_mode_info ();
   if (rV != ALL_OKAY) { return (rV); }

  /*************************************************************************/
  /*  Retrieve the information from the header record.                     */
  /*************************************************************************/

   rV = ir_read_header (idf_data_ptr);
   if (rV != ALL_OKAY) {
      if (rV == RHDR_READ_ERROR)
        return (ALLOC_HDR_READ_ERROR);
      else if (rV == RHDR_HDR_MALLOC)
        return (ALLOC_HDR_MALLOC);
      else if (rV == RHDR_HDR_REALLOC)
        return (ALLOC_HDR_REALLOC);
      else { return (rV); }
    }

   return (ALL_OKAY);
}
