;%W% %G%
;
;-------------------------------------------------------------------------
; Procedure Name: 	lltodr
;	Author:  J. M. O'Meara
; Purpos:  Convert latitude/longitude to  3-D position vector
;
;
; INPUTS:
;   lat                   Latitude of the point on the earth
;   lon                   Longitude of the point on the earth
;   r                     Radius scalar
;
; Outputs:
;		x,y,z                 3-D position vector
;
;------------------------------------------------------------------------

PRO lltodr,lat,lon,r,x,y,z

dtor = !DPI/180.D

theta = (90D - lat)*dtor
phi = lon*dtor
x = r*SIN(theta)*COS(phi)
y = r*SIN(theta)*SIN(phi)
z = r*COS(theta)

END

;-------------------------------------------------------------------------
; Procedure Name:  lltopix_arr
;	Authors:  M. Brittnacher & J. M. O'Meara (Univ. of Washington)
;	Date:			November 19, 1997
; Purpose:  Calculate the row,col pixel locations of a point, given
;           the latitude and longitude of the point on the earth
;           and the spacecraft's orbit and attitude
;
;	Inputs:
;		lat           Latitude of the point on the earth
;		lon           Longitude of the point on the earth
;		emis_hgt      Radiation emission height (km)
;		xax           X axis: direction of l0
;		yax           Y axis: zax X xax
;		zax           Z axis: direction in plane of xax and att perp to xax
;   orb           GCI orbital postion (km)
;		time          Event time vector
;   irotm         INVERSE of rotation matrix supplied by ic_cgi_to_geo
;
;	Outputs:
;		row           Pixel row location
;		col           Pixel column location
;		angle         Angle between Lpix and pos
;
;	Program calls on:
;		lltodr
;-------------------------------------------------------------------------
;------------------------------------------------------------------------------------
; This the latest version of GIFWALK program.  Applying the FUV modification to it
;
;Modified:
;
; Hassan Dougani	10/04/1999			Applying the latest FUV modifications
; Hassan Dougani	10/28/1999			Modify structures for multiple Instrument capability
;
;------------------------------------------------------------------------------------
;

PRO lltopix_arr,ImageInfo,lat,lon,emis_hgt,xax,yax,zax,orb,time,$
          row_arr,col_arr,angle,irotm,xsize,ysize


ncols = xsize
nrows = ysize
pc =    DOUBLE(ImageInfo.angl_res_c)
pr =    DOUBLE(ImageInfo.angl_res_r)
dtor =  !DPI/180.D
rtod =  180.D/!DPI

;radius of the earth (km)
r = 6378.136D + emis_hgt
;
;convert latitude,longitude to p_geo vector
lltodr,lat,lon,r,x,y,z

; obtain position vector for point on Earth's surface
posx = irotm(0,0)*x + irotm(0,1)*y + irotm(0,2)*z
posy = irotm(1,0)*x + irotm(1,1)*y + irotm(1,2)*z
posz = irotm(2,0)*x + irotm(2,1)*y + irotm(2,2)*z

tmpx = posx - orb(0)
tmpy = posy - orb(1)
tmpz = posz - orb(2)

Ntmp = SQRT(tmpx*tmpx + tmpy*tmpy + tmpz*tmpz)
Npos = SQRT(posx*posx + posy*posy + posz*posz)

; obtain unit vector in look direction for this pixel
lpixx = tmpx/Ntmp
lpixy = tmpy/Ntmp
lpixz = tmpz/Ntmp

; Determine angle between Lpix and pos
angle = ACOS((lpixx*posx + lpixy*posy + lpixz*posz)/Npos)

; Determine projections of lpix on the x, y and z axes
lx = lpixx*xax(0) + lpixy*xax(1) + lpixz*xax(2)
ly = lpixx*yax(0) + lpixy*yax(1) + lpixz*yax(2)
lz = lpixx*zax(0) + lpixy*zax(1) + lpixz*zax(2)


yrot = rtod*ATAN(lz,lx)
zrot = rtod*ATAN(ly,lx)

	; I have to subtract 0.5 pixels in y direction to make everything match
	; do not ask why, I do not know, but it works
	; hfrey, July 10, 2002
row_arr = yrot/pr + ((nrows-1)/2.) - 0.5	; y direction
col_arr = zrot/pc + ((ncols-1)/2.)		; x direction

;if (n_elements(lat) eq 1) then doprint=1 else doprint = 0
doprint = 0
IF doprint THEN BEGIN
print,'  '
print,'lltopix................................'
print,'lat/lon   ',lat,lon,format='(a15,2f15.5)'
print,'p_geo     ',x,y,z,format='(a15,3f15.5)'
print,'pos       ',orb,format='(a15,3f15.5)'
print,'lpix      ',lpixx,lpixy,lpixz,format='(a15,3f15.5)'
print,'lx ly lz  ',lx,ly,lz,format='(a15,3f15.5)'
print,'yrot zrot ',yrot,zrot,format='(a15,2f15.5)'
print,'row col   ',row_arr,col_arr,format='(a15,2f15.5)'
print,'end lltopix............................'
print,'  '
ENDIF

END

;-------------------------------------------------------------------------
