#include <Python.h>
#include <endian.h>

#include "depwi.h"

const char depwi_doc[] = 
 "A python wrapper for Scott Allendorf's depwi library\n"
 "\n"
 "Most of the functions in this module take or emit DE PWI records.  In\n"
 "concrete terms, DE PWI records are 1768 byte binary strings which represent\n"
 "442 integers.  The disk files are stored in big-endian order, but the record\n"
 "reading function, getrec returns records that are in native endian order.\n"
 "If the host's native order is not big-endian then records are byte-swapped\n"
 "in memory by getrec before returning to the caller.\n"
 "\n"
 "The rest of this modules functions read PWI records that are EXPECTED to be\n"
 "in native endian order, so use 'getrec' to be safe.\n"
 "\n"
 "WARNING: This module is NOT THREAD SAFE, it wraps fortran code which has\n"
 "common block items that are altered as the program runs.\n";

/*****************************************************************************/
/* getrec */
const char depwihelp_getrec[] = 
"Read a single DE PWI record from a file and return it as a byte string.\n"
"\n"
"NOTE: This function preforms a blocking read.  The global interperater lock\n"
"is held while reading the input, this may need to be altered.\n"
"\n"
"Argument:\n"
"  A file object\n"
"Return:\n"
"  A 1768 byte binary string representing 442 integers in native endian order,\n"
"    -OR-\n"
"  An empty string in the end of the file or a partial record was encountered\n"
;

static PyObject* depwi_getrec(PyObject* self, PyObject* args)
{
	PyObject* pFile = NULL;
	FILE* pCfile = NULL;
	size_t nRead = 0;
	unsigned char uBuf[1768] = {'\0'};
#if __BYTE_ORDER == __LITTLE_ENDIAN
	unsigned char uTmp = 0;
	int i = 0;
#endif	
	
	if(!PyArg_ParseTuple(args, "O:getrec", &pFile))
		return NULL;

	if(!PyFile_Check(pFile)){
		PyErr_SetString(PyExc_TypeError, "File object expected");
		return NULL;
	}
	
	pCfile = PyFile_AsFile(pFile);
	
	if( (nRead = fread(uBuf, 1, 1768, pCfile)) != 1768){
		uBuf[0] = '\0';
		nRead = 0;
	}
#if __BYTE_ORDER == __LITTLE_ENDIAN
	else{
		/* Handle byte swapping if needed */
		for(i = 0; i < 1768; i += 4){
			uTmp = uBuf[i];
			uBuf[i] = uBuf[i+3];
			uBuf[i+3] = uTmp;
			uTmp = uBuf[i+1];
			uBuf[i+1] = uBuf[i+2];
			uBuf[i+2] = uTmp;
		}
	}
#endif
	
	return Py_BuildValue("s#",uBuf,nRead);
}


/*****************************************************************************/
/* gettime */

const char depwihelp_gettime[] = 
  "Read the timestamp from the DE PWI record and return it in an array of\n" 
  "integer numbers. \n"
  "(Calls getdat_ in the fortran libraries)\n"
  "\n"
  "Argument:\n"
  "  A DE PWI Low-Rate Record (see module level doc)\n"
  "\n"
  "Return:\n"
  "  An 8-tuple containing the following integer enteries\n"
  "\n"
  "   0       Date in form YYDDD (January 1 => DDD = 001).\n"
  "   1       Millisecond of the day (0 - 86399999)\n"
  "   2       Year\n"
  "   3       Day of year (January 1 = 1)\n"
  "   4       Hour (0 - 23)\n"
  "   5       Minute (0 - 59)\n"
  "   6       Second (0 - 59)\n"
  "   7       Millisecond (0 - 999)\n"
  "\n";

static PyObject* depwi_gettime(PyObject* self, PyObject* args)
{
	int t[8] = {0};
	int nLen = 0;
	const char* pRec = NULL;
	
	if(!PyArg_ParseTuple(args, "s#:gettime", &pRec, &nLen))
		return NULL;
	
	if(nLen < 1768){
		PyErr_SetString(PyExc_ValueError, "Input record length is not 1768 bytes");
		return NULL;
	}
	
	getdat_((int*)pRec, t);
	
	return Py_BuildValue("(iiiiiiii)",t[0],t[1],t[2],t[3],t[4],t[5],t[6],t[7]);
}

/*****************************************************************************/
/* timecmp */

const char depwihelp_timecmp[] = 
  "This function will compare the two specified times and determine which\n"
  "one is later.\n"
  "\n"
  "Arguments:\n"
  "  Two DE PWI Low-rate records (see module level doc)\n"
  "Return:\n"
  "  An indication of which time is later.  A value of 1 indicates that \n"
  "  the first time is later, a value of -1 indicates that second time is\n"
  " later, a value of 0 indicates that the two times are the same.\n"
  "\n";

static PyObject* depwi_timecmp(PyObject* self, PyObject* args)
{
	int nLen1 = 0, nLen2 = 0;
	const char* pRec1 = NULL;
	const char* pRec2 = NULL;
	
	if(!PyArg_ParseTuple(args, "s#s#:timecmp", &pRec1, &nLen1, &pRec2, &nLen2))
		return NULL;
	
	if(nLen1 < 8){
		PyErr_SetString(PyExc_ValueError, "Record 1 is too short, < 8 bytes");
		return NULL;
	}
	
	if(nLen2 < 8){
		PyErr_SetString(PyExc_ValueError, "Record 2 is too short, < 8 bytes");
		return NULL;
	}
	
	int nRet = timcmp_((int*)pRec1, (int*)pRec2);
	
	return Py_BuildValue("i",nRet);
}

/*****************************************************************************/
/* getSfr, getLfc Helpers */

static int _parseAmpArgs(const char* sFunc, PyObject* args,
		                   const char** ppRec, int* pLen)
{
	const char* sRecv = NULL;
	int nRecv = -1;
	static char sBuf[128] = {'\0'};
	
	snprintf(sBuf, 127, "s#s:%s", sFunc);

	if(!PyArg_ParseTuple(args, sBuf, ppRec, pLen, &sRecv))
		return -1;
	
	if(*pLen != 442*4){
		PyErr_SetString(PyExc_ValueError, "Input record length is not 1768 bytes");
		return -1;
	}
	
	if(strcmp(sRecv, "A") == 0) nRecv = 0;
	if(strcmp(sRecv, "B") == 0) nRecv = 1;
	if(nRecv == -1){
		snprintf(sBuf, 127, "Unexpected receiver value '%s'", sRecv);
		PyErr_SetString(PyExc_ValueError, sBuf);
		return -1;
	}
	
	return nRecv;
}

static const char* _getAntStr(int nAnt, int nSfr)
{
	const char* sAnt = NULL;
	static char sMsg[128] = {'\0'};  
	
	/* Set the antenna str */
	switch(nAnt){
	case 0: sAnt = "Es"; break;
	case 1: sAnt = "Ez"; break;
	case 2: sAnt = "Ex"; break;
	case 3: 
		if(nSfr)
			sAnt = "B";
		else
			sAnt = "H";
		break;
	default:
		snprintf(sMsg, 127, "Unexpected antenna value in data record: %d", nAnt);
		PyErr_SetString(PyExc_ValueError, sMsg);
	}
	
	return sAnt;
}

/*****************************************************************************/
/* getsfr */

const char depwihelp_getsfr[] = 
"Get calibrated SFR data from a DE PWI data record.\n"
"\n"
"Read the SFR data from a DE PWI record, calibrates it, and return it as a\n"
"floating point spectral densities.  In addition the frequency table for the\n"
"amplitudes is returned as well\n"
"\n"
"Arguments:\n"
"  A DE PWI Record\n"
"  A single character string selecting the desired receiver, 'A' or 'B'\n"
"Returns:\n"
"  A string indicating the antenna, one of 'Es', 'Ez', 'Ex', 'B'\n"
"\n"
"  A 32 element list.  Each entry is the list is the 7-tuple:\n"
"    0 Date as a 5-digit integer, so 1981-02-01 is 81032\n"
"    1 The integer millisecond of day for the record\n"
"    2 The frequency value for the channel in Hz\n"
"    3 Amplitude 1\n"
"    4 Amplitude 2\n" 
"    5 Amplitude 3\n"
"    6 Amplitude 4\n"
"\n"
"The returned amplitudes are in units of V**2 m**-2 Hz**-1 for the electric\n"
"antennas, or units of Gammas**2 Hz**-1 for the magnetic antenna\n"
"\n"
"Note: The values in the data tuples are output in order of the values in the\n"
"  Data record.  In many cases the 1st value should be dropped as the\n"
"  instrument  had not yet settled when the 1st measurement was taken\n";

static PyObject* depwi_getsfr(PyObject* self, PyObject* args)
{
	/* Inputs */
	const char* pRec = NULL;
	int nLen = 0;
	int nSfr = -1;
	
	/* Outputs */
	const char* sAnt = NULL;
	int nAnt = -1;
	double dFreq[32][4] = {{0}};
	double dAmp[32][4] = {{0}};
	
	PyObject* pList = NULL;
	PyObject* pTup = NULL;
	PyObject* pRet = NULL;
	
	/* Misc */
	int i = 0, j = 0;
	int nIncMilli = 0;
	int lRecTime[8] = {0};
	int lSampTime[2] = {0};
	int nSubRet = 0;

	if( (nSfr = _parseAmpArgs("getsfr", args, &pRec, &nLen)) < 0)
		return NULL;
	
	/* Get data via the fortran lib, 1st the record time then the amplitudes */
	getdat_((int*)pRec, lRecTime);
	
	getsfr_((int*)pRec, &nSfr, &nAnt, (double**)dAmp, (double**)dFreq);
	
	pList = PyList_New(32);         /* New Reference */
	
	for(i = 0; i<32; i++){
		
		nIncMilli = 1000 * (i/4);
		timadd_(lRecTime, &nIncMilli, lSampTime, &nSubRet);
		
		if(nSubRet != 1){
			Py_DECREF(pList);
			PyErr_SetString(PyExc_ValueError, "Invalid return from libdepwi, timadd_");
			return NULL;
		}
		
		pTup = PyTuple_New(7);                   /* New Ref */
		
		/* Set Time */
		PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));
		PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
		
		/* Set Freq */
		PyTuple_SetItem(pTup, 2, PyFloat_FromDouble(dFreq[i][0]));
			
		for(j = 0; j<4; j++){
			/* The float is a new reference, but SetItem is one of the few
			   reference stealing functions, so it will handle the new
				float and we don't have to */
			PyTuple_SetItem(pTup, j+3, PyFloat_FromDouble( dAmp[i][j] ) );
		}
		
		PyList_SetItem(pList, i, pTup);          /* Ref passed */
	}
	
	if( (sAnt = _getAntStr(nAnt, 1)) == NULL){
		Py_DECREF(pList);
		return NULL;
	}
			
	pRet = Py_BuildValue("sO", sAnt, pList);
	
	Py_DECREF(pList);                           /* Return tup owns list now */
	return pRet;
}

/*****************************************************************************/
/* getsfr - phase */

const char depwihelp_getsfr_phase[] = 
"Get calibrated SFR phase data from a DE PWI record.\n"
"\n"
"Read the SFR phase data from the DE PWI record, calibrate it, and return it.\n"
"\n"
"Arguments:\n"
"  A DE PWI Record\n"
"Returns:\n"
"  * A string providing the antenna connected to SFR-A: Es, Ez, Ex, B\n"
"\n"
"  * A string providing the antenna connected to SFR-B: Es, Ez, Ex, B\n"
"\n"
"  * A 32 element list.  Each entry in the list is the 11-tuple:\n"
"     0 Date as 5-digit integer, so 1982-11-17 is 82321\n"
"     1 The integer millisecond of day for the record\n"
"     2 The frequency value for both SFR channels in Hz\n"
"     3 The phase difference in degrees for the 1st amplitude set\n"
"     4 The phase difference in degrees for the 2nd amplitude set\n"
"     5 The phase difference in degrees for the 3rd amplitude set\n"
"     6 The phase difference in degrees for the 4th amplitude set\n"
"     7 The correlation coefficient for the 1st amplitued set\n"
"     8 The correlation coefficient for the 2nd amplitued set\n"
"     9 The correlation coefficient for the 3rd amplitued set\n"
"    10 The correlation coefficient for the 4th amplitued set\n"
"\n"
"Note: A correlation coefficient of -1.0 means the data are invalid!\n"
"\n";

static PyObject* depwi_getsfr_phase(PyObject* self, PyObject* args)
{
	/* Inputs */
	const char* pRec = NULL;
	int nLen = 0;
	
	/* Misc */
	int i = 0, j = 0;
	int nIncMilli = 0;
	int nSubRet = 0;
	int lSampTime[2] = {0};
	
	/* Outputs */
	int lRecTime[8] = {0};
	int nAntA = -1;
	int nAntB = -1;
	const char* sAntA = NULL;
	const char* sAntB = NULL;
	double phase[32][4] = {{0}};
	double corr[32][4] = {{0}};
	double freq[32][4] = {{0}};
	
	PyObject* pList = NULL;
	PyObject* pTup = NULL;
	PyObject* pRet = NULL;

	if(!PyArg_ParseTuple(args, "s#:getsfr_phase", &pRec, &nLen))
		return NULL;
	
	if(nLen < 1768){
		PyErr_SetString(PyExc_ValueError, "Input record length is not 1768 bytes");
		return NULL;
	}
	
	getdat_((int*)pRec, lRecTime);
	
	getphs_((int*)pRec, &nAntA, &nAntB, (double**)phase, (double**)corr, 
			  (double**)freq);
	
	if( (sAntA = _getAntStr(nAntA, 1)) == NULL)
		return NULL;
	if( (sAntB = _getAntStr(nAntB, 1)) == NULL)
		return NULL;
	
	pList = PyList_New(32);         /* New Reference */
	
	for(i = 0; i<32; i++){
		nIncMilli = 1000 * (i/4);
		timadd_(lRecTime, &nIncMilli, lSampTime, &nSubRet);
		
		if(nSubRet != 1){
			Py_DECREF(pList);
			PyErr_SetString(PyExc_ValueError, "Invalid return from libdepwi, timadd_");
			return NULL;
		}
		
		pTup = PyTuple_New(11);                /* New Ref */
		
		/* Set Time */
		PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));
		PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
		
		/* Set Freq */
		PyTuple_SetItem(pTup, 2, PyFloat_FromDouble(freq[i][0]));
			
		for(j = 0; j<4; j++){
			PyTuple_SetItem(pTup, j+3, PyFloat_FromDouble( phase[i][j] ) );
			PyTuple_SetItem(pTup, j+7, PyFloat_FromDouble( corr[i][j] ) );
		}
		
		PyList_SetItem(pList, i, pTup);          /* Ref passed */
	}
	
	pRet = Py_BuildValue("ssO", sAntA, sAntB, pList);
	
	Py_DECREF(pList);                           /* Return tup owns list now */
	return pRet;
}

/*****************************************************************************/
/* getlfc */

const char depwihelp_getlfc[] = 
"Get calibrated LFC data from a DE PWI data record.\n"
"\n"
"Read the LFC data from a DE PWI record, and return calibrated spectral\n"
"densities.  In addition the frequency table for amplitudes is returned\n"
"as well\n"
"\n"
"Arguments:\n"
"  A DE PWI Record\n"
"  A single character string selecting the desired receiver, 'A' or 'B'\n"
"Returns:\n"
"  A string indicating the antenna, one of 'Es', 'Ez', 'Ex', 'H'\n"
"\n"
"  A 16 element list.  Each entry in the list is the 11-tuple:\n"
"    0 Date as a 5-digit integer, so 1981-02-01 is 81032\n"
"    1 The integer millisecond of day for the record\n"
"    2 The frequency value for the channel in Hz\n"
"    3 Amplitude 1\n"
"    4 Amplitude 2\n" 
"    5 Amplitude 3\n"
"    6 Amplitude 4\n"
"    7 Amplitude 5\n"
"    8 Amplitude 6\n"
"    9 Amplitude 7\n"
"   10 Amplitude 8\n"
"\n"
"The low-band of the LFC stays at a constant frequency for the entire record.\n"
"So you end up with 64 values at the same frequency for lowband data.  The \n"
"low band frequency is one of:  1.78 Hz, 3.12 Hz, 5.62 Hz, or 10.0 Hz for the\n"
"entire record.  These are intermixed with high-band data which sweep though\n"
"thier frequency range twice in the returned records.  The high-band freqs \n"
"are 17.8 Hz, 31.2 Hz, 56.2 Hz, and 100. Hz.\n"
"\n"
"The returned amplitudes are in units of V**2 m**-2 Hz**-1 for the electric\n"
"antennas, or units of Gammas**2 Hz**-1 for the magnetic antenna\n"
"\n"
"Note: The values in the data tuples are output in order of the values in the\n"
"  data record.  In many cases the some of the initial values should be \n"
"  dropped as the instrument had not yet settled when the 1st measurement was\n"
"  taken\n";

static PyObject* depwi_getlfc(PyObject* self, PyObject* args)
{
	/* Inputs */
	const char* pRec = NULL;
	int nLen = 0;
	int nFr = -1;
	
	/* Outputs */
	const char* sAnt = NULL;
	int nAnt = -1;
	double dAmpHi[8][8];
   double dAmpLo[8][8];
   double dFreqHi[8];
   double dFreqLo;
	
	PyObject* pList = NULL;
	PyObject* pTup = NULL;
	PyObject* pRet = NULL;
	
	/* Misc */
	int i = 0;
	int nIncMilli = 0;
	int lRecTime[8] = {0};
	int lSampTime[2] = {0};
	int nSubRet = 0;
	int j = 0;

	if( (nFr = _parseAmpArgs("getlfc", args, &pRec, &nLen)) < 0)
		return NULL;
	
	/* Get data via the fortran lib, 1st the record time then the amplitudes */
	getdat_((int*)pRec, lRecTime);
	
	getlfc_((int*)pRec, &nFr, &nAnt, (double**)dAmpHi, (double**)dAmpLo,
			  dFreqHi, &dFreqLo);
	
	pList = PyList_New(16);   /* New Reference */
	
	for(i = 0; i < 8; i++){
		nIncMilli = 1000 * i;
		timadd_(lRecTime, &nIncMilli, lSampTime, &nSubRet);
		
		if(nSubRet != 1){
			Py_DECREF(pList);
			PyErr_SetString(PyExc_ValueError, "Invalid return from libdepwi, timadd_");
			return NULL;
		}
		
		/* Add low band data */
		pTup = PyTuple_New(11);
		
		PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));  /* Time */
		PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
		
		PyTuple_SetItem(pTup, 2, PyFloat_FromDouble(dFreqLo));   /* Freq */
		
		for(j = 0; j < 8; j++)
			PyTuple_SetItem(pTup, j+3, PyFloat_FromDouble( dAmpLo[i][j] ) );
		
		PyList_SetItem(pList, i*2, pTup);
		
		/* Add high band data */
		pTup = PyTuple_New(11);
		
		PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));  /* Time */
		PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
		
		PyTuple_SetItem(pTup, 2, PyFloat_FromDouble(dFreqHi[i]));   /* Freq */
		
		for(j = 0; j < 8; j++)
			PyTuple_SetItem(pTup, j+3, PyFloat_FromDouble( dAmpHi[i][j] ) );
		
		PyList_SetItem(pList, i*2 + 1, pTup);
	}
	
	if( (sAnt = _getAntStr(nAnt, 0)) == NULL){
		Py_DECREF(pList);
		return NULL;
	}
			
	pRet = Py_BuildValue("sO", sAnt, pList);
	
	Py_DECREF(pList);                           /* Return tup owns list now */
	return pRet;
}

/*****************************************************************************/
/* getlfc */

const char depwihelp_getlfc_phase[] = 
"Get calibrated LFC phase data from a DE PWI record\n"
"\n"
"Read the LFC Phase data from a DE PWI record and return calibrated phase\n"
"differences between the sensors attached to the LFC A and LFC B receivers.\n"
"\n"
"Arguments:\n"
"  A DE PWI Record\n"
"Returns:\n"
"  The antenna connected to LFC-A, one of 'Es', 'Ez', 'Ex', 'H'\n"
"\n"
"  The antenna connected to LFC-B, one of 'Es', 'Ez', 'Ex', 'H'\n"
"\n"
"  A 16 element list.  Each entry in the list is the 19-tuple:\n"
"    0 Date as a 5-digit integer, so 1981-02-01 is 81032\n"
"    1 The integer millisecond of day for the record\n"
"    2 The frequency value for the Receivers in Hz\n"
"    3 Phase difference 1, in degrees\n"
"    4 Phase difference 2, in degrees\n"
"    5 Phase difference 3, in degrees\n"
"    6 Phase difference 4, in degrees\n"
"    7 Phase difference 5, in degrees\n"
"    8 Phase difference 6, in degrees\n"
"    9 Phase difference 7, in degrees\n"
"   10 Phase difference 8, in degrees\n"
"   11 Correlation 1\n"
"   12 Correlation 2\n"
"   13 Correlation 3\n"
"   14 Correlation 4\n"
"   15 Correlation 5\n"
"   16 Correlation 6\n"
"   17 Correlation 7\n"
"   18 Correlation 8\n"
"\n"
"Notes:\n"
"\n"
"* A correlation coefficient of -1.0 means the data are invalid!\n"
"\n"
"* The timing between measurements is 1/8 of a second\n"
"\n"
"* Frequency infromation is similar to getlfc.  See the notes for that\n"
"  function\n";

static PyObject* depwi_getlfc_phase(PyObject* self, PyObject* args)
{
	/* Inputs */
	const char* pRec = NULL;
	int nLen = 0;
	
	/* Misc */
	int i = 0, j = 0;
	int nIncMilli = 0;
	int lSampTime[2] = {0};
	int nSubRet = 0;
	
	/* Outputs */
	int lRecTime[8] = {0};
	int nAntA = -1;
	int nAntB = -1;
	const char* sAntA = NULL;
	const char* sAntB = NULL;
	double phaseHi[8][8] = {{0}};
	double phaseLo[8][8] = {{0}};
	double corrHi[8][8] = {{0}};
	double corrLo[8][8] = {{0}};
	double freqHi[8] = {0};
	double freqLo = 0;
	
	PyObject* pList = NULL;
	PyObject* pTup = NULL;
	PyObject* pRet = NULL;
	
	if(!PyArg_ParseTuple(args, "s#:getlfc", &pRec, &nLen))
		return NULL;
	
	if(nLen < 1768){
		PyErr_SetString(PyExc_ValueError, "Input record length is not 1768 bytes");
		return NULL;
	}
	
	getdat_((int*)pRec, lRecTime);
	
	getphl_((int*)pRec, &nAntA, &nAntB, (double**)phaseHi, (double**)phaseLo,
	        (double**)corrHi, (double**)corrLo,  freqHi, &freqLo);
	
	if( (sAntA = _getAntStr(nAntA, 0)) == NULL)
		return NULL;
	
	if( (sAntB = _getAntStr(nAntB, 0)) == NULL)
		return NULL;
	
	pList = PyList_New(16);         /* New Reference */
	
	for(i = 0; i < 8; i++){
		nIncMilli = 1000 * i;
		timadd_(lRecTime, &nIncMilli, lSampTime, &nSubRet);
		
		if(nSubRet != 1){
			Py_DECREF(pList);
			PyErr_SetString(PyExc_ValueError, "Invalid return from libdepwi, timadd_");
			return NULL;
		}
		
		/* Add the low band data */
		pTup = PyTuple_New(19);
		
		PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));  /* Time */
		PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
		PyTuple_SetItem(pTup, 2, PyFloat_FromDouble(freqLo));    /* Freq */
		
		for(j = 0; j < 8; j++){
			PyTuple_SetItem(pTup, j+3, PyFloat_FromDouble( phaseLo[i][j] ) );
			PyTuple_SetItem(pTup, j+11, PyFloat_FromDouble( corrLo[i][j] ) );
		}
		
		PyList_SetItem(pList, i*2, pTup);
		
		/* And the high band data */
		pTup = PyTuple_New(19);
		
		PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));  /* Time */
		PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
		PyTuple_SetItem(pTup, 2, PyFloat_FromDouble(freqHi[i]));   /* Freq */
		
		for(j = 0; j < 8; j++){
			PyTuple_SetItem(pTup, j+3, PyFloat_FromDouble( phaseHi[i][j] ) );
			PyTuple_SetItem(pTup, j+11, PyFloat_FromDouble( corrHi[i][j] ) );
		}
		
		PyList_SetItem(pList, i*2 + 1, pTup);
	}
				
	pRet = Py_BuildValue("ssO", sAntA, sAntB, pList);
	
	Py_DECREF(pList);                           /* Return tup owns list now */
	return pRet;
}

/*****************************************************************************/
/* GetDC */

const char depwihelp_getdc[] = 
"Reads the DC electric field data from the DE PWI record, calibrates it, \n"
"removes the effects of the spacecraft spin.  The method used is taken from\n"
"Dr. Daniel R. Weimer's Master of Science thesis (University of Iowa, 1983).\n"
"\n"
"WARNING: The first call to this function sets up internal memory structures\n"
"   and always returns empty for the first record.  Always feed records to\n"
"   this function in time order and don't try to trick it into outputting\n"
"   data for the first record by sending it twice, that just produces\n"
"   INVALID data, which is worse than no data at all.\n"
"\n"
"Arguments\n"
"\n"
"  A DE PWI Record\n"
"\n"
"Returns\n"
"\n"
"* A 0-2 element list containing spin plane measurements.  Each element of\n"
"  the list is a 6-tuple containing the following items:\n"
"\n"   
"    0) Date as a 5-digit integer, so 1981-02-01 is 81032\n"
"\n"
"    1) The integer millisecond of day for the record\n"
"\n"
"    2) The string \"Ex\" to indicate data was collected from the Ex antenna\n"
"       which lies in the spin plane of the spacecraft\n"
"\n"
"    3) The component of the electric field measured in the spin plane that\n"
"       is perpendicular to the projection of the local magnetic field on\n"
"       the spin plane, expressed in volts/meter.\n"
"\n"
"    4) The component of the electric field measured in the spin plane that\n"
"       is parallel to the projection of the local magnetic field on the spin\n"
"       plane, expressed in volts/meter.\n"
"\n"
"    5) An indication of the quality of the data taken with the Ex antenna.\n"
"\n"
"  Sometimes one or both of the DC spin-plane fields cannot be determined.  In\n"
"  those cases a 1-element or 0-element list is returned\n"
"\n"
"* A 0-2 element list containing spin axis measurements.  Each element of the\n"
"  list is a 5-tuple containing the following items:\n"
"\n"
"    0) Date as a 5-digit integer, so 1981-02-01 is 81032\n"
"\n"
"    1) The integer millisecond of day for the record\n"
"\n"
"    2) The string \"Ez\" to indicate data was collected from the Ez antenna\n"
"       which lies along the spin axis of the spacecraft\n"
"\n"
"    3) The average electric field measured by the Ez antenna, expressed in\n"
"       volts/meter.\n"
"\n"
"    4) The standard deviation of the electric field measured by the Ez\n"
"       antenna.\n"
"\n"
"  Sometimes one or both of the DC spin-axis fields cannot be determined.  In\n"
"  those cases a 1-element or 0-element list is returned.\n";


static PyObject* depwi_getdc(PyObject* self, PyObject* args)
{
	/* Input */
	int nLen = 0;
	const char* pRec = NULL;
	double exper[2] = {0.0, 0.0};
	double expar[2] = {0.0, 0.0};
	double exchi[2] = {0.0, 0.0};
	double ezavg[2] = {0.0, 0.0};
	double ezsdev[2] = {0.0, 0.0};
	
	/* Output */
	int lRecTime[8] = {0};
	int lSampTime[2] = {0};
	
	PyObject* pExList = NULL;
	PyObject* pEzList = NULL;
	PyObject* pTup = NULL;
	PyObject* pRet = NULL;
	
	/* Misc */
	int i = 0;
	int nExRec = 0, iExRec = 0;
	int nEzRec = 0, iEzRec = 0;
	int nIncMilli = 0;
	int nSubRet = 0;
	
	if(!PyArg_ParseTuple(args, "s#:getdc", &pRec, &nLen))
		return NULL;
	
	if(nLen < 1768){
		PyErr_SetString(PyExc_ValueError, "Input record length is not 1768 bytes");
		return NULL;
	}
	
	getdat_((int*)pRec, lRecTime);
	
	getdc_((int*)pRec, exper, expar, exchi, ezavg, ezsdev);
	
	for(i = 0; i<2; i++){
		if(exchi[i] > -1.0) nExRec++;
		if(ezsdev[i] > -1.0) nEzRec++;
	}
	
	nExRec = 2;
	nEzRec = 2;
	
	pExList = PyList_New(nExRec);   /* New Reference */
	pEzList = PyList_New(nEzRec);   /* New Reference */
	
	for(i = 0; i< 2; i++){
		nIncMilli = 4000*i - 1000;
		
		timadd_(lRecTime, &nIncMilli, lSampTime, &nSubRet);
		
		if(nSubRet != 1){
			Py_DECREF(pExList);
			Py_DECREF(pEzList);
			PyErr_SetString(PyExc_ValueError, "Invalid return from libdepwi, timadd_");
			return NULL;
		}

		if(exchi[i] > -1.0){ 
			pTup = PyTuple_New(6);
			PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));  /* Time */
			PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
			PyTuple_SetItem(pTup, 2, PyString_FromString("Ex"));     /* Ant */
			PyTuple_SetItem(pTup, 3, PyFloat_FromDouble(exper[i]));  /* perp */
			PyTuple_SetItem(pTup, 4, PyFloat_FromDouble(expar[i]));  /* || */
			PyTuple_SetItem(pTup, 5, PyFloat_FromDouble(exchi[i]));  /* chi */
			
			PyList_SetItem(pExList, iExRec, pTup);
			iExRec++;
		}
		
		if(ezsdev[i] > -1.0){
			pTup = PyTuple_New(5);
			PyTuple_SetItem(pTup, 0, PyInt_FromLong(lSampTime[0]));  /* Time */
			PyTuple_SetItem(pTup, 1, PyInt_FromLong(lSampTime[1]));
			PyTuple_SetItem(pTup, 2, PyString_FromString("Ez"));     /* Ant */
			PyTuple_SetItem(pTup, 3, PyFloat_FromDouble(ezavg[i]));  /* avg */
			PyTuple_SetItem(pTup, 4, PyFloat_FromDouble(ezsdev[i])); /* sdev */
			
			PyList_SetItem(pEzList, iEzRec, pTup);
			iEzRec++;
		}
	}
	
	pRet = Py_BuildValue("OO", pExList, pEzList);
	
	Py_DECREF(pExList);   /* Return tup owns lists now */
	Py_DECREF(pEzList);
	
	return pRet;
}

/*****************************************************************************/
/* getorb */
const char depwihelp_getorb[] = 
"Return the orbit data from a DE PWI data record.\n"
"\n"
"This subroutine will read the orbit data from the DE PWI record and\n"
"return it in an array of double precision floating point numbers.\n"
"\n"
"Arguments:\n"
"  A DE PWI Record\n"
"Returns:\n"
"  A 38 tuple as follows:\n"
"\n"
"  Index     Contents                                          Units\n"
"  -----     ------------------------------------------------  -----\n"
"    0       Date as a 5-digit integer  YYDDD                  days\n"
"    1       The integer millisecond of day                    millisec\n"
"    2       GEI satellite velocity vector.              v(x)  km/sec\n"
"    3                                                   v(y)  km/sec\n"
"    4                                                   v(z)  km/sec\n"
"    5       Altitude above the spheroid Earth.                km\n"
"    6       Geodetic latitude of subsatellite point.          degrees\n"
"    7       East longitude of the satellite.                  degrees\n"
"    8       Local magnetic time.                              hours\n"
"    9       McIlwain's shell parameter (L).                   Re\n"
"   10       Invariant latitude.                               degrees\n"
"   11       Magnetic field strength.                          gauss\n"
"   12       GEI magnetic field vector.                  B(x)  gauss\n"
"   13                                                   B(y)  gauss\n"
"   14                                                   B(z)  gauss\n"
"   15       Orbit number.\n"
"   16       3-by-3 rotation matrix for the              X(x)\n"
"   17          transformation from spacecraft           X(y)\n"
"   18          coordinates to GEI.                      X(z)\n"
"   19                                                   Y(x)\n"
"   20                                                   Y(y)\n"
"   21                                                   Y(z)\n"
"   22                                                   Z(x)\n"
"   23                                                   Z(y)\n"
"   24                                                   Z(z)\n"
"   25       GEI satellite position vector.              r(x)  km\n"
"   26                                                   r(y)  km\n"
"   27                                                   r(z)  km\n"
"   28       GEI satellite velocity relative to          R(x)  km/sec\n"
"   29          a rotating atmosphere.                   R(y)  km/sec\n"
"   30                                                   R(z)  km/sec\n"
"   31       GEI unit vector toward the Sun.             U(x)  km\n"
"   32                                                   U(y)  km\n"
"   33                                                   U(z)  km\n"
"   34       Phase angle of spin measured from the             radians\n"
"               velocity vector to the x-axis of\n"
"               the spacecraft.\n"
"   35       Sunlight/Darkness: 0 = Darkness, 1 = Sunlight.\n"
"   36       Geocentric radial distance (Derived)              Re\n"
"   37       Magnetic Latitude (Derived)                       degrees\n"
"\n";

static PyObject* depwi_getorb(PyObject* self, PyObject* args)
{
	int nLen = 0;                      /* Input */
	const char* pRec = NULL;
	
	int lRecTime[8] = {0};             /* Output */
	double orbit[37] = {0};
	PyObject* pTup = NULL;
	
	int i = 0;                         /* Misc */
	
	if(!PyArg_ParseTuple(args, "s#:getdc", &pRec, &nLen))
		return NULL;
	
	if(nLen < 1768){
		PyErr_SetString(PyExc_ValueError, "Input record length is not 1768 bytes");
		return NULL;
	}
	
	getdat_((int*)pRec, lRecTime);
	
	pTup = PyTuple_New(38);         /* New ref */
	
	PyTuple_SetItem(pTup, 0, PyInt_FromLong(lRecTime[0]));
	PyTuple_SetItem(pTup, 1, PyInt_FromLong(lRecTime[1]));
	
	getorb_((int*)pRec, orbit + 2);
	
	for(i=2; i<38; i++)
		PyTuple_SetItem(pTup, i, PyFloat_FromDouble(orbit[i]));
	
	return pTup;                    /* Give the ref away */
}

/*****************************************************************************/
/* Status */

const char depwihelp_getstatus [] = 
"Return the status information from a DE PWI record\n"
"\n"
"Arguments\n"
"  A DE PWI Record\n"
"\n"
"Returns\n"
"  A 29-tuple with the following status information.  Most of the return\n"
"  values are integers.\n"
"\n"
"   0 Date as a 5-digit integer YYDDD\n"
"\n"
"   1 The integer millisecond of day\n"
"\n"
"   2 LFC Low Band Channel:\n"
"      0 - 1.78 Hz, 1 - 3.12 Hz, 2 - 6.25 Hz, 3 - 10.0 Hz\n"
"\n"
"   3 SFR Skip Mode: 1 - Skip 1, 8 - Skip 8\n"
"\n"
"   4 LFC B Antenna: 'Es', 'Ez', 'Ex', 'H'\n"
"\n"
"   5 LFC A Antenna: 'Es', 'Ez', 'Ex', 'H'\n"
"\n"
"   6 SFR B Antenna: 'Es', 'Ez', 'Ex', 'B'\n"
"\n"
"   7 SFR A Antenna: 'Es', 'Ez', 'Ex', 'B'\n"
"\n"
"   8 SFR Gain Select:\n"
"     0 or 2 - Low gain (30 dB attenuator)\n"
"     1 - High gain\n"
"     3 - Toggle (switch every 0.5 seconds)\n"
"\n"
"   9 CMD 22 Cal Inhibit:  0 - Inhibit, 1 - On\n"
"\n"
"  10 SC 17 Cal Enable:  0 - Disable, 1 - Enable\n"
"\n"
"  11 SFR Antenna Select: 0 - Manual, 1 - Automatic\n"
"\n"
"  12 LFC Antenna Select: 0 - Manual, 1 - Automatic\n"
"\n"
"  13 LFC Low Band Sweep: 0 - Sweep, 1 - Lock\n"
"\n"
"  14 LFC High Band Sweep: 0 - Sweep, 1 - Lock\n"
"\n"
"  15 SFR Sweep: 0 - Sweep, 1 - Lock\n"
"\n"
"  16 SFR x4 Rate @ T = 0: 1 (x1), 4 (x4)\n"
"\n"
"  17 LFC High Band Channel @ T = 0:\n"
"     0 - 17.8 Hz, 1 - 31.2 Hz, 2 - 62.5 Hz, 3 - 100 Hz\n"
"\n"
"  18 SFR Step @ T = 0: n (0 to 31) SFR frequency step\n"
"\n"
"  19 SFR x4 Rate @ T = 2: 1 (x1), 4 (x4)\n"
"\n"
"  20 LFC High Band Channel @ T = 2:\n"
"     0 - 17.8 Hz, 1 - 31.2 Hz, 2 - 62.5 Hz, 3 - 100 Hz\n"
"\n"
"  21 SFR Step @ T = 2: n (0 to 31) SFR frequency step\n"
"\n"
"  22 SFR x4 Rate @ T = 4: 1 (x1), 4 (x4)\n"
"\n"
"  23 LFC High Band Channel @ T = 4:\n"
"     0 - 17.8 Hz, 1 - 31.2 Hz, 2 - 62.5 Hz, 3 - 100 Hz\n"
"\n"
"  24 SFR Step @ T = 4: n (0 to 31) SFR frequency step\n"
"\n"
"  25 SFR x4 Rate @ T = 6: 1 (x1), 4 (x4)\n"
"\n"
"  26 LFC High Band Channel (T = 6)\n"
"     0 - 17.8 Hz, 1 - 31.2 Hz, 2 - 62.5 Hz, 3 - 100 Hz\n"
"\n"
"  27 SFR Step @ T = 6: n (0 to 31) SFR frequency step\n"
"\n"
"  28 Wideband Data: 0 - No wideband data, 1 wideband transmitter on\n";

static PyObject* depwi_getstatus(PyObject* self, PyObject* args)
{
	int i, nLen = 0;
	const char* pRec = NULL;
	int lRecTime[8] = {0};
	int status[27] = {0};
	PyObject* pTup = NULL;
	
	int nTmp = 0;
	const char* sAnt = NULL;
	
	if(!PyArg_ParseTuple(args, "s#:getdc", &pRec, &nLen))
		return NULL;
	
	if(nLen < 1768){
		PyErr_SetString(PyExc_ValueError, "Input record length is not 1768 bytes");
		return NULL;
	}
	
	getdat_((int*)pRec, lRecTime);
	
	pTup = PyTuple_New(29);         /* New ref */
	
	PyTuple_SetItem(pTup, 0, PyInt_FromLong(lRecTime[0]));
	PyTuple_SetItem(pTup, 1, PyInt_FromLong(lRecTime[1]));
	
	getsts_((int*)pRec, status);
	
	for(i = 0; i< 27; i++){
		switch(i){
		case 1:                                           /* Skip Mode */
			nTmp = (status[i] == 0) ? 1 : 8 ;
			PyTuple_SetItem(pTup, 2+i, PyInt_FromLong(nTmp));
			break;
			
		case 2:                                           /* LFC Ant */
		case 3:
			sAnt = _getAntStr(status[i], 0);
			PyTuple_SetItem(pTup, 2+i, PyString_FromString(sAnt));
			break;
			
		case 4:
		case 5:                                           /* SFR Ant */
			sAnt = _getAntStr(status[i], 1);
			PyTuple_SetItem(pTup, 2+i, PyString_FromString(sAnt));
			break;
			
		case 14:                                          /* x4 Rate */
		case 17:
		case 20:
		case 23:
			nTmp = (status[i] == 0) ? 1 : 4;
			PyTuple_SetItem(pTup, 2+i, PyInt_FromLong(nTmp));
			break;
			
		default:			
			PyTuple_SetItem(pTup, 2+i, PyInt_FromLong(status[i]));
		}
	}
	
	return pTup;
}

/*****************************************************************************/
/* The method defintions */

static PyMethodDef depwi_methods[] = {
	{"getrec",       depwi_getrec,       METH_VARARGS, depwihelp_getrec       },
	{"gettime",      depwi_gettime,      METH_VARARGS, depwihelp_gettime      },
	{"timecmp",      depwi_timecmp,      METH_VARARGS, depwihelp_timecmp      },
	{"getsfr",       depwi_getsfr,       METH_VARARGS, depwihelp_getsfr       },
	{"getsfr_phase", depwi_getsfr_phase, METH_VARARGS, depwihelp_getsfr_phase },
	{"getlfc",       depwi_getlfc,       METH_VARARGS, depwihelp_getlfc       },
	{"getlfc_phase", depwi_getlfc_phase, METH_VARARGS, depwihelp_getlfc_phase },
	{"getdc",        depwi_getdc,        METH_VARARGS, depwihelp_getdc        },
	{"getorb",       depwi_getorb,       METH_VARARGS, depwihelp_getorb       },
	{"getstatus",    depwi_getstatus,    METH_VARARGS, depwihelp_getstatus    },
	{NULL, NULL, 0, NULL}
};
/*****************************************************************************/
/* Module initialization */

PyMODINIT_FUNC initdepwi(void){
	Py_InitModule3("depwi", depwi_methods, depwi_doc);
}
