pro process_satm, satm_file, yrday_start, msec_start, msec_end, $
                    hlapi, ea, gm, status, debug=debug, useExtMAGA=useExtMAGA
; version = '4.2' ;set version number
version = '4.3'

;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
; This procedure will:
;   1) break the SATM data into sweeps and steps
;   2) separate detectors into electron and ion
;   3) sort sensors to be in order of geometrical location
;   4) interpolate OA data for one point per sweep
;   5) return data in the HLAPI structure
;   6) if debug is set, prints out raw counts in major frame
;
; This procedure is designed to be called from:  REDUCE_SATM
;
; Written by:
; Norman T. Reese, ST Systems Corp., 30 Sep 91
; modified by Josephine N. Humphreys, Hughes STX, 3 Sept 92, for HAPI  
; added longs on increment counters; Bobby Candey 1995 Jan 8
; updated Robert.M.Candey@gsfc.nasa.gov; 1995 Feb 1
; added efficiency speedups; 1995 Feb 16, BC
; added useExtMAGA for using external MAG-A data, 1995 April 4, BC
; changed sweeps_per_major_frame and nsteps to round up, 1999 Aug 12, BC
; test with fewer sweeps thrown out which may be too generous, 1999 Aug 12, BC
; check for start, stop, skip the same throughout, 1999 Aug 12, BC
; 1999 Aug 12: version 4.2
; added minimum on nsweeps when reducing arrays at end, 1999 Sept 1, BC
; 2020Mar6 RMC add brackets
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

; if not keyword_set(debug) then debug = 0

;First read in the SATM file
;###
t=systime(1)
 read_satm, satm_file, yrday_start, msec_start, msec_end, $
              satm, satID, status
print,'Read_satm, seconds', systime(1)-t

;check status variable for error in read_satm
 if (strmid(status,0,2) eq "**") then begin
    print,"** ERROR in READ_SATM:  ",status
    return
 endif
 full_filename = status

;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

     nsensors         = max(satm.num_sensors)
     nframes          = long(n_elements (satm))
     count_size       = n_elements (satm[0].counts)
     steps_per_frame  = count_size / nsensors
     time_per_step    = 8000.0 / steps_per_frame
     steps_per_second = 1000.0 / time_per_step
     print,           'Steps per Second:  ', steps_per_second

;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+

;calculate dimensions for separating count data into sweeps
     pps_size               = n_elements(satm[0].pps)
if satId eq "L" then begin ; LAPI
     sto_pps1               = max(satm.stop_pps1)
;test     sta_pps1               = max(satm.start_pps1)
;test     ski_pps1               = max(satm.skip_pps1)
     sta_pps1               = min(satm.start_pps1) > 1
     ski_pps1               = min(satm.skip_pps1) > 0
;rmc     nsteps                 = (sto_pps1 - sta_pps1) /(ski_pps1 +1) + 1
     nsteps   = long(float(sto_pps1 - sta_pps1) /(ski_pps1 + 1) + 1.9999)

;check 1999Aug12
  if (((mintemp1=(min(satm.start_pps1, max=maxtemp1) > 1))) ne maxtemp1) then $
     print, '#### Start_pps1 min and max different: ', mintemp1, maxtemp1
  if (((mintemp2=(min(satm.start_pps2, max=maxtemp2) > 1))) ne maxtemp2) then $
     print, '#### Start_pps2 min and max different: ', mintemp2, maxtemp2
  if (mintemp1 ne mintemp2) then $
     print, '#### Start_pps1 min and Start_pps2 min different'
  if (((mintemp1=(min(satm.stop_pps1, max=maxtemp1) > 1))) ne maxtemp1) then $
     print, '#### Stop_pps1 min and max different: ', mintemp1, maxtemp1
  if (((mintemp2=(min(satm.stop_pps2, max=maxtemp2) > 1))) ne maxtemp2) then $
     print, '#### Stop_pps2 min and max different: ', mintemp2, maxtemp2
  if (maxtemp1 ne maxtemp2) then $
     print, '#### Stop_pps1 max and Stop_pps2 max different'
  if (((mintemp1=(min(satm.skip_pps1, max=maxtemp1) > 0))) ne maxtemp1) then $
     print, '#### Skip_pps1 min and max different: ', mintemp1, maxtemp1
  if (((mintemp2=(min(satm.skip_pps2, max=maxtemp2) > 0))) ne maxtemp2) then $
     print, '#### Skip_pps2 min and max different: ', mintemp2, maxtemp2
  if (mintemp1 ne mintemp2) then $
     print, '#### Skip_pps1 min and Skip_pps2 min different'
  if (((mintemp1=(min(satm.step_pps1, max=maxtemp1) > 1))) ne maxtemp1) then $
     print, '#### Step_pps1 min and max different: ', mintemp1, maxtemp1
  if (((mintemp2=(min(satm.step_pps2, max=maxtemp2) > 1))) ne maxtemp2) then $
     print, '#### Step_pps2 min and max different: ', mintemp2, maxtemp2
  if (mintemp1 ne mintemp2) then $
     print, '#### Step_pps1 min and Step_pps2 min different'

  if keyword_set(debug) then begin
    print, 'sta_pps1 min, max: ', min(satm.start_pps1), max(satm.start_pps1)
    print, 'sto_pps1 min, max: ', min(satm.stop_pps1), max(satm.stop_pps1)
    print, 'ski_pps1 min, max: ', min(satm.skip_pps1), max(satm.skip_pps1)
    print, 'step_pps1 min, max: ', min(satm.step_pps1), max(satm.step_pps1)
    print, 'sta_pps2 min, max: ', min(satm.start_pps2), max(satm.start_pps2)
    print, 'sto_pps2 min, max: ', min(satm.stop_pps2), max(satm.stop_pps2)
    print, 'ski_pps2 min, max: ', min(satm.skip_pps2), max(satm.skip_pps2)
    print, 'step_pps2 min, max: ', min(satm.step_pps2), max(satm.step_pps2)
    print, 'nSteps: ', (sto_pps1 - sta_pps1) /(ski_pps1 +1) + 1, nsteps
  endif ; debug
     if nsteps ne max(satm.step_pps1) then begin
        print, "WARNING: step and skip values don't agree"
         print,format='(a,8i5)','PPS(start,stop,skip,step)(1,2)', $
              satm[0].start_pps1,satm[0].stop_pps1,              $
              satm[0].skip_pps1,satm[0].step_pps1,               $
              satm[0].start_pps2,satm[0].stop_pps2,              $
              satm[0].skip_pps2,satm[0].step_pps2
	 print,nsteps,max(satm.step_pps1)
     endif                
;rmc     sweeps_per_major_frame = pps_size /2 /nsteps
     sweeps_per_major_frame = long(float(pps_size) /2. /nsteps + 0.9999)
endif else begin ; HAPI
     sto_pps               = max(satm.stop_pps)
;test     sta_pps               = max(satm.start_pps)
;test     ski_pps               = max(satm.skip_pps)
     sta_pps               = min(satm.start_pps) > 1
     ski_pps               = min(satm.skip_pps) > 0

; check 1999 Aug 12
  if (((mintemp=(min(satm.start_pps, max=maxtemp) > 1))) ne maxtemp) then $
     print, '#### Start_pps min and max different: ', mintemp, maxtemp
  if (((mintemp=(min(satm.stop_pps, max=maxtemp) > 1))) ne maxtemp) then $
     print, '#### Stop_pps min and max different: ', mintemp, maxtemp
  if (((mintemp=(min(satm.skip_pps, max=maxtemp) > 0))) ne maxtemp) then $
     print, '#### Skip_pps min and max different: ', mintemp, maxtemp
  if (((mintemp=(min(satm.step_pps, max=maxtemp) > 1))) ne maxtemp) then $
     print, '#### Step_pps min and max different: ', mintemp, maxtemp

;rmc     nsteps                 = (sto_pps - sta_pps) /(ski_pps +1) + 1
     nsteps   = long(float(sto_pps - sta_pps) /(ski_pps + 1) + 1.9999)
     samples_per_step      = 64./max(satm.step_pps)
     nsteps                = nsteps * samples_per_step
;rmc     sweeps_per_major_frame = pps_size /nsteps
     sweeps_per_major_frame = long(float(pps_size) /nsteps + 0.9999)
endelse
     num_of_sweeps          = sweeps_per_major_frame * (nframes + 1)
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;
;DEBUG MODE (raw dump of major frames)
;
 if keyword_set(debug) then begin
    for i=0L, nframes-1 do begin
        print,'***** N E W   M A J O R    F R A M E *****'
        print,'DATE(yrday), TIME(ms):  ',satm[i].date,satm[i].time
        print,'OA (ilat, mlt, alt, lat, lon, lst, lshell, orbit,',$
                   'gei_speed, sza, dark_light)'
        print,format='(10f7.2,i2)',                              $
              satm[i].ilat,satm[i].mlt,satm[i].alt,              $
              satm[i].lat,satm[i].lon,satm[i].lst,satm[i].lshell, $
              satm[i].orbit,satm[i].gei_speed,satm[i].sza,       $
              satm[i].dark_light
        if satID eq "L" then begin
         print,format='(a,8i5)','PPS(start,stop,skip,step)(1,2)', $
              satm[i].start_pps1,satm[i].stop_pps1,              $
              satm[i].skip_pps1,satm[i].step_pps1,               $
              satm[i].start_pps2,satm[i].stop_pps2,              $
              satm[i].skip_pps2,satm[i].step_pps2                
         print,'SEA:  ',satm[i].sea
         print,'GM (0 deg, 90 deg) MAG (Bx, By, Bz)'
         for k=0,7 do print,format='(2i10,3f10.2)',               $
                     satm[i].gm_data[*,k],satm[i].mag_data[*,k]
         print,format='(20i4)',satm[i].sensor_id[0:nsensors-1]
         for j=0L, steps_per_frame-1 do                            $
            print,format='(32i4)',                               $
                  satm[i].counts[j*nsensors:(j+1)*nsensors-1],   $
                  satm[i].pps[j*2:j*2+1]
        endif else begin
         print,format='(a,4i5)' ,'PPS(start,stop,skip,step)', $
              satm[i].start_pps,satm[i].stop_pps,              $
              satm[i].skip_pps,satm[i].step_pps               
;         for k=0,127 do print,format='(2f10.2)',               $
;                     satm[i].mag_data[*,k]
         print,format='(8i4)',satm[i].sensor_id[0:nsensors-1]
         for j=0L, steps_per_frame-1 do                            $
            print,format='(10i4)',                               $
                  satm[i].counts[j*nsensors:(j+1)*nsensors-1],   $
                  satm[i].pps[j]
        endelse
   endfor ;i
 endif     ;debug
;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;

;xx sw_frame = intarr(num_of_sweeps * 2)
;xx sw_line  = intarr(num_of_sweeps * 2)
 counts  = bytarr(nsensors,nsteps,num_of_sweeps)
 sensors = bytarr(nsensors,num_of_sweeps)
 if satID eq "L" then $
  pps     = bytarr(2,nsteps,num_of_sweeps) else $
  pps     = bytarr(1,nsteps,num_of_sweeps) 
 swtime  = lonarr(num_of_sweeps)

;***set initial variables
 istep             = -1
 isweep            = 0
 swtime(isweep)    = satm[0].time
 sensors(*,isweep) = satm[0].sensor_id[0:nsensors-1] 

;####
t=systime(1)
if satID eq "L" then begin
 for i=0L, nframes-1 do for j=0L, steps_per_frame-1 do begin
;     print,format="(4h####,20i4)",satm[i].counts[j*nsensors:(j+1)*nsensors-1],$
;           satm[i].pps[j*2:j*2+1]
     if ( ( (satm[i].counts[j*nsensors] eq 1) and     $
          (satm[i].counts[j*nsensors+1] eq 1) and     $
          (satm[i].pps[j*2] gt 63)  )   or            $
          (istep ge (nsteps-1) ) ) then begin

          if (istep ne nsteps-1) then begin
;test          if (istep gt (nsteps-1)) then begin
;            *** throw out sweep
             print,"===> THROWING OUT SWEEP, NOT ENOUGH STEPS"
             print,swtime[isweep],ms2hhmmss(swtime[isweep])
             print,"next sweep at major frame sweep line after this:  "
             print, "      major frame:  ",i,",  frame step:  ",j
             print,"number of steps in this bad sweep is:  ",istep
             print,format="(20i4)",sensors[*,isweep]
             for k=0L,istep do $
                 print,format="(20i4)",counts[*,k,isweep],pps[*,k,isweep]
             print,"xxxxxx"
          endif else begin
            isweep = isweep + 1
          endelse
          istep = -1
          swtime[isweep]   = satm[i].time + (j+1) * time_per_step
          sensors[0,isweep] = satm[i].sensor_id[0:nsensors-1] 
;          print,"====STARTING NEW SWEEP:  ",MS2HHMMSS(SWTIME[ISWEEP])
     endif else begin
        istep = istep + 1
;sp        for isensor=0L,nsensors-1 do $
;sp            counts(isensor,istep,isweep) = satm[i].counts(isensor+j*nsensors)
;sp        for k=0,1 do $
;sp            pps(k,istep,isweep) = satm[i].pps(k+j*2)  
        counts[0,istep,isweep] = satm[i].counts[j*nsensors:(j+1L)*nsensors-1L]
        pps[0,istep,isweep] = satm[i].pps[j*2L:j*2L+1L]
;            print,i,j,satm[i].pps[j]
     endelse
 endfor ; i, j

endif else begin ; HAPI

 for i=0L, nframes-1 do for j=0L, steps_per_frame-1 do begin
;     print,format="(4h####,20i4)",satm[i].counts(j*nsensors:(j+1)*nsensors-1),$
;           satm[i].pps(j*2:j*2+1)
     if ( ( (satm[i].counts[j*nsensors] eq 1) and     $
          (satm[i].counts[j*nsensors+1] eq 1) and     $
          (satm[i].pps[j] gt 63)  )   or            $
          (istep ge (nsteps-1) ) ) then begin

          if (istep ne (nsteps-1)) then begin
;test          if (istep gt (nsteps-1)) then begin
;            *** throw out sweep
             print,"===> THROWING OUT SWEEP, NOT ENOUGH STEPS"
             print,swtime[isweep],ms2hhmmss(swtime[isweep])
             print,"next sweep at major frame sweep line after this:  "
             print, "      major frame:  ",i,",  frame step:  ",j
             print,"number of steps in this bad sweep is:  ",istep
             print,format="(20i4)",sensors[*,isweep]
             for k=0L,istep do $
                 print,format="(20i4)",counts[*,k,isweep],pps[*,k,isweep]
             print,"xxxxxx"
          endif else begin
            isweep = isweep + 1
          endelse
          istep = -1
          swtime[isweep]   = satm[i].time + (j+1) * time_per_step
          sensors[0,isweep] = satm[i].sensor_id[0:nsensors-1] 
;          print,"====STARTING NEW SWEEP:  ",MS2HHMMSS(SWTIME(ISWEEP))
     endif else begin
        istep = istep + 1
;sp        for isensor=0L,nsensors-1 do $
;sp            counts(isensor,istep,isweep) = satm[i].counts(isensor+j*nsensors)
        pps[0,istep,isweep] = satm[i].pps[j]
        counts[0,istep,isweep] = satm[i].counts[j*nsensors:(j+1L)*nsensors-1L]
;              print,i,j,satm[i].pps[j]
     endelse
 endfor ; i, j

endelse ; HAPI
;####
print,'Move to Counts/PA arrays, seconds', systime(1)-t

; check to make sure last sweep is a complete sweep:
; if (istep ne nsteps-1) then nsweeps = isweep - 1 else nsweeps = isweep
 if (istep ne nsteps-1) then nsweeps=(isweep-1) > 1 else nsweeps=isweep
 counts    = counts[*,*,0:nsweeps-1]
 pps       = pps[*,*,0:nsweeps-1]
 swtime    = swtime[0:nsweeps-1]
 sensors   = sensors[*,0:nsweeps-1]

;FIXIT:  must use 'smart routine to determine sensors(*,0) is real and not fill
;(kdh) If #5 record doesn't exist, use last record:
if nframes gt 5 then check_record=5 else check_record=nframes-1
print,"***sensor ids for #0,nth record", check_record
print,satm[0].sensor_id[0:nsensors-1]
print,satm[check_record].sensor_id[0:nsensors-1]
;
;fix:  ignore any sensor ids greater than or equal to 30!
if satId eq "L" then $
;  wgood_sensors = where (satm(5).sensor_id(0:nsensors-1) lt 30, wc)  else $
;  wgood_sensors = where (satm(5).sensor_id(0:nsensors-1) lt 10, wc) 
  wgood_sensors=where(satm[check_record].sensor_id[0:nsensors-1] lt 30,wc) else $
  wgood_sensors=where(satm[check_record].sensor_id[0:nsensors-1] lt 10,wc) 
  if (wc gt 0) then begin
;    good_sensors = satm(5).sensor_id(wgood_sensors)
    good_sensors = satm[check_record].sensor_id[wgood_sensors]
    nsensors_elec = n_elements ( where ((good_sensors mod 2) eq 0 ) )
    nsensors_ions = n_elements ( where ((good_sensors mod 2) eq 1 ) )
  endif else begin
    message, 'Error: No valid sensors found'
  endelse
print,"NSENSORS_ELEC,NSENSORS_IONS",nsensors_elec,nsensors_ions,nsensors
;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;
;Now set up data structures
;
;*** must use idl 'execute' command in order to create the 'EA' structure
;      because a different structure name is needed for each variation of:
;      (nsteps, nsensors_elec, nsensors_ions).  Structure name will be 
;      formed as follows:  EA_ns_ne_ni.
;
 nse     = string(nsensors_elec,format='(i2.2)')
 nsi     = string(nsensors_ions,format='(i2.2)')
 nst     = string(nsteps,format='(i2.2)')
 EA_NAME = "EA"+ "_" + nst + "_" + nse + "_" + nsi
 EA_CMD1 = "time:0l, ilat:0.0, mlt:0.0, alt:0.0, lat:0.0, lon:0.0, " +      $
           "lst:0.0, lshell:0.0, GEI_speed:0.0, sza:0.0, dark_light:0b, " 
 EA_CMD2 = "sensors_elec:bytarr(" + nse + "), PA_elec:fltarr(" + nse +      $
           "), counts_elec:bytarr(" + nse + "," + nst  +                    $
           "), sensors_ions:bytarr(" + nsi + "), PA_ions:fltarr(" +         $
           nsi + "), counts_ions:bytarr(" + nsi + "," + nst + ")"

 EA_CMD3 = "sensors_elec:bytarr(" + nse + "), PA_elec:fltarr(" + nse +      $
           "," + nst + "), counts_elec:bytarr(" + nse + "," + nst  +        $
           "), sensors_ions:bytarr(" + nsi + "), PA_ions:fltarr(" +         $
           nsi + "," + nst + "), counts_ions:bytarr(" + nsi + "," + nst + ")"
if satID eq "L" then begin
 ea_cmd  = "EA_STR = { " + EA_NAME + "," + EA_CMD1 + EA_CMD2 + "}"  
 status  =  execute(ea_cmd)
 ea = replicate ( ea_str, nsweeps )
endif else begin
 ea_cmd_h  = "EA_STR2 = { " + EA_NAME + "," + EA_CMD1 + EA_CMD3 + "}"  
 status  =  execute(ea_cmd_h)
 ea = replicate ( ea_str2, nsweeps )
endelse

; ea = $
; {EA_REC, time:0l, ilat:0.0, mlt:0.0, alt:0.0, lat:0.0, lon:0.0, lst:0.0,    $
;          gmLat:0.0, GEI_speed:0.0, sza:0.0, dark_light:0b,                  $
;          sensors_elec:bytarr(nsensors_elec), PA_elec:fltarr(nsensors_elec), $
;          counts_elec:bytarr(nsensors_elec,nsteps),                          $
;          sensors_ions:bytarr(nsensors_ions), PA_ions:fltarr(nsensors_ions), $
;          counts_ions:bytarr(nsensors_ions,nsteps) }
if satID eq "L" then begin
 gm_str = {GM_REC, time:0l, gm_0:0.0, gm_90:0.0 }
 gm = replicate ( gm_str, 8*nframes )
 HLAPI_CMD1 = 'yrday:0l, satid:"", satm_file:"", version:"", nsensors:0,' +   $
              'orbit:0.0, nsweeps:0, nsteps:0, steps_per_second:0,' +         $
              'nsensors_elec:0, nsensors_ions:0, pps:bytarr(' + nst + '),' +  $
              'start_pps1:0b, stop_pps1:0b, skip_pps1:0b, step_pps1:0b,' +    $
              'start_pps2:0b, stop_pps2:0b, skip_pps2:0b, step_pps2:0b'
endif else begin
 HLAPI_CMD1 = 'yrday:0l, satid:"", satm_file:"", version:"", nsensors:0,' +   $
              'orbit:0.0, nsweeps:0, nsteps:0, steps_per_second:0,' +         $
              'nsensors_elec:0, nsensors_ions:0, pps:bytarr(' + nst + '),' +  $
              'start_pps:0b, stop_pps:0b, skip_pps:0b, step_pps:0b'
endelse
 HLAPI_NAME = "HLAPI_"+nst
 HLAPI_CMD  = 'hlapi = { ' + hlapi_name + ',' + hlapi_cmd1 + '}'
 status     = execute(hlapi_cmd)
; hlapi = $
; {HLAPI_REC, yrday:0l, satid:"", satm_file:"", version:"", nsensors:0,       $
;             orbit:0.0, nsweeps:0, nsteps:0, steps_per_second:0,             $
;             nsensors_elec:0, nsensors_ions:0, pps:bytarr(nsteps),           $
;             start_pps1:0b, stop_pps1:0b, skip_pps1:0b, step_pps1:0b,        $
;             start_pps2:0b, stop_pps2:0b, skip_pps2:0b, step_pps2:0b }
;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;
     hlapi.yrday            = satm[0].date
     hlapi.satid            = satid
     hlapi.satm_file        = full_filename
     hlapi.version          = version
     hlapi.nsteps           = nsteps
     hlapi.nsweeps          = nsweeps
     hlapi.nsensors         = nsensors
     hlapi.nsensors_elec    = nsensors_elec
     hlapi.nsensors_ions    = nsensors_ions
     hlapi.steps_per_second = steps_per_second
if satID eq "L" then begin
     hlapi.start_pps1       = satm[0].start_pps1
     hlapi.stop_pps1        = satm[0].stop_pps1
     hlapi.skip_pps1        = satm[0].skip_pps1
     hlapi.step_pps1        = satm[0].step_pps1
     hlapi.start_pps2       = satm[0].start_pps2
     hlapi.stop_pps2        = satm[0].stop_pps2
     hlapi.skip_pps2        = satm[0].skip_pps2
     hlapi.step_pps2        = satm[0].step_pps2
endif else begin
     hlapi.start_pps       = satm[0].start_pps
     hlapi.stop_pps        = satm[0].stop_pps
     hlapi.skip_pps        = satm[0].skip_pps
     hlapi.step_pps        = satm[0].step_pps
endelse
     hlapi.orbit            = satm[0].orbit
     hlapi.pps              = pps[0,*,0] 

;sp     for i=0L,nsweeps-1 do ea(i).time = swtime(i)
;    ea.time = swtime
    ea.time[0] = swtime


; help,counts
; help,pps

;if pps for any sweeps does not agree with expected array, goto,start_debug
goto,done_debug
start_debug:
;FIXIT: debug
;debug++++++++++++++++++++
 help

 print,'cnts: (blocked in original order)'
 w=indgen(nsensors)
 for k=0L, nframes-1 do for j=0L,steps_per_frame-1 do $
     print,'$(34i4)',satm[k].counts[w+j*nsensors],888, $
                     satm[k].pps[j*2],satm[k].pps[j*2+1]

 print,'counts:  (blocked in sensor/step/sweep order)'
 for k=0L,nsweeps-1 do for j=0L,nsteps-1 do $
      print,'$(34i4)',counts[*,j,k],888,pps[*,j,k]

;debug++++++++++++++++++++
done_debug:

;
;clear unnecessary arrays!!
     cnts=0
     ppsval=0
;####
t =systime(1)
;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;
;interpolate the OA values; #### may be some inaccuracy near time breaks
; 
interpIndices, satm.time, ea.time, nearest, nextNearest
ea.ilat    = interpValues(satm.time,satm.ilat,ea.time,nearest,nextNearest,0.)
ea.mlt     = interpValues(satm.time,satm.mlt, ea.time,nearest,nextNearest,24.)
ea.alt     = interpValues(satm.time,satm.alt, ea.time,nearest,nextNearest,0.)
ea.lat     = interpValues(satm.time,satm.lat, ea.time,nearest,nextNearest,0.)
ea.lon     = interpValues(satm.time,satm.lon, ea.time,nearest,nextNearest,360.)
ea.lst     = interpValues(satm.time,satm.lst, ea.time,nearest,nextNearest,24.)
ea.lshell  = interpValues(satm.time,satm.lshell,ea.time,nearest,nextNearest,0.)
ea.GEI_speed=interpValues(satm.time,satm.GEI_speed,ea.time,$
	nearest,nextNearest,0.)
ea.sza     = interpValues(satm.time,satm.sza, ea.time,nearest,nextNearest,0.)
; ea.ilat     = hlapi_interp( satm.time, satm.ilat,       ea.time, 0.  )
; ea.mlt      = hlapi_interp( satm.time, satm.mlt,        ea.time, 24. )
; ea.alt      = hlapi_interp( satm.time, satm.alt,        ea.time, 0.  )
; ea.lat      = hlapi_interp( satm.time, satm.lat,        ea.time, 0.  )
; ea.lon      = hlapi_interp( satm.time, satm.lon,        ea.time, 360.)
; ea.lst      = hlapi_interp( satm.time, satm.lst,        ea.time, 24. )
; ea.lshell    = hlapi_interp( satm.time, satm.lshell,      ea.time, 0.  )
; ea.GEI_speed= hlapi_interp( satm.time, satm.GEI_speed,  ea.time, 0.  )
; ea.sza      = hlapi_interp( satm.time, satm.sza,        ea.time, 0.  )
;####
print,'First hlapi_interp, seconds', systime(1)-t
t=systime(1)

if satID eq "L" then begin
;  there are 8 sets of mag data (and GM data) per frame (8 seconds, 8000 msec)
;sp  for i=0L, nframes-1 do for j=0, 7 do gm(i*8+j).time = satm[i].time + j*1000
  increments = lindgen(8)*1000L
  stime = satm.time
  for i=0L, nframes-1 do gm[i*8L:i*8L+7L].time = stime[i] + increments
;  FIXIT:  define bx,by,bz , also do not use satm.time, but deviation of it    
  interpIndices, gm.time, ea.time, nearest, nextNearest
  Bx = interpValues(gm.time,satm.mag_data[0,*],ea.time,nearest,nextNearest,0.)
  By = interpValues(gm.time,satm.mag_data[1,*],ea.time,nearest,nextNearest,0.)
  Bz = interpValues(gm.time,satm.mag_data[2,*],ea.time,nearest,nextNearest,0.)
;   Bx  = hlapi_interp( gm.time, satm.mag_data(0,*),  ea.time, 0. )
;   By  = hlapi_interp( gm.time, satm.mag_data(1,*),  ea.time, 0. )
;   Bz  = hlapi_interp( gm.time, satm.mag_data(2,*),  ea.time, 0. )

;sp   for i=0L, nframes-1 do for j=0,7 do begin
;sp     gm(i*8+j).GM_0    = satm[i].GM_data( 0, j )
;sp     gm(i*8+j).GM_90   = satm[i].GM_data( 1, j )
;sp   endfor
;sp might be able to do gm.gm_0 = (transpose(satm.gm_data(0,*)))(*)
;sp   for i=0L, nframes-1 do begin
;sp     gm(i*8L:i*8L+7L).GM_0    = satm[i].GM_data( 0, * )
;sp     gm(i*8L:i*8L+7L).GM_90   = satm[i].GM_data( 1, * )
;sp   endfor
  gm.gm_0  = (satm.gm_data[0,*])[*]
  gm.gm_90 = (satm.gm_data[1,*])[*] 

;sp  time_shaft = lonarr( 4 * (nframes + 1) )
  time_shaft = lonarr( 4 * nframes )
  sea_to_degrees = 0.006141921 * 180.0 / !pi
;sp for i=0L,nframes-1 do for j=0,3 do time_shaft(i*4+j) = satm[i].time + j*2000l
  increments = lindgen(4)*2000L
  for i=0L,nframes-1 do time_shaft[i*4L:i*4L+3L] = satm[i].time + increments
  interpIndices, time_shaft, ea.time, nearest, nextNearest
  encoder_angle = interpValues(time_shaft, satm.sea*sea_to_degrees, ea.time, $
 	nearest, nextNearest, 360.)
;  encoder_angle = hlapi_interp(time_shaft,satm.sea*sea_to_degrees,ea.time,360.)

endif else begin ; HAPI

;sp    steptime = lonarr(nframes*steps_per_frame)
    steptime = lonarr(nsteps, nsweeps)
;sp    for i=0L,nframes-1 do for j=0L,steps_per_frame-1 do  $
;sp      steptime(i*steps_per_frame + j) =satm[i].time + j *8000./steps_per_frame
;sp    increments=long((dindgen(steps_per_frame)*8000.d0) / double(steps_per_frame))
;sp    for i = 0L, nframes-1 do $
;sp      steptime(i*steps_per_frame:(i+1)*steps_per_frame-1) = stime(i) + increments
    increments = long(lindgen(nsteps)*time_per_step)
    for i = 0L, nsweeps-1 do steptime[*,i] = ea[i].time + increments
;### unsure of above line and effect on hlapi_interp lines

;  useExtMAGA = 0 ; use external MAG-A data (1) or use values in SATM file (0)
  if (n_elements(useExtMAGA) le 0) then useExtMAGA = 0
  if (useExtMAGA eq 0) then begin
    stime = satm.time ; for use in indexed arrays
    magtime = lonarr(nframes*128L)
;sp    for i=0L,nframes-1 do for j=0L,127 do $
;sp      magtime(i*128+j) = satm[i].time + j*8000./128.
    increments = long( (dindgen(128L) * 8000.d0) / 128.d0 )
    for i = 0L, nframes-1 do magtime[i*128L:i*128L+127] = stime[i] + increments

interpIndices, magtime, steptime, nearest, nextNearest
Bphi=interpValues(magtime,satm.mag_data(0,*),steptime,nearest,nextNearest,360.)
Btheta=interpValues(magtime,satm.mag_data(1,*),steptime,nearest,nextNearest,180.)
;    Bphi   = hlapi_interp(magtime,satm.mag_data(0,*), steptime, 360. )
;    Btheta = hlapi_interp(magtime,satm.mag_data(1,*), steptime, 180. )

  endif else begin ; use external MAG-A data
    read_maga, hlapi.yrday, steptime, Bphi, Btheta
  endelse ; external MAG-A

  print, 'Bphi', min(Bphi, max=xmax),xmax
  print, 'Btheta', min(Btheta, max=xmax),xmax
  if (xmax gt 180.) then print,'Change Btheta wrap value back to 360.'

endelse ; HAPI
;###
print,'Second hlapi_interp, seconds', systime(1)-t

;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;

 if not keyword_set(debug) then debug = 0

if satID eq "L" then begin ; LAPI
  pi = 4. * atan(1.0)
  dr = pi/180.
  theta = 78.3
  phi   = -69.0
  s_gmlat = sin(ea.lat * dr) * sin(theta * dr) +  $ 
    cos(ea.lat * dr) * cos(theta * dr) * cos((ea.lon-phi) * dr)
  gmlat = asin(s_gmlat) / dr
  PA = calc_pa_lapi (Bx,By,Bz, encoder_angle, sensors, gmlat) 
  sort_sens_lapi, counts, sensors, PA, EA, hlapi, debug=debug
endif else begin ; HAPI
;###
t=systime(1)
  PA = calc_pa_hapi (Bphi,Btheta,sensors,nsteps)
print,'Calc_pa_hapi, seconds', systime(1)-t
t=systime(1)
  sort_sens_hapi, counts, sensors, PA, EA, hlapi, debug=debug
print,'Sort sensors, seconds', systime(1)-t
endelse
;
;+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
;
;Make EA and GM cover only the exact time requested
start = where(ea.time le msec_start, wc) & if wc ge 1 then ea = ea(max(start):*)
end1 = where(ea.time ge msec_end, wc2) & if wc2 ge 1 then ea = ea(0:min(end1))
hlapi.nsweeps = n_elements(ea)
if satid eq 'L' then begin ; LAPI
   start2 = where (gm.time le msec_start, wc3) 
   if wc3 ge 1 then gm = gm[max(start2):*]
   end2 = where (gm.time ge msec_end, wc4)
   if wc4 ge 1 then gm = gm[0:min(end2)]
endif   

return
end ; process_satm
