function  conv_vax_unix, var_in, TARGET_ARCH=target, GFLOAT=gfloat, NOSWAP=noswap
;+
; NAME:
;      CONV_VAX_UNIX     
; PURPOSE:
;      To convert VAX IDL data types to UNIX (Sun,MIPS,etc.) IDL data types.
; EXPLANTION:
;      Generally used on non-Vax machines to parse data created on Vaxes.
;      The architecture is obtained from IDL sys.var. !VERSION.ARCH.   
;
; CALLING SEQUENCE:
;      var_unix = conv_vax_unix( var_vax, [TARGET_ARCH = ] )
;
; INPUT PARAMETER:
;      var_in - The data variable to be converted.  This may be a scalar
;            or an array.  All IDL datatypes are valid (including 
;            structures).   The result of the conversion is returned by the
;            function.
;
; OPTIONAL INPUT KEYWORD:  
;      TARGET_ARCH = name (string) of desired target architecture
;            (e.g. 'sparc' or 'mipsel').    If not supplied, then 
;            !VERSION.ARCH is used to determine the target architecture.
;            Note that CONV_VAX_UNIX will leave variables unchanged on a
;            VMS machine, unless the TARGET_ARCH keyword is set.
;            
;      GFLOAT = 1 for G-float else D-float for double precision
;
;      NOSWAP = 1 to over-ride swap_floats = 1 
;
; EXAMPLE:
;      Read a 100 by 100 matrix of floating point numbers from a data
;      file created on a VAX.  Then convert the matrix values into Sun format.
;
;      IDL> openr,1,'vax_float.dat'
;      IDL> data = fltarr(100,100)
;      IDL> readu,1,data
;      IDL> data = conv_vax_unix( data )
; NOTE:
;       Prior to IDL V5.1, the architecture "alpha" was ambiguous, since VMS 
;       alpha IDL used VAX D-float while OSF/1 alpha IDL uses little-endian 
;       IEEE.    The program uses !VERSION.OS to do the right thing when
;       converting to a representation appropriate for the current
;       platform.  To convert to a representation appropriate for
;       an OSF/1 alpha on a VAX or (pre V5.1) VMS alpha, please specify
;       the "mipsel" (or "i386") architecture.      
;
; MODIFICATION HISTORY:
;       Written   F. Varosi               August 1990
;       Modified  P. Keegstra             April 1992
;           Implemented MIPSEL architecture
;       Modified  P. Keegstra             July 1994
;           Implemented ALPHA architecture, distinguishing VMS and OSF
;       Modified  P. Keegstra             February 1995
;           Added 386 PC based architectures
;       Modified  P. Keegstra             March 1995
;           Added note, restored and fixed old specifiers 
;           for 386 PC based architectures
;      Modified W. Landsman for VAX problems in V4.0        August 1995
;      Work for double complex variables                    August 1995
;      Remove informational messages under VMS              August 1997
;      Since V5.1, IDL VMS uses little endian IEEE          June 1998
;      Convert to IDL V5.0                                  June 1998
; Add G_float to D_float with /GFLOAT switch, Robert.M.Candey@nasa.gov 2009 Mar 13
; Change to use more options to byteorder, RMC 2009 May 7
; add kludge for NOSWAP, RMC 2009 Oct 9
;-                                   
;****************************************************************************
;
;  Check to see if VAR_IN is defined.

 if n_elements( var_in ) eq 0 then begin
      message,'*** VAR_IN not defined',/CONTIN
      retall
 endif
 variable = var_in ; copy since conversion done in place

 if N_elements( target ) EQ 1 then arch = target  else arch = !VERSION.ARCH 
 little_endian = 0
 
 if not keyword_set(gfloat)  then gfloat = 0

 CASE arch OF

 "sparc": 

; Little endian machines include the Demo Version of PV-WAVE for Linux 
; (arch = '386'), IDL for MS-WINDOWS reports itself as arch="3.1".
; IDL for Linux reports itself as 'x86', Dec ultrix reports itself as 'mipsel'

 'i386': little_endian = 1      
 '3.1':  little_endian = 1
 '386i': little_endian = 1
 '386': little_endian = 1
 'x86': little_endian = 1
 'mipsel': little_endian = 1
 'x86_64':little_endian=1  ;HAL added 

 "vax": IF !VERSION.OS EQ 'vms' THEN return, variable $
        ELSE little_endian = 1

 "alpha": IF (!VERSION.OS EQ 'vms') and (!VERSION.RELEASE LT '5.1') $
         THEN return,variable $
         ELSE little_endian = 1

 else:               ;Default is to assume big-endian 'sparc' format
 ENDCASE

 if little_endian EQ 1 then begin
      swap_ints = 0
      swap_float = 2
 endif else begin
      swap_ints = 1
      swap_float = 1
 endelse
if keyword_set(noswap) then swap_float=1

 svar = size( variable )
 var_type = svar[svar[0]+1]
 scalar = (svar[0] eq 0)
;print,'little_endian=',little_endian
;print,'swap_ints, swap_float =',swap_ints,swap_float
;print,gfloat

 CASE var_type OF

  1: return, variable                                    ; byte

  2: BEGIN                                          ; integer
      if (swap_ints GT 0) then  byteorder, variable, /Sswap
      return, variable
      END

  3: BEGIN                                          ; longword
      if (swap_ints GT 0) then byteorder, variable, /Lswap
      return, variable
      END

  4: BEGIN                                       ; floating point
;      if (swap_float GT 0) then byteorder, variable, /Sswap
      byteorder, variable, /vaxtof
;      if (swap_float GT 1) then byteorder, variable, /Lswap
      return, variable
     END

  5: BEGIN                                           ; double precision
    if gfloat eq 0 then begin ; D_float
       byteorder, variable, /vaxtod
;;;??       if (swap_float GT 1) then byteorder, variable, /l64swap
       return, variable
     endif else begin ; G_float
       byteorder, variable, /sswap
       if (swap_float GT 1) then byteorder, variable, /l64swap
       return, variable/4 ; excess-1024 to excess-1022 conversion
     endelse
     END

  6: return, complex( conv_vax_unix( float( variable ), TARGET=target,gfloat=gfloat ),  $
                  conv_vax_unix( imaginary( variable ), TARGET=target,gfloat=gfloat ) )

  7: return, variable                  ; string

  8: BEGIN                        ; structure
      var_out = variable
      Ntag = N_tags( variable )
      for t = 0L, Ntag-1 do  var_out.(t) = $
                        conv_vax_unix( variable.(t), TARGET=target,gfloat=gfloat )
      return, var_out
      END

  9: return, dcomplex( conv_vax_unix( double( variable ), TARGET=target,gfloat=gfloat ),  $
                  conv_vax_unix( imaginary( variable ), TARGET=target,gfloat=gfloat ) )

  else: BEGIN
      message,'*** Data type ' + strtrim(var_type,2) + ' unknown',/CONTIN
      return,variable
      END

  ENDCASE
end
