FUNCTION read_packet, mode_id, key_number, key_time, $
                      count_modes = count_modes, $
                      get_hk_data = get_hk_data
;
;  read_packet.pro
;
;  created by: Mark Chutter, UNH
;  July 20, 1999
;
;  Modified:
;  Aug.  6, 1999 MWC removed extraneous key_time check
;                    added another check for INVALID_EDI_DATA because
;                    Jack got a CDM file who first packets were full
;                    of 0's
;  Aug.  9, 1999 MWC The last pick record is now saved in the
;                    last_record common block.  Kosta did this in the
;                    old pick library.
;  Sep. 15, 1999 MWC Changed the area where mode is checked to allow
;                    for files that contain packets from different sources,
;                    i.e., EDI HK, or SC HK
;  Oct. 19, 1999 MWC used common block for structure definitions
;  Jan.  6, 2000 MWC when setting packet_type from data, treat ENM2,
;                    ENM3, and EBM2 as ENM1
;  Jan. 27, 2000 MWC added count_modes keyword
;  Feb. 22, 2000 MWC fixed hk_fill call for rdm and ctm
;  Mar. 14, 2000 MWC added check to see if input file had any data at
;                    all before trying to return a record.  This was
;                    added because the Cluster test CDs had RDM
;                    headers followed by garbage.  The Pick Library
;                    had been cleverly designed to ignore meaningless
;                    packets. 
;  Apr. 19, 2000 MWC added 'msf' file type because msf and rdm files
;                    differ because msf use only a portion of the
;                    spacecraft housekeeping packet which is a part of
;                    the msf file where rdm files use a seperate file
;                    which contains the entire spacecraft housekeeping 
;                    packet.  There is also a msf_get_tspin routine now.
;  May   4, 2000 MWC added tspin = 4000000L for cdm files
;  May  19, 2000 MWC added check to see if Tspin is negative which
;                    indicates an error
;  Aug. 14, 2000 MWC changed srt[0] to 0L and tspin to 0L for cdm
;  Aug. 26, 2000 MWC got rid of shk check for ctm files
;  Aug. 27, 2000 MWC added another check for bad telemetry
;  Aug. 28, 2000 MWC added another check for invalid time tags; added
;                    ctm_sat_id because the SC ID is not in ctm
;                    telemetry
;  Sep.  5, 2000 MWC added shk common block
;  Sep. 13, 2000 MWC added mag_lun common block and call to
;                    get_calib_fgm
;  Sep. 14, 2000 MWC slightly changed how fgm is retrieved
;  Sep. 19, 2000 MWC added mag_lun common
;  Sep. 26, 2000 MWC now packets where sub_mode does not match
;                    packet_length are pitched
;  Sep. 27, 2000 MWC added fake B data if none is available;
;                    replaced bunch of IF IF-ELSE IF packet data
;                    mismatch
;  Oct.  5, 2000 MWC check for shk error before trying to get sun
;                    pulses and tspin
;  Oct. 10, 2000 MWC added more FGM fake fill data when there is no
;                    calibrated FGM file
;  Oct. 27, 2000 MWC now you get FGM only if you have the
;                    PICK_FGM_OLD_STYLE environment variable set
;  Nov.  9, 2000 MWC I seemed to have messed up getting SRTs from MSF
;                    files sometime during commissioning, hopefully it
;                    is now fixed.
;  Nov. 13, 2000 MWC added sc_id to nm1_fill and bm1_fill calls
;  Dec.  1, 2000 MWC rdm_get_packet now gets pick_lun as an argument
;  Jan.  5, 2001 MWC added data_stream for RDM and MSF
;  Feb. 10, 2001 MWC added NM4; packet_type is now NM1 for NM2, NM3, NM4, BM2
;  Apr.  3, 2001 MWC changed pick_struct COMMON block because of the
;                    addition of pacmo4
;  Aug.  9, 2001 MWC added get_hk_data keyword
;  Sep. 20, 2001 MWC added pick_tm_check routine to check for screwy
;                    data intervals and mark or fix them if possible
;  Jan. 14, 2002 MWC added amb_geos struct
;  May  23, 2003 MWC added check for 0.468383 second jump in packet
;                    time.  When this occurs, adjust the packet time
;                    of the science and possibly sun_data and hk.
;  July  5, 2007 MWC When looking at an ambient file, I noticed that
;                    there was an error stemming from a packet with an
;                    srt of 0.  I think this is an error, so I added a
;                    check for an srt of 0.
;
;  Calling Interface:
;  mode_id    P_LONG (i/o) Science Mode
;  key_number P_LONG (i/o) Packet Type Search Key
;  key_time   P_LONG (i/o) Packet Time Search Key
;
;  Keywords:
;  get_hk_data if the file type is MSF and this keyword is set, then
;              EDI HK data will be included in the returned structure
;
;  Return Value:
;  One of the following depending on mode_id:
;    SC_PICK_ENM_STRUCT
;    SC_PICK_EBM1_STRUCT
;    SC_PICK_EBM3_STRUCT
;  is returned.  If the status field is less than zero, then the structure is
;  not valid.
;
;  Affected Variables:

;  Description:
;  read_packet is called from one of pickn_XX, picka_XX, etc.  It looks for a
;  packet of the requested type in an open data file.  If a specific packet
;  type is sought or key_number or key_time is set, it searches through the 
;  data file until an appropriate packet is found or the end of the file is 
;  reached.

COMMON pick_structs, $
  sc_pick_ehdr_struct, $
  sc_pick_eshd_struct, $
  sc_pick_esm8_struct, sc_pick_esmc_struct, sc_pick_esmd_struct, $
  sc_pick_esm1_struct, sc_pick_esm1dgx_struct, sc_pick_esm1dgb_struct, $
  sc_pick_esm5_struct, sc_pick_esm11_struct, sc_pick_esm13_struct, $
  sc_pick_dfgm_struct, sc_pick_amb_geos_struct, $
  sc_pick_gd_struct, sc_pick_bci16_struct, $
  sc_pick_npacmo5_struct, sc_pick_npacmo4_struct, sc_pick_npacmo2_struct, $
  sc_pick_npacmo1_struct, sc_pick_npacmo0_struct, $
  sc_pick_bpacmo4_struct, sc_pick_bpacmo1_struct, sc_pick_bpacmo0_struct, $
  sc_pick_enm_struct, sc_pick_ebm1_struct, sc_pick_ebm3_struct, $
  sc_pick_pck_struct, sc_pick_headers_struct
COMMON pick, file_type, pick_lun
COMMON pick_buffer, pick_buffer, pick_buf_ptr
COMMON shk, pick_shk_lun
COMMON last_record, last_record
COMMON ctm_sat_id, ctm_sat_id
COMMON mag_lun, pick_mag_lun

@pick_const

NM1_SIZE = 980
BM1_SIZE = 6944
BM2_SIZE = 992
BM3_SIZE = 23188

pickX_pk = -1
status = 0
packet_found = 0
packet_type = 0
right_mode = 0
packet_data_match = 0
good_data = -1
key_check = -1
tm_rate = 0
missing_fgm = 0

;Added Aug. 9, 2001 MWC If the data file is MSF, it makes sense to
;unpack the EDI HK packet for the same time
IF keyword_set(get_hk_data) THEN get_hk = 1 $
ELSE get_hk = 0

IF (mode_id GE 1000) THEN BEGIN ;IDL call was a pickX_pk type
    pickX_pk = 1
    mode_id = mode_id - 1000
ENDIF

; keep looking for a packet with good data of the type requested until
; one is found or eof
WHILE (good_data LT 1) DO BEGIN

    packet_data_match = 0

    CASE file_type OF
        'cdm':packet = cdm_get_packet(key_number, key_time)
        'rdm':packet = rdm_get_packet(key_number, key_time, $
                                      file_type, pick_lun)
        'msf':BEGIN ;msf files contain 3 different packets for each time
            ;SC HK first
            sun_data = rdm_get_packet(key_number, key_time, $
                                      file_type, pick_lun)
            ;EDI HK second
            packet_hk = rdm_get_packet(key_number, key_time, $
                                       file_type, pick_lun)
            ;EDI SC third
            packet = rdm_get_packet(key_number, key_time, $
                                    file_type, pick_lun)
        END
        'ctm':packet = ctm_get_packet(key_number, key_time)
    ENDCASE

    rec_check = n_elements(last_record)
    
    IF (packet.status EQ EOF) THEN BEGIN
        ; check to see if there was any data 
        rec_check = n_elements(last_record)
        IF rec_check NE 0 THEN BEGIN
            record = last_record
            record.status = EOF
            return, record
        ENDIF ELSE return, packet
    ENDIF

    ;if count_modes then exit after extracting header
    IF keyword_set(count_modes) THEN BEGIN
        hdr = edi_header_fill(packet.data, $
                              packet.packet_number, packet.packet_time)
        last_record = hdr
        return, hdr
    ENDIF
    
    ; try to read another packet if key conditions not met
    IF (packet.status LE 0) THEN BEGIN
        key_check = -1
    ENDIF ELSE key_check = 1

    IF (key_check GT 0) THEN BEGIN

        IF (mode_id EQ 1) THEN BEGIN ; EDI HK for pickn_hk or pickst_hk
            IF file_type EQ 'cdm' THEN hk_record = hk_fill(packet.hk) $
            ELSE hk_record = hk_fill(packet.data)
            hk_record.packet_type = PICK_HK_PACKET
            hk_record.packet_number = packet.packet_number
            hk_record.packet_time[0] = packet.packet_time[0]
            hk_record.packet_time[1] = packet.packet_time[1]
            last_record = hk_record
            return, hk_record
        ENDIF

        IF (mode_id EQ 1001) THEN BEGIN ; EDI HK for pickX_pk
            sc_packet = sc_pick_pck_struct
            sc_packet.packet_type = PICK_HK_PACKET
            sc_packet.packet_number = packet.packet_number
            sc_packet.packet_time[0] = packet.packet_time[0]
            sc_packet.packet_time[1] = packet.packet_time[1]
            sc_packet.packet_length = HK_SIZE/2
            FOR i=0, HK_SIZE, 2 DO $ 
              sc_packet.data[i] = fix(packet.hk[i]) * '100'x + $
              fix(packet.hk[i+1])
            sc_packet.status = PICK_HK_PACKET
            last_record = sc_packet
            return, sc_packet
        ENDIF

                                ; for EDI, determine the packet type;
                                ; treat ENM2, ENM3, ENM4, and EBM2 as ENM1
        IF (packet.source EQ PICK_EDI_PACKET) THEN BEGIN

            type = (packet.data[1] AND '7'x)
            CASE type OF 
                0: packet_type = INVALID_EDI_DATA
                1: packet_type = SC_PICK_ENM1_PACKET
                2: packet_type = SC_PICK_ENM1_PACKET
                3: packet_type = SC_PICK_ENM1_PACKET
                4: packet_type = SC_PICK_ENM1_PACKET
                5: packet_type = SC_PICK_EBM1_PACKET
                6: packet_type = SC_PICK_ENM1_PACKET
                7: packet_type = SC_PICK_EBM3_PACKET
                ELSE: packet_type = INVALID_EDI_DATA
            ENDCASE

      ; When the mode_id is greater than 0 (for pickn_XX types),
      ; check to see if the current packet is of the desired type.
      ; If not, break and try again.
            right_mode = 0
            IF packet_type LE 0 THEN right_mode = 0 $ ;invalid data
            ELSE IF (mode_id EQ 0) THEN BEGIN ; any packet
                right_mode = 1
            ENDIF ELSE IF (mode_id GT 0) THEN BEGIN
                IF ((mode_id EQ PICK_SCI_PACKET) AND (packet_type GT 1)) THEN $
                  right_mode = 1 $ ; any science mode
                ELSE IF ((mode_id EQ SC_PICK_ENM_PACKET) AND $
                         ((packet_type GT 10) AND (packet_type LT 20))) THEN $
                  right_mode = 1 $
                ELSE IF ((mode_id EQ SC_PICK_EBM_PACKET) AND $
                         (packet_type GT 20)) THEN $
                  right_mode = 1 $
                ELSE IF (mode_id EQ packet_type) THEN $
                  right_mode = 1
            ENDIF

      ENDIF ELSE IF (packet.source EQ PICK_HK_PACKET) THEN BEGIN
          right_mode = -1 ; here we are looking for science data,
                          ; hk is returned above
      ENDIF ELSE BEGIN

          print, $
            'The Pick Library can only be used for EDI packets.'
          return, NOT_EDI_PACKET
      ENDELSE 

      size_check = n_elements(packet.data)
      IF file_type EQ 'cdm' THEN packet_data_match = 0 ELSE $
      IF packet_type EQ SC_PICK_ENM1_PACKET AND $
        size_check GT NM1_size+12 THEN BEGIN ;12 is for BM2
          packet_data_match = INVALID_EDI_DATA
      ENDIF ELSE IF packet_type EQ SC_PICK_EBM1_PACKET AND $
        size_check NE BM1_size THEN BEGIN
          packet_data_match = INVALID_EDI_DATA
      ENDIF ELSE IF packet_type EQ SC_PICK_EBM3_PACKET AND $
        size_check NE BM3_size THEN BEGIN
          packet_data_match = INVALID_EDI_DATA
      ENDIF

      IF (right_mode EQ 1) AND (packet_data_match EQ 0) THEN BEGIN
          ; desired return type is sc_pick_pck_struct
          IF (pickX_pk GT 0) THEN BEGIN 

              sc_packet.packet_type = packet_type
              sc_packet.packet_number = packet.packet_number
              sc_packet.packet_time[0] = packet.packet_time[0]
              sc_packet.packet_time[1] = packet.packet_time[1]
              sc_packet.packet_length = packet.packet_length/2
              FOR i = 0, packet.packet_length-1, 2 DO $ 
                sc_packet.data[i] = fix(packet.data[i]) * '100'x + $
                fix(packet.data[i+1])

              return, sc_packet
          ENDIF 

	  ; fill the pick structure if the packet contains good edi data
          IF (packet_type EQ SC_PICK_ENM1_PACKET) THEN BEGIN

              pick_record = $
                nm1_fill(packet.data, packet.packet_number, $
                         packet.packet_time, packet.sc_id)
              pick_record.sc_id = packet.sc_id
              
          ENDIF ELSE IF ((packet_type EQ SC_PICK_EBM1_PACKET) OR $
                         (packet_type EQ SC_PICK_EBM3_PACKET)) THEN BEGIN
              IF (packet_type EQ SC_PICK_EBM1_PACKET) THEN BEGIN
                  pick_record = bm1_fill(packet.data, packet.packet_number, $
                                         packet.packet_time, packet.sc_id)
                  pick_record.sc_id = packet.sc_id
              ENDIF
              IF (packet_type EQ SC_PICK_EBM3_PACKET) THEN BEGIN
                  pick_record = bm3_fill(packet.data, packet.packet_number, $
                                         packet.packet_time)
                  pick_record.sc_id = packet.sc_id
              ENDIF 
          ENDIF

          ; data stream is byte 13 of the DDS header
          IF file_type EQ 'rdm' OR $
            file_type EQ 'msf' THEN $
            pick_record.data_stream = packet.data_stream

          IF (pick_record.status GT 0) THEN BEGIN

              good_data = 1

              IF file_type EQ 'ctm' THEN pick_record.sc_id = ctm_sat_id
              IF (file_type EQ 'rdm' OR $
                  file_type EQ 'msf') AND $
                ((packet_type EQ SC_PICK_ENM1_PACKET) OR $ 
                 (packet_type EQ SC_PICK_EBM1_PACKET)) THEN BEGIN 
                                ;if there is no SC_HK data available
                  IF pick_shk_lun EQ -3 AND file_type EQ 'rdm' THEN BEGIN
                      pick_record.srt[0] = 99 ; an obviously nonsensical time
                  ENDIF ELSE BEGIN
                      IF file_type EQ 'rdm' THEN $
                        sun_data = get_sun_data(file_type, packet.packet_time)
                      IF sun_data.status GT 0 THEN BEGIN
                          get_tspin, sun_data, srp, srp1, Tspin
                                ;the following line was added on
                                ;20070705 because I discovered that
                                ;some packets had an srt of 0
                          IF srp[0] EQ 0 THEN good_data = -1
                          IF Tspin GE 0L THEN BEGIN
                              pick_record.srt = srp
                              pick_record.srt1 = srp1
                              pick_record.tspin = Tspin
                          ENDIF
                      ENDIF
                  ENDELSE
;try to read calibrated FGM data if it is available                 
                  missing_fgm = 0
                  old_style = getenv('PICK_FGM_OLD_STYLE')
                  IF pick_mag_lun NE -3 AND old_style NE '' THEN BEGIN
                      start_time = double(pick_record.hdr.packet_time[0]) + $
                        double(pick_record.hdr.packet_time[1])*1.d-6 -5.152221D
                      fgm = get_calib_fgm(start_time, 5.152221D)
                      IF fgm.status GT 0 AND $
                        fgm.count GT 0 THEN BEGIN
                          pick_record.mag_time[0:fgm.count-1] = fgm.mag_time
                          pick_record.bxyz[0:fgm.count-1, 0] = fgm.bx
                          pick_record.bxyz[0:fgm.count-1, 1] = fgm.by
                          pick_record.bxyz[0:fgm.count-1, 2] = fgm.bz
                      ENDIF ELSE missing_fgm = 1
                  ENDIF
                  IF pick_mag_lun EQ -3 OR missing_fgm EQ 1 THEN BEGIN 
                      start_time = double(pick_record.hdr.packet_time[0]) + $
                        double(pick_record.hdr.packet_time[1])*1.d-6 -5.152221D
                      pick_record.mag_time[0:149] = start_time + $
                        dindgen(150) * 5.152221D/150.D
                      pick_record.bxyz[0:149, 0]= 1.
                      pick_record.bxyz[0:149, 1]= 1.
                      pick_record.bxyz[0:149, 2]= 1.
                  ENDIF
              ENDIF
              IF file_type EQ 'cdm' AND $
                ((packet_type EQ SC_PICK_ENM1_PACKET) OR $
                 (packet_type EQ SC_PICK_EBM1_PACKET)) THEN BEGIN
                  pick_record.srt[0] = 99
                  pick_record.tspin = 0L ;4000000L
              ENDIF

              IF file_type EQ 'msf' AND get_hk EQ 1 THEN BEGIN
                  hk_record = hk_fill(packet_hk.data)
                  hk_record.packet_type = PICK_HK_PACKET
                  hk_record.packet_number = packet.packet_number
                  hk_record.packet_time[0] = packet.packet_time[0]
                  hk_record.packet_time[1] = packet.packet_time[1]
                  hk_record.sc_id = packet.sc_id
                  pick_record.hk = hk_record
              ENDIF
              pick_tm_check, pick_record
          ENDIF                 ; if pick_record.status
      ENDIF                     ; right_mode
  ENDIF                         ; key_check
ENDWHILE                        ; while (good_data < 1)

pick_record.status = packet_type
last_record = pick_record
return, pick_record
END
