FUNCTION rdm_get_tcal, key_time
; CLUSTER II IDL Function
;
; Abstract: extract time calibration data from a RDM packet
;
; Created by: Mark Chutter, UNH 
; Apr. 12, 2001
;
; Modified:
;
; Calling Interface:
; key_time   lonarr(2) (i) packet time search key
;
; Return Value:
; A rdm structure is returned.  If the error field is
; less than zero, then the structure is not valid.
;
; Affected Variables:
;
; Description:
; This function reads the header of a RDM packet to determine the
; source (which instrument and what kind of data) and the length of
; the packet.  It then reads the packet itself.
; For rdm files, the packet_number is the same as the second element of
; the packet time array.  The packet_time array is a two element array whose
; values come from the Spacecraft Event Time (SCET) from the first eight
; bytes in the DDS header (the packet header).  The first 2 bytes contain
; the number of days since January 1, 1958.  The next four bytes contain the
; number of milliseconds since the start of the current day.  The next 2
; bytes contain the number of microseconds within the current millisecond.
COMMON tcal, pick_tcal_lun

PICK_TCAL_PACKET =  6
NOT_TCAL_PACKET = -999

@pick_const
RDM_HEADER_SIZE = 15
rdm_header = bytarr(RDM_HEADER_SIZE)
packet_length = 0l
source_table = [6]
source_id = -1
source = -1
pos = 0l

on_ioerror, eof
IF pick_tcal_lun NE -3 THEN BEGIN
    readu, pick_tcal_lun, rdm_header
    packet_length = long(rdm_header[9]) * '10000'xL + $
      long(rdm_header[10]) * '100'xL + $
      long(rdm_header[11])
ENDIF ELSE packet_length = 1

data = bytarr(packet_length)

rdm = {$
        sc_id: -1, $
        source: -1, $
        packet_number: 0L, $
        packet_time: lonarr(2), $
        data: bytarr(packet_length), $
        status: 1 $
      }

;if there is no SC_HK data available
IF pick_tcal_lun EQ -3 THEN BEGIN
    rdm.packet_time[0] = 99
    rdm.status = 1
    return, rdm
ENDIF

IF (rdm_header[8] EQ source_table[0]) THEN source = PICK_TCAL_PACKET

readu, pick_tcal_lun, data

IF (source NE PICK_TCAL_PACKET) THEN BEGIN
    rdm.status = NOT_TCAL_PACKET
    return, rdm
ENDIF

rdm.packet_number = long(rdm_header[2]) * '1000000'xL + $
  long(rdm_header[3]) * '10000'xL + $
  long(rdm_header[4]) * '100'xL + $
  long(rdm_header[5])

rdm_get_packet_time, rdm_header, sec, msec
rdm.packet_time[0] = sec
rdm.packet_time[1] = msec

rdm.data = data

; if key_time[0] EQ 0, then the next tcal packet is sought,
; otherwise an tcal packet with a matching packet_time is sought
IF (key_time[0] NE 0) THEN BEGIN
    IF (rdm.packet_time[0] LT key_time[0]) THEN BEGIN
        rdm.status = PACKET_TIME_LOW
        return, rdm
    ENDIF ELSE IF (rdm.packet_time[0] GT key_time[0]) THEN BEGIN
        rdm.status = PACKET_TIME_HIGH
;BUGGER, this might not work for VMS
        point_lun, -1 * pick_tcal_lun, pos ;rewind the file by one packet
        point_lun, pick_tcal_lun, pos - RDM_HEADER_SIZE - packet_length
        rdm.packet_time[0] = 99
        return, rdm
    ENDIF
ENDIF

return, rdm
eof: rdm_error = {$
                   status: -1 $
                 }
rdm_error.status = EOF
return, rdm_error
END


