;	spaux_info.pro,v 1.11 2005/05/20 18:35:50 ppq Exp	
;+
; NAME: spaux_info.pro
;
; PURPOSE: Extract data from Hungarian SPAUX CDF file and return this
; along with other ephemeris/rotation-related parameters.  These
; parameters are returned synchronized (nearest neighbor)
; (not interpolated) to user-specified input times.
;
; CATEGORY: Ephemeris/Coordinate Transformation Data retrieval/derivation
;
; CALLING SEQUENCE:
; istat = spaux_info(scnum, time_in, ttype_in, $ ; IN
;                    date=date_in, spaux_path=spaux_path, $ ; IN
;                    diagplot=diagplot, ps=ps, $ ; IN
;                    vgse_ref=vgse_ref, vsc_gse=vsc_gse, $ ; OUT
;                    ref_pgse=ref_pgse, pgse=pgse, $ ; OUT
;                    prel_gse=prel_gse, $ ; OUT
;                    spinvec_lat_gse=spinvec_lat_gse, $ ; OUT
;                    spinvec_long_gse=spinvec_long_gse, $ ; OUT
;                    gse2gsm_rotangle=gse2gsm_rotangle, $ ; OUT
;                    dp_tilt=dp_tilt, $ ; OUT
;                    rotmat=rotmat, gse2gsm_rotmat=gse2gsm_hun, $ ; OUT
;                    sat2gse_rotmat=sat2gse_hun, $ ; OUT
;                    rmag=rmag, maglat=maglat, lshell=lshell, $ ; OUT
;                    invlat=invlat, local_time=local_time, mlt=mlt)
;
; INPUTS:
; scnum              '1', '2', '3', or '4'    Spacecraft number
; time_in            Scalar or 1-D array of desired times
; ttype_in           Format of time_in.  The 4 options are:
;         't70'          time_in is double type, in units of
;                            decimal seconds since 
;                            01-Jan-1970 00:00:00.000
;         'vax_string'   time_in is of string type, with format
;                           'DD-MMM-YYYY hh:mm:ss.mmm', e.g.
;                           '09-Dec-2000 02:59:42.583'
;         'epoch'        time_in is double type, in units of
;                            milliseconds since 
;                            01-Jan-0000 00:00:00.000
;         'ssm'          time_in is decimal seconds since midnight
;                            The 'date' keyword MUST be set if this is
;                            the format for time_in. 
;
; KEYWORD PARAMETERS:
; date=var_in                'YYYYMMDD'   MUST be present if ttype_in='ssm'
; spaux_path=var_in          String containing path to SPAUX files
;                            DEFAULT: Uses environment variable
;                            SPAUX_PATH if set; If not, uses
;                            /home/ppq/sp_data/
; /diagplot, /ps             Diagnostic keywords
;
;-----------------------------------------------------------
; Quantities directly from SPAUX CDF file, synchronized (NOT
; interpolated) to the time_in input times
;-----------------------------------------------------------
; prel_gse=var_out       Position of SC relative to reference
;                        spacecraft, km, GSE
; vgse_ref=var_out       Velocity in GSE of reference spacecraft, km/s
;
; vsc_gse=var_out        Velocity in GSE of desired spacecraft, km/s
;                        This is not directly extracted from the
;                        SPAUX file, but rather it is constructed
;                        using the position vectors and the time
;                        differences via:
;                          vx(i) = (x(i+1)-x(i-1))/(t(i+1)-t(i-1))
;                        and similarly for the other two components.
;                        (Simple, yes, but good enough...)
;
; ref_pgse=var_out       Position in GSE of reference spacecraft, km
; pgse=var_out           Position in GSE of SC, km
; spinvec_lat_gse=var_out  Spin-vector latitude of SC in GSE, [-90,90] 
;                          degrees, angle from ecliptic plane
; spinvec_long_gse=var_out Spin-vector longitude of SC in GSE, [0,360] 
;                          degrees, angle from x in x-y GSE plane
; gse2gsm_rotangle=var_out Rotation angle from GSE to GSM, [-90,90] 
;                          degrees, positive from +z towards +y
; dp_tilt=var_out          Dipole tilt in GSM z-x plane , [-90,90] 
;                          degrees, positive from +z towards +x
;
;-----------------------------------------------------------
; Quantities derived from the synchronized SPAUX CDF variables
;-----------------------------------------------------------
; /rotmat                   Set this in order to access gse2gsm_rotmat
;                           and sat2gse_rotmat; The default is to NOT
;                           have these available because they can take up
;                           alot of space.
; gse2gsm_rotmat=var_out    GSE to GSM rotation matrices, DIM=[3,3,ntime]
; sat2gse_rotmat=var_out    SAT to GSE rotation matrices, DIM=[3,3,ntime]
;
;  IMPORTANT ROTMAT NOTICE:  
;        The rotation matricies gse2gsm and sat2gse are stored in the
;        traditional physics/math sense of (irow,icol).  So that to
;        rotate a vector:
;             vprime(i) = M(i,k)*v(k)  (Einstein notation, sum over k)
;        This is contrary to the fact that IDL is row major and that
;        matricies are assumed to be stored as (icol,irow).  So,
;        BE CAREFUL WHEN YOU USE IDL MATRIX OPERATORS!
;
; rmag=var_out              Radial distance of SC from center of
;                           earth, km
; maglat=var_out            Magnetic latitude (positive measured from
;                           the magnetic equator towards the 
;                           positive dipole direction), degrees
; lshell=var_out            LSHELL parameter, based on dipole model
; invlat=var_out            Invariant latitude, based on dipole model
;                           Also, invlat is assigned the sign of maglat.
; local_time=var_out        Local time in seconds since midnight (GSE)
;
; mlt=var_out               Magnetic local time in seconds since midnight
;
; COMMON BLOCKS: NONE
;
; SIDE EFFECTS: Uses some code from EDI_PISO distribution.
;
; RESTRICTIONS:
;
; PROCEDURE: Get data from SPAUX CDF file; synchronize to input times,
; derive additional ephemeris/rotation-related quantities.
;
; EXAMPLE:
; time_in = findgen(21600)/21599.*86400. ; 4-second resolution times
; istat = spaux_info('1', time_in, 'ssm', $
;                    date = '20010223', spaux_path = $
;                    '/home/ppq/sp_data/', $
;                    rmag=rmag, maglat=maglat, lshell=lshell, $
;                    invlat=invlat)
; if (istat) then message, 'spaux_info was successful', /cont
;
;
; MODIFICATION HISTORY: Written 05/2001 by 
;                       Pamela A. Puhl-Quinn, ppq@mpe.mpg.de
;-

function spaux_info, scnum, time_in, ttype_in, $ ; IN
                     date=date_in, spaux_path=spaux_path, $ ; IN
                     diagplot=diagplot, ps=ps, $ ; IN
                     vgse_ref=vgse_ref, ref_pgse=pgse_ref, pgse=pgse, $ ; OUT
                     prel_gse=prel_gse, $ ; OUT
                     vsc_gse=vsc_gse, $ ; OUT
                     spinvec_lat_gse=spinvec_lat_gse, $ ; OUT
                     spinvec_long_gse=spinvec_long_gse, $ ; OUT
                     gse2gsm_rotangle=gse2gsm_rotangle, $ ; OUT
                     dp_tilt=dp_tilt, $ ; OUT
                     rotmat=rotmat, gse2gsm_rotmat=gse2gsm_hun, $ ; OUT
                     sat2gse_rotmat=sat2gse_hun, $ ; OUT
                     rmag=rmag, maglat=maglat, t_ssm=t_ssm, $
                     lshell=lshell, radearth=radearth, $ ; OUT
                     invlat=invlat, local_time=local_time, mlt=mlt, $ ; OUT
                     dr_min=dr_min, dr_max=dr_max

if (ttype_in ne 'vax_string') then begin
    if (min(time_in) lt 0.) then begin
        message, 'Time array has negative value(s)', /cont
        return, 0
    endif
endif

rotmat = keyword_set(rotmat)
diagplot = keyword_set(diagplot)
ps = keyword_set(ps)

if (n_elements(spaux_path) eq 0) then begin
    spaux_path = getenv('SPAUX_PATH')
    if (spaux_path eq '') then spaux_path = '/nfs/cl11/CDF/spaux/'
endif

; '1', findgen(86400), 'ssm', date = '20010223', /diagplot, /ps

rfill = -1e31
;message, 'ep_time_handler start: ', /cont
epoch_in = ep_time_handler(time_in, ttype_in, 'epoch', date=date_in)
ssm_in = ep_time_handler(time_in, ttype_in, 'ssm', date=date_in)
;message, 'ep_time_handler stop: ', /cont


; Determine the date of the Hungarian SPAUX CDF file we need
ntime_in = n_elements(epoch_in)
cdf_epoch, epoch_in(0), year, month, day, $
  hour, minute, second, msec, /break
sdate = string(year,'(i4.4)')+string(month,'(i2.2)')+string(day,'(i2.2)')
cdf_epoch, epoch_in(ntime_in-1), year, month, day, $
  hour, minute, second, msec, /break
edate = string(year,'(i4.4)')+string(month,'(i2.2)')+string(day,'(i2.2)')
if (sdate ne edate) then begin
    message, 'Not coded yet to cross day boundaries', /cont
    return, 0
endif
date = sdate

; Extract the SPAUX data
;message, 'ep_extract_spaux_data start: ', /cont
estat = ep_extract_spaux_data(date, scnum, $
                              epoch_spaux, ssm_spaux, $
                              vgse_ref, pgse_ref, prel_gse, $
                              pgse, spinvec_lat_gse, spinvec_long_gse, $
                              gse2gsm_rotangle, dp_tilt, spaux_path, $
                              dr_min=dr_min, dr_max=dr_max)
;message, 'ep_extract_spaux_data stop: ', /cont
if (not estat) then begin
    message, 'SPAUX CDF file extraction unsuccessful', /cont
    return, 0
endif
ntime_synch = n_elements(epoch_spaux)

; Approximate the velocities for each of the SC (note: only the
; reference SC velocity actually exists...)
vsc_gse = make_array(size=size(vgse_ref))
n = n_elements(ssm_spaux)
t3 = make_array(3,/float,value=1.)#ssm_spaux ; [3,n]
vsc_gse(0:2,1:n-2) = (pgse(0:2,2:n-1)-pgse(0:2,0:n-3))/ $ ; km/s
  (t3(0:2,2:n-1)-t3(0:2,0:n-3))
vsc_gse(0:2,0) = (pgse(0:2,1)-pgse(0:2,0))/ $
  (t3(0:2,1)-t3(0:2,0))
vsc_gse(0:2,n-1) = (pgse(0:2,n-1)-pgse(0:2,n-2))/ $
  (t3(0:2,n-1)-t3(0:2,n-2))

;message, 'Synch start: ', /cont
; Synchronize the CDF quantities to the time_in times.
; Interpolation is a bit tricky because of the angular
; quantities...must think about this some more...
dt = min(ssm_spaux(1:ntime_synch-1)-ssm_spaux(0:ntime_synch-2))
isynch = long((ssm_in-ssm_spaux(0))/dt + .5)

max_diff = max(abs(ssm_in - ssm_spaux(isynch)))
if ((min(isynch) lt 0) or $
    (max(isynch) ge ntime_synch) or $
    (max_diff gt dt/2.)) then begin
    message, 'Something wrong with synchronization...Talk to Pam Puhl-Quinn, ppq@mpe.mpg.de', /cont
    return, 0
endif

epoch_synch = epoch_spaux(isynch)
ssm_synch = ssm_spaux(isynch)
vgse_ref = vgse_ref(0:2,isynch)
vsc_gse = vsc_gse(0:2,isynch)
pgse_ref = pgse_ref(0:2,isynch)
pgse = pgse(0:2,isynch)
prel_gse = prel_gse(0:2,isynch)
spinvec_lat_gse = spinvec_lat_gse(isynch)
spinvec_long_gse = spinvec_long_gse(isynch)
gse2gsm_rotangle = gse2gsm_rotangle(isynch)
dp_tilt = dp_tilt(isynch)
dr_min = dr_min(isynch)
dr_max = dr_max(isynch)

ntime_synch = n_elements(epoch_synch)

;message, 'Synch stop: ', /cont

;message, 'gse2gsm alloc start: ', /cont
; Get the Hungarian rotation matrices
; GSE2GSM--------------------------
gse2gsm_hun = make_array(3,3,ntime_synch,/double,value=rfill)
sipsi = sin(double(gse2gsm_rotangle) * !dpi/180.d0)
copsi = cos(double(gse2gsm_rotangle) * !dpi/180.d0)
gse2gsm_hun(0,0,*) = 1.0
gse2gsm_hun(0,1,*) = 0.0
gse2gsm_hun(0,2,*) = 0.0
gse2gsm_hun(1,0,*) = 0.0
gse2gsm_hun(1,1,*) = copsi
gse2gsm_hun(1,2,*) = -sipsi
gse2gsm_hun(2,0,*) = 0.0
gse2gsm_hun(2,1,*) = sipsi
gse2gsm_hun(2,2,*) = copsi
;message, 'gse2gsm alloc stop: ', /cont

; SAT2GSE--------------------------
; This rotation matrix is not required by the other ephemeris
; quantities, so only calculate this if the user wants it returned (a
; space saver).
if (rotmat) then begin
    sat2gse_hun = make_array(3,3,ntime_synch,/double,value=rfill)
    RAC = double(spinvec_long_gse)*!dpi/180d0
    DEC = double(spinvec_lat_gse)*!dpi/180d0
    EPX = COS(DEC) * COS(RAC)
    EPY = COS(DEC) * SIN(RAC)
    EPZ = SIN(DEC)
    COSA = SQRT(EPY*EPY + EPZ*EPZ)
    SINA = EPX
    COSD = EPZ/COSA
    SIND = EPY/COSA
    sat2gse_hun(0,0,*) = COSA
    sat2gse_hun(0,1,*) = 0.
    sat2gse_hun(0,2,*) = SINA
    sat2gse_hun(1,0,*) = -SINA * SIND
    sat2gse_hun(1,1,*) = COSD
    sat2gse_hun(1,2,*) = COSA * SIND
    sat2gse_hun(2,0,*) = -SINA * COSD
    sat2gse_hun(2,1,*) = -SIND
    sat2gse_hun(2,2,*) =  COSA * COSD
endif

;message, 'Ephem start: ', /cont
; Magnetic dipole related quantities
; Need the position of the SC in a coordinate system where B-dipole is
; along the z-axis (the DIP coordinate system)
zdip = make_array(3,ntime_synch,/double)
zdip(2,*) = cos(dp_tilt*!dpi/180d0)
zdip(1,*) = 0.
zdip(0,*) = sin(dp_tilt*!dpi/180d0)
bgsm = zdip

ydip = make_array(3,ntime_synch,/double)
ydip(0,*) = 0d0
ydip(1,*) = 1d0
ydip(2,*) = 0d0

xdip = make_array(3,ntime_synch,/double)
xdip(0,*) = ydip(1,*)*zdip(2,*) - ydip(2,*)*zdip(1,*)
xdip(1,*) = ydip(2,*)*zdip(0,*) - ydip(0,*)*zdip(2,*)
xdip(2,*) = ydip(0,*)*zdip(1,*) - ydip(1,*)*zdip(0,*)

gsm2dip = make_array(3,3,ntime_synch,/double)
gsm2dip(0:2,0,*) = xdip(0:2,*)
gsm2dip(0:2,1,*) = ydip(0:2,*)
gsm2dip(0:2,2,*) = zdip(0:2,*)
if (ntime_synch ne 1) then gsm2dip = transpose(gsm2dip,[1,0,2]) else gsm2dip = transpose(gsm2dip) ; Inverse = Transpose

bdip = make_array(size=size(bgsm))
pgsm = make_array(size=size(pgse))
for ic=0,2 do begin
    bdip(ic,*) = gsm2dip(ic,0,*)*bgsm(0,*) + $
      gsm2dip(ic,1,*)*bgsm(1,*) + gsm2dip(ic,2,*)*bgsm(2,*)
    
    pgsm(ic,*) = gse2gsm_hun(ic,0,*)*pgse(0,*) + $
      gse2gsm_hun(ic,1,*)*pgse(1,*) + gse2gsm_hun(ic,2,*)*pgse(2,*)
endfor

pdip = make_array(size=size(pgsm))
for ic=0,2 do begin
    pdip(ic,*) = gsm2dip(ic,0,*)*pgsm(0,*) + $
      gsm2dip(ic,1,*)*pgsm(1,*) + gsm2dip(ic,2,*)*pgsm(2,*)
endfor

radearth = 6380d0               ; km
rmag = sqrt(total(pdip^2,1,/double)) ; km
colat = acos(reform(pdip(2,*))/rmag) ; rad
maglat = !dpi/2d0 - colat
r0 = rmag/(sin(colat))^2
lshell = r0/radearth

invlat = acos(sqrt(1d0/lshell)) ; rad
sign_maglat = maglat/abs(maglat)
invlat = invlat*sign_maglat

phi = atan(reform(pgse(1,*)),reform(pgse(0,*))) ; rad, [-180,180]
secday = 86400.
local_time = secday/2./!pi*phi + secday/2. ; seconds since midnight

; Add magnetic local time to output
; This has been checked against sscweb.gsfc.nasa.gov and is
; correct. PPQ, 2/2004
iin=n_elements(gse2gsm_rotangle)
psm=fltarr(3,iin)
mlt=fltarr(iin)
for iii=0L, iin-1L do begin
    rotmtr1=[[1.,0.,0.],$
             [0.,cos(gse2gsm_rotangle(iii)*!pi/180.),$
              -sin(gse2gsm_rotangle(iii)*!pi/180.)],$
             [0.,sin(gse2gsm_rotangle(iii)*!pi/180.),$
              cos(gse2gsm_rotangle(iii)*!pi/180.)]]
    rotmtr2=[[cos(dp_tilt(iii)*!pi/180.),0.,-sin(dp_tilt(iii)*!pi/180.)],$
             [0.,1.,0.],$
             [sin(dp_tilt(iii)*!pi/180.),0.,cos(dp_tilt(iii)*!pi/180.)]]
    pgsm1=reform(rotmtr1##pgse(*,iii))
    psm(*,iii)=reform(rotmtr2##pgsm1)
    mlt(iii)=atan(-psm(1,iii),-psm(0,iii))*43200./!pi
    if (mlt(iii) lt 0.) then mlt(iii) = mlt(iii)+86400.
endfor

; Change units for output
maglat = maglat*!radeg
invlat = invlat*!radeg
t_ssm = ssm_in

;message, 'Ephem stop: ', /cont

if (diagplot) then begin
    psname = 'c'+scnum+'_'+date+'_spaux_info_diagplot.ps'
    if (ps) then begin
        set_plot, 'PS'
        device, file=psname, /portrait, $
          xoff=0.25, yoff=0.25, xs=8., ys=10.5, /inches
    endif else window, 0, xs=800, ys=900
    
    !p.multi = [0,1,7]
    
    _extra = {xtickformat:'tick_ephem_spaux',xrange:[0,24]*3600., $
              xtickv:findgen(7)/6.*24.*3600.,xticks:6,xminor:4}
    
    plot, ssm_synch, asin(reform(pgse(2,*))/rmag)*!radeg, _extra=_extra, ytitle='GSE_Lat', title=date+', C'+scnum, yrange=[-100,100], ystyle=1
    plot, ssm_synch, atan(reform(pgse(1,*)),reform(pgse(0,*)))*!radeg, _extra=_extra, ytitle='GSE_Long', yrange=[-200,200], ystyle=1
    plot, ssm_synch, asin(reform(pgsm(2,*))/rmag)*!radeg, _extra=_extra, ytitle='GSM_Lat', yrange=[-100,100], ystyle=1
    plot, ssm_synch, atan(reform(pgsm(1,*)),reform(pgsm(0,*)))*!radeg, _extra=_extra, ytitle='GSM_Long', yrange=[-200,50], ystyle=1
    
    plot, ssm_synch, rmag/radearth, _extra=_extra, ytitle='rmag/R!dE!n', yrange=[4,18], ystyle=1
    
    plot, ssm_synch, lshell, _extra=_extra, ytitle='lshell', yrange=[0,400], ystyle=1
    
    plot, ssm_synch, invlat, _extra=_extra, ytitle='invlat',yrange=[60,90], ystyle=1
    
    time_stamp_edi, label='Code: spaux_info.pro!cPlot: '+psname

    if (ps) then begin
        device, /close_file
        set_plot, 'X'
    endif
endif

return, 1

;============================================================
; Get the Hapgood rotation matrices for comparison
vec_scs_in = make_array(3,ntime_synch,/float,value=1.) ; Dummy
;istat =  ep_scs2gse(scnum, epoch_synch, vec_scs_in, vec_gse, $
;                    vtype='B', ttype='epoch', $
;                    sat2gse=sat2gse_hap, gse2gsm=gse2gsm_hap)
;if (not istat) then message, 'Hapgood failure'


!p.multi = [0,2,2]

window, 0, xs=900, ys=900
xrange = [-1.1,1.1]
yrange = xrange
plot, sat2gse_hap, sat2gse_hun, xtitle='sat2gse_hap', ytitle='sat2gse_hun', $
  psym=3, title=date+', C'+scnum+', All day, all 9 components', $
  xrange=xrange, yrange=yrange, xstyle=1, ystyle=1

plot, abs(sat2gse_hap-sat2gse_hun), psym=3, /ylog, ytitle='abs(sat2gse_hap-sat2gse_hun)', xtitle='Index'

xrange = [-1.1,1.1]
yrange = xrange
plot, gse2gsm_hap, gse2gsm_hun, xtitle='gse2gsm_hap', ytitle='gse2gsm_hun', $
  psym=3, title=date+', C'+scnum+', All day, all 9 components', $
  xrange=xrange, yrange=yrange, xstyle=1, ystyle=1

plot, abs(gse2gsm_hap-gse2gsm_hun), psym=3, /ylog, ytitle='abs(gse2gsm_hap-gse2gsm_hun)', xtitle='Index'

stop

return, 1
end
