; ============================================================================
; Project: Cluster-II EDI
; Author : Hans Vaith, MPE Garching
; Creation Date: 08-May-2001
;
; RunEst: Estimation of runner order
;
; This function is intended to support the BESTTARG program by providing
; information about the runner order of beam hits.
; The information about the runner order is obtained by looking at long
; stretches of data, typically 1 hour. First, the gyro time Tg, obtained from
; B data, is fitted to the ToF data. An offset for B is derived
; in order to minimize the average deviation of ToF from the Tg curve. Second,
; probabilities are calculated for each ToF, for being a single- or
; multi-runner, up to the order which is passed in keyword MAXORDER(default 6).
; For a description of the returned information see the return structure
; definition in the code below.
; ============================================================================
; $Id: runest.pro,v 1.12 2004/07/06 16:50:57 hav Exp $

; $Log: runest.pro,v $
; Revision 1.12  2004/07/06 16:50:57  hav
; more dev
;
; Revision 1.11  2004/03/30 13:20:02  hav
; more dev
;
; Revision 1.10  2004/03/29 17:33:11  hav
; more dev
;
; Revision 1.8  2003/10/28 09:33:10  hav
; migration from FVEC3_T to fltarr(3,NN)
;
; Revision 1.7  2003/10/24 17:11:07  hav
; migrating to cdf vector structure ( fltarr(3,NN) )
;
; Revision 1.6  2003/10/23 18:48:32  hav
; more dev
;
; Revision 1.5  2003/09/05 08:44:54  hav
; more development
;
; Revision 1.4  2003/07/18 16:49:01  hav
; more development
;
; Revision 1.3  2003/07/11 15:06:55  hav
; more development
;
; Revision 1.2  2003/07/04 16:45:20  hav
; more development
;
; Revision 1.1.1.1  2003/03/28 18:54:35  hav
; initial import
;
; Revision 1.1.1.1  2002/03/25 08:48:19  hav
; Initial Import
;
; Revision 1.20  2002/03/22 14:34:53  hav
; bug fix: see 1.19 + -1L vs [ -1L ] in B offset determination function !!!
;
; Revision 1.18  2002/03/03 17:02:26  hav
; added B data gap detection and handling (make EDI points within gaps
; class C beams).
;
; Revision 1.17  2002/02/04 11:27:56  hav
; new keywords smooth and pl_reduce, more plotsets, rearrangement of
; input section
;
; Revision 1.16  2001/10/26  12:40:00  hav
; changed interface: q is passed (ppq); /useq1 (hav) determines if q=1
; points are used
; Determination of B offset and sfac is done with q gt 2 points only
;
; Revision 1.15  2001/10/26 11:27:15  hav
; added support for runner order estimation of q=1 data
;
; Revision 1.14  2001/09/27  15:45:51  hav
; bug fix in format of error message
;
; Revision 1.13  2001/09/25 14:43:19  hav
; bug fix: position of RE_DetermineBoffset, call axlabel() only if plotset gt 0
;
; Revision 1.12  2001/09/21 12:16:23  hav
; sfac determined automatically, but can be overridden via keyword sfac=...
;
; Revision 1.11  2001/09/17  12:35:21  hav
; bug fix: call axlabel only if plotset gt 0
;
; Revision 1.10  2001/09/17 07:57:42  hav
; keyword RELAX allows specification of value for B time begin/end condition
;
; Revision 1.9  2001/09/17 07:51:30  hav
; changed back to default SFAC=0.25; new keyword RELAX for B timedition
; text may have been screwed up, so here it is again:
; changed back to default SFAC=0.25; new keyword RELAX for B time begin/end
; condition
;
; Revision 1.8  2001/09/14  15:11:39  hav
; changed probability width factor from 0.25 to 0.5; changeable via keyword SFAC
;
; Revision 1.7  2001/09/14  10:25:24  hav
; checkin prior to possibly major changes
;
; Revision 1.6  2001/08/07 11:32:32  hav
; added equivalent single runner times-of-flight to return structure
;
; Revision 1.5  2001/07/05 14:21:03  hav
; second pline replaced by interpol
;
; Revision 1.4  2001/07/05 13:54:51  hav
; spline -> linear interpolation of B
;
; Revision 1.3  2001/06/19  13:42:56  hav
; changed keyword st to stime, use bb_sr and st_sr rather than bs.avg...
;
; Revision 1.2  2001/05/17  14:32:12  hav
; only changes which affect plotting
;
; Revision 1.1  2001/05/15  16:15:26  hav
; Initial revision
;
; ============================================================================

@readlib.pro
@boffs2.pro
@equivsrtof.pro

; ============================================================================
  PRO CheckForGaps,  b_ct,  edi_ct,  x_no_gap, x_in_gap,  $
                     debug = debug, ret=ret
; ============================================================================
; This procedure checks for gaps in the magnetic field data and
; determines if any EDI data are located within these gaps. Two index
; arrays are being returned which allow to separate the EDI data into
; groups of points which are outside of gaps and those which are
; inside the gaps and should not be used for multi-runner analysis
;
; Parameters:
;   b_ct      IN     array of B data times (ct = seconds since 1-Jan-1970)
;   edi_ct    IN     array of EDI data times ("beam times")
;   x_no_gap  OUT    array of indices for EDI data outside of gaps
;   x_in_gap  OUT    array of indices for EDI data inside of gaps
;
; side effects: none (i.e. b_ct and edi_ct are not being modified)
; ****************************************************************************

  if not keyword_set(debug) then debug = 0 else debug = 1

  nm_spacing =  0.04466d           ; spacing of FGM samples in NM1
  max_spacing =  3 * nm_spacing    ; define max allowed spacing
  N_B   =  n_elements(b_ct)        ; number of B data points
  N_EDI =  n_elements(edi_ct)      ; number of EDI data points

  actual_spacing =  b_ct[1:N_B-1] - b_ct[0:N_B-2]
  avg_spacing    =  total(actual_spacing) / (N_B-1)
  x_gap =  where( actual_spacing gt max_spacing, N_GAPS)

  if debug ne 0 then begin
     print, 'avg B data spacing    : ', avg_spacing, '   ', $
            'number of gaps        : ', N_GAPS
  endif

  x_in_gap =      -1L ; initialize array which holds the indices of EDI
                      ; points which will not be included in the runner
                      ; order analysis because they fall into a gap of
                      ; the B data

  x_no_gap =  lindgen(N_EDI)

  if N_GAPS gt 0 then begin

     x_no_gap = -1L

     if debug ne 0 then begin
        print, 'smallest gap   : ', min(actual_spacing[x_gap]), '    ', $
               'largest gap    : ', max(actual_spacing[x_gap])
     endif


     ; loop through gaps and see if there
     ; are any EDI points located inside
     ; -----------------------------------
     for gap_idx = 0, N_GAPS-1 do begin

        ct_left  = b_ct[x_gap[gap_idx]]    ; left border of gap
        ct_right = b_ct[x_gap[gap_idx]+1]  ; right border of gap

        ; for first round through loop set the right border of
        ; the previous (i.e. nonexisting) gap to some arbitrary value
        ; lower than edi_ct(0)
        if gap_idx eq 0 then ct_right_prev = edi_ct[0]-1.0 $
        else                 ct_right_prev = b_ct[x_gap[gap_idx-1]+1]


        x_outside = where( edi_ct ge ct_right_prev and edi_ct le ct_left, $
                           N_OUTSIDE)
        x_inside =  where( edi_ct gt ct_left and edi_ct lt ct_right, $
                           N_INSIDE)
;        if N_OUTSIDE gt 0 then x_no_gap = [ x_no_gap, x_outside ]
;        if N_INSIDE  gt 0 then x_in_gap = [ x_in_gap, x_inside ]
        x_no_gap = [ x_no_gap, x_outside ]
        x_in_gap = [ x_in_gap, x_inside ]

        if debug ne 0 then begin
           print, 'gap ' + strtrim(gap_idx,2) + '  ' + $
                  strmid(clutimeval2str(ct_left), 0, 10) + '  ' + $
                  strmid(clutimeval2str(ct_left),11,12) + '  -  ' + $
                  strmid(clutimeval2str(ct_right),11,12) + ' :  ' + $
                  strtrim(N_INSIDE,2) + ' EDI points'

           for j = 0, N_INSIDE-1 do begin
              print, '   ', strmid(clutimeval2str(edi_ct[x_inside[j]]),0,23)
           endfor
        endif
     endfor

     ; finally add points right of last gap to x_no_gap
     x_outside = where( edi_ct ge ct_right, N_OUTSIDE )
;     if N_OUTSIDE gt 0 then x_no_gap = [ x_no_gap, x_outside ]
     x_no_gap = [ x_no_gap, x_outside ]

     ; remove dummy elements
     ; ---------------------
     x_valid = where(x_in_gap ne -1, N_IN_GAP)
     if N_IN_GAP gt 0 then x_in_gap = x_in_gap[x_valid]

     x_valid = where(x_no_gap ne -1, N_NO_GAP)
     if N_NO_GAP gt 0 then x_no_gap = x_no_gap[x_valid]

  endif

  if x_in_gap[0] eq -1 then N_IN_GAP = 0

  if debug ne 0 then begin
;      help, edi_ct, x_no_gap, x_in_gap, N_IN_GAP
      for i=0L,N_IN_GAP-1 do print, x_in_gap[i]
      print,  'checking integrity & listing EDI points in gaps'
      if x_in_gap[0] eq -1 then begin
          print,  '   nothing to do!'
      endif else begin
          for i = 0, n_elements(x_in_gap)-1 do begin
              x = where(x_no_gap eq x_in_gap[i],  cnt)
              print, '  index ',  x_in_gap[i], ' : ',  cnt, '   ', $
                CluTimeVal2Str(edi_ct[x_in_gap[i]])
          endfor
      endelse
  endif


  if x_no_gap[0] eq -1 then begin
      ret = { status:1, $
              msg: 'RunEst B-gap detection : no EDI data outside of gaps' }
  endif else $
      ret = { status:0, msg:'' }


END

; ===========================================================================
  FUNCTION RE_DetermineBoffset, p_rtof, p_tgi, p_width, p_quality, $
                                debug=debug, usedb=usedb, db=db, $
                                xclose=xclose
; ===========================================================================
; Parameters:
;    rtof     dealiased time of flight
;    tgi      gyro time as calculated from initial bmag, interpolated to
;             times of EDI hits
;    width    half width of acceptance window around tgyro
;
; Try to find offset for bmag data
;   - calculate mean difference between tgi and rtof and use this difference
;     to correct tgi -> tgi1
;   - now consider only points closer to tgi1 than +/- width
;     correct tgi1 again using only these points : --> tgi2
;   - use the total correction tgi -> tgi2 to calculate correction for bmag
; =========================================================================

@constants.inc

if keyword_set(debug) then debug = 1 else debug = 0

ret = { status:0, msg:'Ok', db:0.0 }

q_ge2 = where( p_quality ge 2, NN)
if NN eq 0 then begin
   ret.status = 1
   ret.msg = 'No data points of q>=2 for B offset determination.'
   return,  ret
endif

rtof  = p_rtof[q_ge2]
tgi   = p_tgi[q_ge2]
width = p_width[q_ge2]


if keyword_set(usedb) then begin
   if keyword_set(db) then ret.db = db
   if debug then print, 'using passed B offset : ', ret.db
   tgi1 = tgi - tgi^2/BTG_CONV * ret.db
   xclose=where( abs(rtof-tgi) lt width )
   if xclose[0] eq -1 then begin
      ret.status=1
      ret.msg='no data points left close to tGyro (xclose[0]=-1)'
   endif
   return, ret
endif


; ---------------------
; find first correction
; ---------------------
delta1 = total(rtof-tgi) / NN
tgi1   = tgi+delta1
if debug gt 0 then print, '<rtof - tgi>                  : ', delta1, ' us'

; ---------------------------------------------------------------------
; now select only points close to tgi1 and calculate another correction
; ---------------------------------------------------------------------
xclose = where( abs(rtof-tgi1) lt width, NCLOSE )
if NCLOSE eq 0 then begin
   ret.status = 1
   ret.msg = 'Error: no points close to Tgyro found. Giving up!'
   return, ret
endif

delta2 = total(rtof(xclose)-tgi1(xclose)) / NCLOSE
if debug gt 0 then print, '<rtof(xclose) - tgi1(xclose)> : ', delta2, ' us'

delta = delta1+delta2 ; combined correction

; -----------------------------------
; use this delta to find B correction
; -----------------------------------

xclose = q_ge2[xclose]
avg_tof = total(rtof(xclose))/NCLOSE
ret.db  = -BTG_CONV/avg_tof^2 * delta

return, ret

END


; ============================================================================
  FUNCTION RunEst, $
           ; WW data and B data (hav passing mechanism)
           ; ------------------------------------------
           ds, bs, $
           ; WW data and B data (ppq passing mechanism)
           ; ------------------------------------------
           tof=tof, $
           stime=ct, $
           ctype=ctype, $
           nc = nc, mc = mc, $
           gdu = gdu, q = q, $
           st_sr=st_sr, bb_sr=bb_sr,  $
           ; configuration keywords
           ; ----------------------
           MAXORDER=MAXORDER, $
           CONF_LIM=CONF_LIM, $ ; confidence threshold
           P_LIM=P_LIM, $       ; probability limit
           SFAC=SFAC, $         ;
           smooth = smooth, $
           useq1=useq1, $
           usedb = usedb, $
           db = db, $          ; use this delta B instead of determining it !
           dtof21=dtof21, dtof12=dtof12, $
           bxoffs=bxoffs, byoffs=byoffs, bzoffs=bzoffs, $
           bfit=bfit, $
           ; other options
           ; -------------
           debug = debug, $       ; print debugging messages
           quiet=quiet, $
           writeout=writeout, $   ; write  ToF data for Edita
           hav_x_sel = hav_x_sel
; ============================================================================
; parameters:
;    ds   : WW data structure as returned by GetWWData() in readlib.pro
;    bs  : a structure containing magnetic field data
;          The structure bs is the result of function
;          GetMagData() in module readlib.pro
; ============================================================================

@constants.inc
@datast.inc

; -----------------------
; first evaluate keywords
; -----------------------
if not keyword_set(MAXORDER) then MAXORDER=6
if not keyword_set(CONF_LIM) then CONF_LIM=20.0d
if not keyword_set(P_LIM) then P_LIM=exp(-2.0) ; gt p(2*sigma)
if not keyword_set(smooth) then smooth = 1
if not keyword_set(useq1) then useq1 = 0
if not keyword_set(usedb) then usedb = 0
if not keyword_set(db)    then db = 0.0

if not keyword_set(debug) then debug=0
if not keyword_set(quiet) then quiet=0 else quiet=1

if keyword_set(tof) then hav_call=0 else hav_call=1

if MAXORDER gt 10 then begin
    print, 'maximum allowed order is 10'
    MAXORDER=10
endif

; ============================================================================
; THIS IS THE INPUT SECTION
; =============================================================================

; all of the following are arrays with as many elements as there
; are PACMO1/5 structures
; --------------------------------------------------------------
; Note, that for data prior to 27 February, m is not part of the
; PACMO1/5 structures, but the code below requires (or better assumes)
; m to be available at this time resolution. To solve this problem
; when working with the pick library take the (raw) m from the second
; header (word 1, bits 4,5), convert it ( [0,1,2,3] -> [16,8,4,2] ) and
; create an m array with PACMO1/5 resolution
; --------------------------------------------------------------

if not quiet then begin
   print, '---------------------------------------------------------------'
   print, 'Change notes: '
   print, 'keyword <relax> does no longer exist'
   print, '---------------------------------------------------------------'
endif

; ----------------------------------------------------------------------
if not hav_call then begin      ; This is the initialization branch
                              ; for the ppq passing mechanism.
                              ; Derive the size of the arrays from
                              ; tof

    if debug gt 0 then print, 'ppq passing mechanism.'

    if keyword_set(bfit) then $
      print, 'B fitting not possible with ppq passing mechanism'

    N_ALL = n_elements(tof)
    ww = replicate( {WW_T}, N_ALL )

    ww.ct  = ct
    ww.tof = tof
    ww.nc  = nc
    ww.mc  = mc
    ww.q   = q
    ww.ctype = ctype
    ww.gd    = 12
    x = where(gdu eq 2) ; 'gdu' means 'gun' in the ppq passing mechanism !!!
    if x[0] ne -1 then ww[x].gd = 21

    bd_bmag = bb_sr             ; get B magnitude and associated time tags
    bd_ct   = st_sr

    sc_id =  0                  ; dummy

; ----------------------------------------------------------------------
endif else begin                ; This is the initialization branch
                                ; if parameters have been passed via
                                ; d,bs (hav passing mechanism)

    if (ds.hdr.sc_id ne bs.hdr.sc_id) then $
      message, 'd and bs data are not from same SC!'

    if ds.hdr.status ne 0 then message, ds.hdr.msg
    if bs.hdr.status ne 0 then message, bs.hdr.msg

    sc_id = ds.hdr.sc_id
    ww = ds.data

    if debug gt 0 then print, 'hav passing mechanism.'

    if not keyword_set(bxoffs) then bxoffs = 0.
    if not keyword_set(byoffs) then byoffs = 0.
    if not keyword_set(bzoffs) then bzoffs = 0.
    if not keyword_set(dtof21) then dtof21 = 0.
    if not keyword_set(dtof12) then dtof12 = 0.

    x21 = where(ww.gd eq 21, complement=x12)
    if x21[0] ne -1 then ww[x21].tof = ww[x21].tof + dtof21
    if x12[0] ne -1 then ww[x12].tof = ww[x12].tof + dtof12

    if keyword_set(bfit) then begin
        b_fitted = GetFittedB(bs)
        bd_bmag = sqrt( (b_fitted.x+bxoffs)^2 + $
                        (b_fitted.y+byoffs)^2 + $
                        (b_fitted.z+bzoffs)^2 )
        bd_ct   = b_fitted.ct
    endif else begin
        bd_bmag = sqrt( (bs.data.val[0]+bxoffs)^2 + $
                        (bs.data.val[1]+byoffs)^2 + $
                        (bs.data.val[2]+bzoffs)^2 ) ; Bmag data
        bd_ct   = bs.data.ct          ; associated times
    endelse

    ; select data according to quality
    ; --------------------------------
    if useq1 eq 0 then $
      x = where(ww.q ge 2 and ww.mch eq 7 and ww.ovfl eq 0, N_ALL) $
    else $
      x = where((ww.q eq 1 or (ww.q ge 2 and ww.mch eq 7)) and $
                ww.ovfl eq 0, N_ALL)

    if N_ALL eq 0 then $
      return, { status:1,  msg:'Error: no data of selected quality!' }

    ww = ww[x]

    hav_x_sel = x

endelse
; ----------------------------------------------------------------------


; smoothing of magnetic field data
; --------------------------------
if smooth gt 1 then bd_bmag = smooth(bd_bmag, smooth)


; ============================================================================
; END OF INPUT SECTION
; do not change anything below this line (unless you know what you're doing)
; ============================================================================

; -------------------------------------------------------------------------
; return structure definition
; ---------------------------

data = replicate( {RUNEST_T}, N_ALL )

ret = { data:data, $            ; runner data array
        status:0, $             ; return status: 0=ok, other = error
        msg:'', $               ; return message (error details)
        sc_id:0, $
        db:0.0, $
        smooth:smooth, $
        maxorder:MAXORDER, $
        p_lim: P_LIM, $
        conf_lim: CONF_LIM, $
        bxoffs: bxoffs, $
        byoffs: byoffs, $
        bzoffs: bzoffs, $
        dtof21: dtof21, $
        dtof12: dtof12, $
        sfac: 0.0 }
; --------------------------------------------------------------------------


; sort in time
; ------------
sortindex = sort(ww.ct)
ww_all = ww[sortindex]

; check number of data points
; ---------------------------
if debug gt 0 then print, 'number of EDI data points: ', N_ALL
if N_ALL lt 2 then begin
   ret.status = 1
   ret.msg = 'Error: less than two data points of selected quality!'
   goto, undo_mods
;   return, ret
endif


; calculate tchip and tcode (assuming short code: 15 chips)
; ---------------------------------------------------------
nchips = intarr(N_ALL) + 15 ; number of chips in Pseudo Noise Code
x = where(ww_all.ctype eq 1) ; see where long code has been used
if x[0] ne -1 then nchips[x] = 127 ; set number of chips for long code
tchip_all = 2.0d^(-24)*2*ww_all.mc*ww_all.nc * 1.0d6 ; chip length in micro-s
tcode_all = nchips*tchip_all               ; code length in micro-s



; check the B data for gaps
; =========================
CheckForGaps, bd_ct, ww_all.ct, $   ; IN
              x_no_gap, x_in_gap, $ ; OUT
              debug = debug, $
              ret=rettmp
if rettmp.status ne 0 then begin
    ret=rettmp
    goto, undo_mods
endif

; tmp : fake gap detection : no gaps
; ----------------------------------
;print, '*** Attention ***  faking gap detection: no gaps'
;x_in_gap = -1L
;x_no_gap = lindgen(N_ALL)

; select the data outside of B gaps
; ---------------------------------
ww    = ww_all[x_no_gap]  ; ----- THIS OVERWRITES THE ORIGINAL WW ARRAY ---
tchip = tchip_all[x_no_gap]
tcode = tcode_all[x_no_gap]
N_NO_GAP =  n_elements(x_no_gap)


; interpolate gyro time to edi data times : --> tgi
; then dealias EDI ToF values
; -------------------------------------------------
if debug gt 0 then print, 'Interpolating Tg data to times of EDI hits...'
tgi = interpol(BTG_CONV/bd_bmag, bd_ct-ww[0].ct, ww.ct-ww[0].ct)

if debug gt 0 then print, 'Moving data into tube +/- 0.5*tcode around tgi'
n = floor( (tgi-ww.tof)/tCode + 0.5 )
rtof = ww.tof + n*tCode


f = 1.0    ; use only points closer to tgi than f*tchip

; Determine B offset
; ------------------
dbs = RE_DetermineBoffset(rtof, tgi, f*tchip, ww.q, $        ; IN
                          debug=debug, usedb=usedb, db=db, $ ; IN
                          xclose=xclose)                     ; OUT
if dbs.status ne 0 then begin
    ret.status = dbs.status
    ret.msg = dbs.msg
    goto, undo_mods
    ; return, ret
endif

; recalculate Tgyro with new B offset and dealias ToF
; ---------------------------------------------------
tgi3 = interpol(BTG_CONV/(bd_bmag+dbs.db), bd_ct-ww[0].ct, ww.ct-ww[0].ct)
n = floor( (tgi3-rtof)/tCode  +  0.5 )
rtof = rtof + n*tCode

if debug gt 0 and usedb eq 0 then print, 'determined dB to : ' + strtrim(dbs.db,2)

; test
; ----
if usedb eq 0 and debug gt 0 then $
   dummy = RE_DetermineBoffset(rtof, tgi3, f*tchip, ww.q, debug=debug)


; calculate standard deviation of tof around tgi3
; -----------------------------------------------
;help, rtof, xclose
stdev = sqrt( total((rtof[xclose]-tgi3[xclose])^2) / n_elements(xclose) )
avg_tchip = total(tchip[xclose]) / n_elements(xclose)
if debug gt 0 then begin
    print, 'analysis of q>=2 data points:'
    print, 'standard deviation (us) : ', stdev
    print, 'average chip width (us) : ', avg_tchip
    print, 'ratio stdev/tchip       : ', stdev / avg_tchip
endif

if not keyword_set(SFAC) then begin
   sfac = stdev / avg_tchip
   if debug gt 0 then print, 'using sfac of ', sfac
endif


; ======================================================================
; Calculate distance of tof from n * tgyro (multiples of tgi3)
; ======================================================================
dist = fltarr(MAXORDER, N_NO_GAP)
for i=0,MAXORDER-1 do begin
   dist[i,*] = (rtof - (i+1)*tgi3) mod tCode
   x = where(dist[i,*] gt 0.5*tCode)
   if x[0] ne -1 then dist[i,x] = dist[i,x] - tCode[x]
   x = where(dist[i,*] lt -0.5*tCode)
   if x[0] ne -1 then dist[i,x] = dist[i,x] + tCode[x]
endfor

; ======================================================================
; assign order assuming gaussian probability distributions
; and use p_max^2/sum(others) as a confidence criterion
; ======================================================================
stdev = SFAC*tchip ; this is the stdev of the gaussian distributions

p_run = fltarr(MAXORDER, N_NO_GAP) ; define some arrays
p_tot = fltarr(N_NO_GAP)
p_max = p_tot
order = intarr(N_NO_GAP)

; calculate probabilities
; -----------------------
for i=0,MAXORDER-1 do begin
   p_run[i,*] = exp(-0.5*(dist[i,*]/stdev)^2)

   x = where(abs(dist[i,*]) gt tchip)  ; is this reasonable to do ?
   if x[0] ne -1 then p_run[i,x] = 0.0

   p_tot = p_tot + p_run[i,*]
   x = where(p_run[i,*] gt p_max)
   if x[0] ne -1 then begin
      p_max[x] = p_run[i,x]
      order[x] = i+1
   endif
endfor

x = where(p_max lt P_LIM)    ; set order to 'unknown' if p is below threshold
if x[0] ne -1 then order[x] = 0

; --------------------------------------------------------
; now calculate a confidence level for the runner order which
; has the highest probability (some sort of SNR^2)
; set runner order to 'undetermined' (0) where the confidence level is
; below the threshold
; --------------------------------------------------------
p_other = p_tot - p_max
x = where(p_other lt 1.d-10)
if x[0] ne -1 then p_other[x] = 1.0d-10 ; avoid numerical problems
conf = p_max^2 / p_other ; confidence level

x = where(conf lt CONF_LIM)  ; set order to 'unknown' if c is below threshold
if x[0] ne -1 then order[x] = 0

; -------------------------------------------------------------------------
; Here is what the function returns in case everything is ok
; -------------------------------------------------------------------------

  ret.status = 0 ; Ok
  ret.msg    = 'OK'
  ret.sc_id  = sc_id
  ret.sfac   = sfac
  ret.db     = dbs.db

  ret.data.beamtime = ww_all.ct
  ret.data.ct       = ww_all.ct
  if not hav_call then ret.data.gdu      = gdu[sortindex]
  if hav_call then ret.data.gd = ww_all.gd
  ret.data.q        = ww_all.q
  ret.data.en       = ww_all.en
  ret.data.ph       = ww_all.ph
  ret.data.th       = ww_all.th
  ret.data.tchip    = tchip_all
  ret.data.tcode    = tcode_all
  ret.data.nc       = ww_all.nc
  ret.data.sortindex = sortindex

  ; transform gun2 firing directions into gdu1 system
  ; NOTE: makes sense only for hav_call=1 because otherwise
  ; ret.data.gd is not set correctly !!!
  x = where(ret.data.gd eq 21)
  if x[0] ne -1 then begin
      ret.data[x].ph = 360. - ret.data[x].ph
      ret.data[x].th = 180. - ret.data[x].th
  endif

  ; fill in the data for points outside of B gaps
  ; ---------------------------------------------
  ret.data[x_no_gap].tg0 = tgi
  ret.data[x_no_gap].tg  = tgi3
  ret.data[x_no_gap].tof = rtof
  ret.data[x_no_gap].runorder = order

  for i=0,MAXORDER-1 do begin
     x = where(p_run[i,*] ge P_LIM)
     if x[0] ne -1 then ret.data[x_no_gap[x]].flag[i] = 1
     tmp = dblarr(N_NO_GAP)
     tmp[*] = p_run[i,*]
     ret.data[x_no_gap].prob[i] = tmp

     esr_ret =  EquivSrToF(rtof, i+1, tgi3, tcode)
     if esr_ret.status ne 0 then begin
        ret.status = esr_ret.status
        ret.msg    = esr_ret.msg
        goto, undo_mods
        ;return, ret
     endif
     ret.data[x_no_gap].estof[i] = esr_ret.estof
  endfor

  ; fill in the data for points within B gaps (make them class C beams)
  ; -------------------------------------------------------------------
  if x_in_gap[0] ne -1 then begin
     ret.data[x_in_gap].tg  = -1.0
     ret.data[x_in_gap].tof = -1.0
     ret.data[x_in_gap].runorder = 0
     ret.data[x_in_gap].flag[*] = 0
     ret.data[x_in_gap].prob[*] = 0.0
     ret.data[x_in_gap].estof[*] = -1.0
  endif

; --------------------------------------------------------------------------

  if keyword_set(writeout) then begin
      if size(writeout,/type) eq 7 then filename = '~/' + writeout $
      else begin
         date_time = clutimeval2str(ret.data[0].beamtime)
         filename = '~/cl' + strtrim(ds.hdr.sc_id,2) + '_' + $
                    strmid(date_time,0,10) + 't' + strmid(date_time,11,2) + $
                    '.eg'
      endelse


      openw, fp, filename, /get_lun

      for i=0L,n_elements(ret.data)-1 do begin
          this = ret.data[i]
          if this.runorder gt 0 then begin
              printf, fp, format='(F15.3,F10.3)', this.beamtime, $
                this.estof[this.runorder-1]
          endif
      endfor

      free_lun, fp
      print, 'ToF data written to ' + filename
  endif

; --------------------------------------------------------------------------

undo_mods: ; undo modification of keywords if ppq call mechanism was used
           ; well, not all modifications: return the input data SORTED!!!
  if not hav_call then begin
      tof   = ww_all.tof
      ct    = ww_all.ct
      ctype = ww_all.ctype
      nc    = ww_all.nc
      mc    = ww_all.mc
      gdu   = intarr(n_elements(ww_all)) + 1
      x = where(ww_all.gd eq 21)
      if x[0] ne -1 then gdu[x] = 2 ; --> gdu means gun !! ?
      q     = ww_all.q
      th    = ww_all.th         ; these three are actually not part
      ph    = ww_all.ph         ; of the keywords, but nevermind...
      en    = ww_all.en         ;
  endif


; -------------------------------------------------------------------------

  return, ret

END

