; +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; $Id: caa_editm.pro,v 1.9 2006/04/04 18:39:01 hav Exp $
; +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

@readlib.pro
@gaps.pro
@xout.pro

; ===========================================================================
  FUNCTION create_header_clist, sc, date, version, gendate, program_version
; ===========================================================================
;
; ***************************************************************************

   if n_params() ne 5 then begin
      message, 'Expecting five parameters (sc,date,version,gendate,program_version)', /cont
      retall
   endif

   name = 'C'+sc+'_CQ_EDI_CLIST__'+date+'_V'+version


   start_year  = fix(strmid(date,0,4))
   start_month = fix(strmid(date,4,2))

   end_month = start_month + 1
   end_year = start_year
   if end_month gt 12 then begin
      end_month = 1
      end_year = end_year + 1
   endif

   iso_time_range = string(format='(I4.4,"-",I2.2,"-01T00:00:00Z/",'+$
                                   'I4.4,"-",I2.2,"-01T00:00:00Z")', $
                           start_year, start_month, end_year, end_month)


   header_lines = $
      [ 'FILE_NAME="'+name+'.cef"', $
        'FILE_FORMAT_VERSION="CEF-2.0"', $
        '!', $
        'START_META = LOGICAL_FILE_ID', $
        ' ENTRY = "'+name+'"', $
        'END_META = LOGICAL_FILE_ID', $
        '!', $
        'START_META = VERSION_NUMBER', $
        ' ENTRY = "'+version+'"', $
        'END_META = VERSION_NUMBER', $
        '!', $
        'START_META = FILE_TIME_SPAN', $
        ' VALUE_TYPE = ISO_TIME_RANGE', $
        ' ENTRY = '+iso_time_range, $
        'END_META = FILE_TIME_SPAN', $
        '!', $
        'START_META = GENERATION_DATE', $
        ' VALUE_TYPE = ISO_TIME', $
        ' ENTRY = '+gendate, $
        'END_META = GENERATION_DATE', $
        '!', $
        'START_META = FILE_CAVEATS', $
        ' ENTRY = "'+program_version+'"', $
        'END_META = FILE_CAVEATS', $
        '!', $
        '! include CaveatList (DATASET) HEADER File for Cluster-'+sc, $
        '! with variable definitions, metadata_type and _version', $
        'include="C'+sc+'_CQ_EDI_CLIST.ceh"', $
        '!', $
        'DATA_UNTIL=EOF', $
        '!' ]

   return, header_lines

END


; ===========================================================================
  FUNCTION CAA_EDITM_MERGE, list, gapspec=gapspec, debug=debug
; ===========================================================================
; 'list' is an array of type CAA_EDI_TMIV_T (defined in procedure CAA_EDITM)
;
; This procedure will merge adjacent time intervals in 'list' that are
; separated by less than the gap specification and where EDI is in the same
; science mode
; ***************************************************************************

COMMON CAA_EDITM, TMR

  if not keyword_set(debug) then debug = 0 else debug = 1
  if not keyword_set(gapspec) then gapspec = 3.5*TMR
 
  ; Copy input data in order not to overwrite,
  ; then sort by ascending left border value
  ; ------------------------------------------
  out = list
  out = out[sort(out.ct_beg)]

  NN = n_elements(out)

  ; Make sure there is more than one interval
  ; -----------------------------------------
  if NN eq 1 then begin
     if debug then message, 'single element array --> will be returned as is', /cont
     return, out
  endif

  ; Find indices of intervals that are close enough for merging. This is
  ; done in a loop in order to account for the possibility that several
  ; intervals need to be merged into a single one (i.e. the body of the
  ; loop performs only two-interval merging, and multiple intervals are
  ; merged by iteration)
  ; The algorithm for two-interval merging first finds the indices of
  ; all intervals that are close enough for merging and then copies the
  ; left border value (ct_beg) of the first interval to the left border
  ; value of the second interval. The first interval can then be removed.
  ; The removal is done after the loop has been quit, which happens if no
  ; more changes to the left border values have occurred.
  ; --------------------------------------------------------------------
  if debug then message, 'about to enter interval start shifting loop', /cont
  itercnt = 0L
  repeat begin
     itercnt = itercnt + 1
     if debug then message, 'iteration count: ' + strtrim(itercnt,2), /cont

     a = out[0:NN-2]  &  b = out[1:NN-1]
     x = where(b.ct_beg-a.ct_end lt gapspec and a.modid eq b.modid, cnt, complement=xin)
     if cnt eq 0 then begin
        if debug then message, '   no gaps to be removed --> leaving loop and returning', /cont
        return, out
     endif

     if debug then message, '   shifting interval begin times, #occurrences : ' + strtrim(cnt), /cont
     newout = out
     newout[x+1].ct_beg = newout[x].ct_beg
     xdiff = where(newout.ct_beg ne out.ct_beg, diff_cnt)
     if debug then message, '   number of differences found : ' + strtrim(diff_cnt,2), /cont
     out = newout
     if debug then help, out

  endrep until diff_cnt eq 0

  ; print the indices of the intervals that will be removed
  if debug then begin
     message, 'behind loop. xout = '+strjoin(x), /cont
     message, 'behind loop. xin = '+strjoin(xin), /cont
  endif

  xin = [ xin, NN-1 ] ; add last element (because it will never be removed, according to the algorithm)

  out = out[xin] ; interval removal: keep only those that have a neighbor
                 ; that either has a different science mode or is sufficiently
                 ; far away to be considered unsuitable for merging

  return, out

END



; ===========================================================================
  PRO CAA_EDITM, sc_id, year, month, version, day=day, ndays=ndays
; ===========================================================================

COMMON CAA_EDITM, TMR

  program_version = 'CAA_EDITM_V1_03  2006-11-30T08:30:00Z'

  ; -------------------------------------------------------------------------
    filter_for_modes = [ 5, 21 ] ; all other modes will be discarded in the
                                 ; final list
                                 ; 5  = Windshield Wiper Mode
                                 ; 21 = Ambient Electron Mode (Ambient Geos)

    ; filter_for_modes = indgen(64) ; use this to effectively avoid filtering

    gap_spec = 4       ; a gap of >= gap_spec TMRs will be considered
                                 ; a gap in telemetry
  ; -------------------------------------------------------------------------


  SECS_PER_DAY = 86400.d
  TMR = 5.152221d

  ; Make a few checks on the parameters
  ; -----------------------------------
  if n_params() ne 4 then begin
     print, 'Call syntax: caa_editm, sc, year, month, version, day=day, ndays=ndays'
     retall
  endif

  ; make sure we are using integers internally
  i_sc    = fix(sc_id)
  i_year  = fix(year)
  i_month = fix(month)

  i_version = fix(version)
  if i_version lt 0 or i_version gt 99 then begin
     message, 'Error: version must be between 0 and 99', /cont
     retall
  endif

  test_flag = 0

  if not keyword_set(day) then i_day = 1 $
  else begin
     i_day = fix(day)
     test_flag = 1
  endelse

  s_sc      = strtrim(i_sc,2)
  s_year    = strtrim(i_year,2)
  s_month   = string(format='(I2.2)',i_month)
  s_version = string(format='(I2.2)',i_version)
  s_day     = string(format='(I2.2)',i_day)

  iso_time_string = s_year + "-" + s_month + "-" + s_day
  ct_begin = CluTimeStr2Val(iso_time_string)

  if not keyword_set(ndays) then begin
     end_month = i_month + 1
     end_year = i_year
     if end_month gt 12 then begin
        end_month = 1
        end_year = end_year + 1
     endif

     iso_time_string = string(format='(I4.4,"-",I2.2,"-01")', end_year, end_month)
     ct_end = CluTimeStr2Val(iso_time_string) - SECS_PER_DAY
  endif else begin
     test_flag = 1
     ct_end   = ct_begin + (fix(ndays)-1) * SECS_PER_DAY  ; (ndays-1) so that a total of
                                                          ; 'ndays' days get processed
  endelse


  ; -------------------------------------------------------------------------

  ct_begin = ct_begin + 12.*3600 ; noon


  ; Define prototype for output data
  ; --------------------------------
  proto = { CAA_EDI_TMIV_T, ct_beg:0.0d, ct_end:0.0d, modid:0 } ; modid=-1 means data gap

  ct_now = ct_begin - SECS_PER_DAY
  first = 1

  out = proto ; define initial dummy element

  ; Loop through all days within specified interval
  ; -----------------------------------------------
  while 1 eq 1 do begin

     ct_now = ct_now + SECS_PER_DAY
     if ct_now gt (ct_end+SECS_PER_DAY) then break

     ymd = strmid(CluTimeVal2Str(ct_now),0,10)
     print, 'SC ' + strtrim(sc_id,2) + '  day ' + ymd
     fn = GetFileName(i_sc, ymd=ymd, /latest)
     if fn.status ne 0 then begin
        print, '   ' + fn.msg
        continue
     endif

     ; the modified call interface of GetHkData() in readlib.pro requires the specification
     ; of a time interval (ctr) when the keyword 'file' is used
     ctr_tmp = CluTimeStr2Val(ymd) + [0,SECS_PER_DAY]
     hk = GetHkData(i_sc, ctr=ctr_tmp, file=fn.filename,/quiet) ; if we use year,month,day then the time interval
                                                   ; will be extended, causing data from adjacent days to be read
                                                   ; (although the daily MSFs do contain a small amount of data
                                                   ; from the previous and next day). The additionally opened files
                                                   ; slow down the process. Therefore the filename is directly
                                                   ; specified.
     if hk.hdr.status ne 0 then begin
        print, '   ' + hk.hdr.msg
        continue
     endif

     ; Strip off all data outside of current day (MSFs do contain such data!)
     ; -----------------------------------------
     ctr = CluTimeStr2Val(ymd) + [0,SECS_PER_DAY]
     x=where(hk.data.ct ge ctr[0] and hk.data.ct lt ctr[1], cnt)
     if cnt eq 0 then begin
        print, '   no data after stripping data from neighbor days'
        continue
     endif
     data = hk.data[x]

     ; Create array of occurring science modes (each mode only once)
     ; ---------------------------------------
     modid = ishft(data.data[6],-10)
     all_modid = modid[uniq(modid, sort(modid))]

;     help, all_modid & print, all_modid



     ; Store intervals of EDI science modes
     ; ------------------------------------
     for i=0L,n_elements(all_modid)-1 do begin ; loop through all occurring science modes
        cur_modid = all_modid[i]
;        print, 'Current science mode : ' + strtrim(cur_modid,2)
        xin = where(modid eq cur_modid) ; find indices of tm packets for current science mode
        N_IN = n_elements(xin)
        ivs = GetGaps(data[xin].ct, TMR*1.5) ; find gaps

        if ivs[0].ct_beg lt 0 then NN = 1 $
        else                       NN = n_elements(ivs) + 1

        for j=0L,NN-1 do begin
           if j eq 0 then proto.ct_beg = data[xin[0]].ct $
           else           proto.ct_beg = ivs[j-1].ct_end

           if j lt NN-1 then proto.ct_end = ivs[j].ct_beg $
           else              proto.ct_end = data[xin[N_IN-1]].ct

           proto.modid = cur_modid

           out = [ out, proto ]
;           PrintTimeInterval, [proto.ct_beg,proto.ct_end], leading='   '

        endfor ; loop through intervals

     endfor ; loop through EDI science modes

  endwhile ; date loop

  if n_elements(out) eq 1 then begin
     message, 'No EDI housekeeping data found!', /cont
     retall
  endif

  out = out[1:n_elements(out)-1] ; strip off initial dummy element
  out = out[sort(out.ct_beg)]    ; sort intervals

  ; Remove unwanted modes
  ; ---------------------
  N_MODES = n_elements(filter_for_modes)
  xin = -1L
  for i=0L,N_MODES-1 do begin
     x = where(out.modid eq filter_for_modes[i], cnt)
     if cnt gt 0 then xin = [ xin, x ]
  endfor
  xin = xin[1:n_elements(xin)-1] ; remove leading dummy element
  xin = xin[sort(xin)]           ; sort indices
  out = out[xin]                 ; use index array to remove unwanted modes


  ; See if we need to merge intervals (should occur only at day boundaries)
  ; -----------------------------------------------------------------------
  out = CAA_EDITM_Merge(out, gapspec=(gap_spec+0.5)*TMR)

  ; write to screen
  ; ---------------
;  print, '------------------------'
;  print, 'after merging'
;  for i=0,n_elements(out)-1 do begin
;     PrintTimeInterval, [out[i].ct_beg,out[i].ct_end], leading='   ', trailing='   mode: ' + strtrim(out[i].modid,2)
;  endfor

   ; Prepare formatted output
   ; ------------------------
   NN = n_elements(out)
   lines = strarr(NN)
   for i=0L,NN-1 do begin
      lines[i] = strmid(CluTimeVal2Str(out[i].ct_beg),0,23) + "Z/" + $
                 strmid(CluTimeVal2Str(out[i].ct_end + TMR),0,23) + "Z, "      ;changed by eg 14/10/2005
      case out[i].modid of
         5 : lines[i] = lines[i] + '"WW"'
        21 : lines[i] = lines[i] + '"AE"'
        else : begin
           message, 'Unexpected value for modid : ' + strtrim(out[i].modid,2), /cont
           retall
        end
      endcase
   endfor

   ; Construct file name for output
   ; ------------------------------
   date = s_year + s_month + '00'
   out_filename = 'C'+s_sc+'_CQ_EDI_CLIST__'+s_year+s_month+'00_V'+s_version+'.cef'
   if test_flag eq 1 then out_filename = out_filename + "-test"

   iso_generation_date = strmid(CluTimeVal2Str(systime(1)),0,19) + 'Z'
   header_lines = create_header_clist(s_sc, date, s_version, $
                                      iso_generation_date, program_version)

   ; Write data to output file
   ; -------------------------
   MyFwrite, out_filename, [ header_lines, lines ]

   print, 'Data written to ' + out_filename

END

; ===========================================================================
  PRO CAA_EDITM_TEST
; ===========================================================================

COMMON CAA_EDITM, TMR

  TMR = 5.152221d

  ; Define prototype for output data
  ; --------------------------------
  proto = { CAA_EDI_TMIV_T, ct_beg:0.0d, ct_end:0.0d, modid:0 } ; modid=-1 means data gap

  out = replicate( proto, 8 )

  ct0 = CluTimeStr2Val('2001-07-04t')
  out[0].ct_beg = ct0      & out[0].ct_end = ct0 + 60  & out[0].modid = 5
  out[1].ct_beg = ct0+70   & out[1].ct_end = ct0 + 80  & out[1].modid = 5
  out[2].ct_beg = ct0+81   & out[2].ct_end = ct0 + 90  & out[2].modid = 5
  out[3].ct_beg = ct0+100  & out[3].ct_end = ct0 + 120 & out[3].modid = 5
  out[4].ct_beg = ct0+121  & out[4].ct_end = ct0 + 140 & out[4].modid = 5
  out[5].ct_beg = ct0+141  & out[5].ct_end = ct0 + 160 & out[5].modid = 5
  out[6].ct_beg = ct0+180  & out[6].ct_end = ct0 + 190 & out[6].modid = 5
  out[7].ct_beg = ct0+191  & out[7].ct_end = ct0 + 200 & out[7].modid = 5

  ; write to screen
  ; ---------------
  print, 'before merging'
  help, out
  for i=0,n_elements(out)-1 do begin
     PrintTimeInterval, [out[i].ct_beg,out[i].ct_end], leading='   ', trailing='   mode: ' + strtrim(out[i].modid,2)
  endfor
  print, '------------------------'

  ; See if we need to merge intervals (should occur only at day boundaries)
  ; -----------------------------------------------------------------------
  debug = 0
  out = CAA_EDITM_Merge(out, debug=debug)

  ; write to screen
  ; ---------------
  print, '------------------------'
  print, 'after merging'
  help, out
  for i=0,n_elements(out)-1 do begin
     PrintTimeInterval, [out[i].ct_beg,out[i].ct_end], leading='   ', trailing='   mode: ' + strtrim(out[i].modid,2)
  endfor

END
