/******************************************************************************
* Copyright 1996-2013 United States Government as represented by the
* Administrator of the National Aeronautics and Space Administration.
* All Rights Reserved.
******************************************************************************/
/*
 * This program is for converting a cdf.dtd or cdf.xsd compliant XML file 
 * to a CDF file. The SAX parser is used for parsing the document.
 */ 
import java.lang.reflect.*;
import gsfc.nssdc.cdf.*;
import gsfc.nssdc.cdf.util.*;

import java.io.*; 
import java.util.*;
import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler; 
import org.xml.sax.helpers.*;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;

public class CDFML2CDF extends DefaultHandler implements CDFConstants {
    private static final int NONE = 0, GAttributes = 1, 
			     VAttributes = 2, VarRecordData = 3, VarElementData = 4;
    private static String[] CDFelements = { // in order of frequency in cdfml file
                "record", "element", "entry", "variable", "attribute", "cdfVarInfo", 
		"cdfVAttributes", "cdfVarData", "cdfGAttributes", "CDF", "cdfFileInfo"};
    private static String newCDFName = null;
    private static boolean removeOldCDF = false;
    private static boolean loadData = true;
    private static boolean backward = false;
    private static boolean progressReport = false;
    private static boolean validateCDFML = false;
    private static boolean entryNumFlag;
    private static String cdfmlFileName;
    private static String outputDir = null;

    private CDF cdf;
    private Variable var;
    private Entry entry;
    private Attribute attribute;
    private String cdfName, varName, attrName;
    private String elementDelimiter, indexString;
    private long attrNum, varNum, scope, varDataType,
                 entryDataType, numElements, recNum;
    private long[] dimSizes, dimVariances, dimIndex;
    private long[] dimIndices, dimCounts, dimIntervals;
    private long recCount, recInterval;
    private long dim, recVariance;
    private int phyDims, physicalSize;
    private Object dataObject;
    private long entryNum = -1;
    private int inScope;
    private boolean entryBlock, recordBlock, elementBlock, processElement;
    private Stack stack;
    private CDFML2CDF aXML2CDF;
    private StringBuffer dataBuffer;

    public static void main(String argv[]) {

        if (argv.length < 1) CDFML2CDF.howto();
        
	int i = 0;
	while (i < argv.length) {
	  if (argv[i].charAt(0) == '-' || 
	      (i != argv.length-1 && argv[i].charAt(0) == '/')) {
	    if (argv[i].substring(1).equalsIgnoreCase("delete")) {
	      removeOldCDF = true;
	      i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("nodelete")) {
              removeOldCDF = false;
              i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("validate")) {
              validateCDFML = true;
              i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("novalidate")) {
              validateCDFML = false;
              i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("progress")) {
              progressReport = true;
              i++;
	    } else if (argv[i].length() > 4 && 
	               argv[i].substring(1,4).equalsIgnoreCase("cdf")) {
	      newCDFName = argv[i].substring(5);
	      i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("data")) {
              loadData = true;
              i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("nodata")) {
              loadData = false;
              i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("backward")) {
              backward = true;
              i++;
            } else if (argv[i].substring(1).equalsIgnoreCase("nobackward")) {
              backward = false;
              i++; 
	    } else 
	      CDFML2CDF.howto();
	  } else 
	    cdfmlFileName = argv[i++];
	}
	if (cdfmlFileName == null) CDFML2CDF.howto();
	CDFML2CDF.checkCDFMLFile(cdfmlFileName);
	if (newCDFName != null) CDFML2CDF.checkCDFFile(newCDFName);
	CDFML2CDF tmp = new CDFML2CDF();

    }

    CDFML2CDF() {

	aXML2CDF = this;
	System.err.println("Start processing");
	aXML2CDF.startProcessing();
    }

    /**
     * Checks whether the specified xml document exists. 
     */

    private static void checkCDFMLFile(String cdfml) {

        File cdfmlFile = new File (cdfml);
        if (!cdfmlFile.exists()) {
          System.err.println("*** Error: cdfml file:"+cdfmlFile+" does not exist! ***");
          System.exit(0);
	}

    }

    /**
     * Checks whether the passed file is a directory name or a CDF file.
     * If deleting the existing CDF file is specified, that file will be
     * deleted. 
     */

    private static void checkCDFFile(String cdfFile) {

	if (cdfFile == null) return;
	File aFile = new File(cdfFile);
	if (aFile.isDirectory()) {
	  if (cdfFile.endsWith(System.getProperty("file.separator")))
	    outputDir = cdfFile;
	  else
	    outputDir = cdfFile + System.getProperty("file.separator"); 
	  return;
	}
        String file1 = null, file2 = null;
        if (cdfFile.indexOf(".cdf") == -1 && cdfFile.indexOf(".CDF") == -1) {
          file1 = new StringBuffer(cdfFile).append(".cdf").toString();
          file2 = new StringBuffer(cdfFile).append(".CDF").toString();
        } else {
          file1 = cdfFile;
        }
        File cdfFile1 = new File(file1);
        boolean check2 = true;
        if (cdfFile1.exists()) {
          if (removeOldCDF) {
            cdfFile1.delete();
            check2 = false;
          } else {
            System.err.println("\nError... CDF: "+file1+" already exists...");
            System.exit(0);
          }
        }
        if (check2 && file2 != null) {
          check2 = false;
          File cdfFile2 = new File(file2);
          if (cdfFile2.exists()) {
            if (removeOldCDF) {
              cdfFile2.delete();
            } else {
              System.err.println("\nError... CDF: "+file2+" already exists...");
              System.exit(0);
            }
          }
        }
    }

    /**
     * Set up the necessary steps and ready to scan and process the xml document.
     */

    private void startProcessing() {

	dimSizes = null;
	dimVariances = null;
	dimIndices = null;
	inScope = NONE; 
	dataBuffer = new StringBuffer();
	stack = new Stack();
        Throwable t1;

        try {
	    XMLReader parser = XMLReaderFactory.createXMLReader();
	    if (validateCDFML)
	      parser.setFeature("http://xml.org/sax/features/validation", true);
	    parser.setErrorHandler(aXML2CDF);
            parser.setContentHandler(aXML2CDF);
	    parser.parse(new InputSource(CDFML2CDF.fileToURL(new File(cdfmlFileName))));
        } catch (Throwable t) {
            /* One more check: see if the Sun's JDK V 1.4 is used. */
            System.setProperty("org.xml.sax.driver", 
                               "org.apache.crimson.parser.XMLReaderImpl");
            try {
              XMLReader parser = XMLReaderFactory.createXMLReader();
              if (validateCDFML)
                parser.setFeature("http://xml.org/sax/features/validation", true);
              parser.setErrorHandler(aXML2CDF);
              parser.setContentHandler(aXML2CDF);
              parser.parse(new InputSource(CDFML2CDF.fileToURL(new File(cdfmlFileName))));
           } catch (Throwable t2) {
              System.err.println(" ");
              t2.printStackTrace();
              System.exit(0);
           }
/*
        } catch (Throwable t) {
	    System.err.println(" ");
            t.printStackTrace();
	    System.exit(0);
*/
        }
        StringBuffer fName = new StringBuffer(cdfName);
        if (cdfName.indexOf(".cdf") == -1 && cdfName.indexOf(".CDF") == -1)
          fName.append(".cdf");
        System.err.println("Completed!  CDF file: "+fName.toString()+" is created.");

        System.exit(1);

    }     

    //===========================================================
    // SAX DocumentHandler methods
    //===========================================================     

    public void startDocument() {

    }     

    public void endDocument() {

    }     

    public void startElement(String namespaceURI,
                             String lName, // local name
                             String qName, // qualified name
                             Attributes attrs) throws SAXException {

        String eName = lName; // element name
        if ("".equals(eName)) eName = qName; // namespaceAware = false
	for (int i = 0; i < CDFelements.length; i++) {
	  if (eName.equalsIgnoreCase(CDFelements[i])) {
	    try {
	      dataBuffer.setLength(0);
	      handleStartCDFelement(eName, attrs);
	      break;
	    } catch (CDFException e) {
	      System.err.println("\n ***** CDFException 1..: "+e);
	      System.exit(0);
	    }
	  } 
	}

    }     

    public void endElement(String namespaceURI,
                           String sName, // simple name
                           String qName  // qualified name
                          ) throws SAXException {

	String elementName;
	if ("".equals(sName)) elementName = qName;
	else elementName = sName;
        for (int i = 0; i < CDFelements.length; i++) {
          if (elementName.equalsIgnoreCase(CDFelements[i])) {
	    try {
              handleEndCDFelement(elementName);
              break;
            } catch (CDFException e) {
              System.err.println("\n ****** CDFException 2..: "+e);
	      System.exit(0);
            }
          }
        }
	dataBuffer.setLength(0);

    }     

    public void characters(char buf[], int offset, int len) 
                throws SAXException {

        // buffer the characters in case they are broken down in several 
	// callbacks.
	dataBuffer.append(buf, offset, len);

    }

    /**
     * Ignorable whitespace. 
     */

    public void ignorableWhitespace(char ch[], int start, int length) {

    } // ignorableWhitespace(char[],int,int);

    //
    // ErrorHandler methods
    //

    /**
     * Warning. 
     */

    public void warning(SAXParseException ex) {

        System.err.println("[Warning] "+
                           getLocationString(ex)+": "+
                           ex.getMessage());
    }

    /**
     * Error. 
     */

    public void error(SAXParseException ex) {

        System.err.println("[Error] "+
                           getLocationString(ex)+": "+
                           ex.getMessage());

    }

    /**
     * Fatal error. 
     */

    public void fatalError(SAXParseException ex) throws SAXException {

        System.err.println("[Fatal Error] "+
                           getLocationString(ex)+": "+
                           ex.getMessage());
    }

    /**
     * Returns a string of the location. 
     */

    private String getLocationString(SAXParseException ex) {

        StringBuffer str = new StringBuffer();

        String systemId = ex.getSystemId();
        if (systemId != null) {
            int index = systemId.lastIndexOf('/');
            if (index != -1)
                systemId = systemId.substring(index + 1);
            str.append(systemId);
        }
        str.append(':');
        str.append(ex.getLineNumber());
        str.append(':');
        str.append(ex.getColumnNumber());

        return str.toString();

    } // getLocationString(SAXParseException):String

    //===========================================================
    // Utility Methods ...
    //===========================================================     
    // 

    /**
     * A start tag for a valid CDF element is encountered. 
     */ 

    private void handleStartCDFelement(String s, Attributes attrs) 
			throws CDFException, SAXException {

	stack.push(s);
        if (s.equalsIgnoreCase("variable")) setVariable(attrs);
        else if (s.equalsIgnoreCase("cdfVarInfo")) createVariable(attrs);
        else if (s.equalsIgnoreCase("cdfVAttributes")) {
           scope = VARIABLE_SCOPE;
           inScope = VAttributes;
	} else if (s.equalsIgnoreCase("cdfVarData")) recNum = 0;
        else if (s.equalsIgnoreCase("attribute")) createAttribute(attrs);
        else if (s.equalsIgnoreCase("entry")) {
	   if (entryBlock) throw new SAXException("entry element error: previous entry not closed");
	   entryBlock = true;
	   setEntry(attrs);
        } else if (s.equalsIgnoreCase("record")) {
           if (recordBlock) throw new SAXException("record element error: previous record not closed");
	   elementDelimiter = null;
           recordBlock = true;
	   processElement = false;
           if (attrs != null) {
             for (int i = 0; i < attrs.getLength(); i++) {
                  String aName = attrs.getLocalName(i); // Attr name
                if ("".equals(aName)) aName = attrs.getQName(i);
                if (aName.equalsIgnoreCase("recNum"))
                  recNum = new Long(attrs.getValue(i)).longValue();
	        else if (aName.equalsIgnoreCase("elementDelimiter"))
	          elementDelimiter = attrs.getValue(i);
             }
           }
           inScope = VarRecordData;
        } else if (s.equalsIgnoreCase("element")) {
	   if (elementBlock) throw new SAXException("element error: previous data element not closed");
	   elementDelimiter = null;
	   elementBlock = true;
	   processElement = true;
           if (attrs != null) {
             for (int i = 0; i < attrs.getLength(); i++) {
                  String aName = attrs.getLocalName(i); // Attr name
                if ("".equals(aName)) aName = attrs.getQName(i);
                if (aName.equalsIgnoreCase("index"))
                  indexString = attrs.getValue(i);
		else if (aName.equalsIgnoreCase("elementDelimiter"))
		  elementDelimiter = attrs.getValue(i);
             }
           } else {
	     throw new SAXException("Index is missing for single element data ");
	   }
	   inScope = VarElementData;
	} else if (s.equalsIgnoreCase("CDF")) createCDF(attrs);
	else if (s.equalsIgnoreCase("cdfFileInfo")) setCDF(attrs);
	else if (s.equalsIgnoreCase("cdfGAttributes")) {
	  	scope = GLOBAL_SCOPE;
		inScope = GAttributes;
	}
    }

    /**
     * A end tag for a valid CDF element is encountered. 
     */

    private void handleEndCDFelement(String s) throws CDFException, SAXException {
	if (!stack.peek().equals(s)) 
	  throw new SAXException("parsing problem...the start element:"+stack.peek()+
                                 " doesn't match to the close element "+s); 
	stack.pop();
        if (s.equalsIgnoreCase("record")) {
          recordBlock = false;
	  if (processElement) {
	    recNum++;
	    return;
	  }
          try {
            processCDFEntryORVarData();
            inScope = NONE;
	    elementDelimiter = null;
          } catch (SAXException e) {
            System.err.println("\n SAXException....: "+e);
            System.exit(0);
          }
	} else if (s.equalsIgnoreCase("entry")) {
          entryBlock = false;
          try {
                processCDFEntryORVarData();
		elementDelimiter = null;
          } catch (SAXException e) {
            System.err.println("\n SAXException....: "+e);
            System.exit(0);
          }
        } else if (s.equalsIgnoreCase("element")) {
	  elementBlock = false;
	  if (dataBuffer.length() == 0) return;
	  try {
	    processCDFEntryORVarData();
            inScope = NONE;
	    elementDelimiter = null;
          } catch (SAXException e) {
            System.err.println("\n SAXException....: "+e);
            System.exit(0);
          }
	} else if (s.equalsIgnoreCase("variable")) {
	  dimSizes = null;
	  dimVariances = null;
	  dimIndices = null; 
	  dimCounts = null;
	  dimIntervals = null;
	} else if (s.equalsIgnoreCase("cdfVariables")) inScope = NONE;
        else if (s.equalsIgnoreCase("cdfVAttributes")) inScope = NONE;
        else if (s.equalsIgnoreCase("attribute")) return;
        else if (s.equalsIgnoreCase("cdfGAttributes")) return;
	else if (s.equalsIgnoreCase("cdfFileInfo")) return;
	else if (s.equalsIgnoreCase("CDF")) closeCDF();

    }

    /**
     * A data, either an attribute entry or record value(s), is ready to 
     * be processed.                                                     
     */

    private void processCDFEntryORVarData() throws SAXException {

        if (dataBuffer.length() == 0) return;
        String tmp = dataBuffer.toString();
        if (((inScope == GAttributes || inScope == VAttributes) && 
	     (entryDataType != CDF_CHAR && entryDataType != CDF_UCHAR)) ||
	    (loadData && (inScope == VarRecordData || inScope == VarElementData) &&
	     (varDataType != CDF_CHAR && varDataType != CDF_UCHAR))) {
	  tmp = tmp.trim();
          if (tmp.length() > 0) tmp = handleNumbers(tmp);
	}
        try {
          if (inScope == GAttributes || inScope == VAttributes)
            createEntry(tmp);
          else if (loadData) {
		 if (inScope == VarRecordData) loadVarRecordData(tmp);
		 else if (inScope == VarElementData) loadVarElementData(tmp);
	       }
        } catch (CDFException e) {
          System.err.println("\n CDFException 3..: "+e);
	  System.exit(0);
        }
        dataBuffer.setLength(0);

    }

    /**
     * The data is a non-char type. Need to remove heading/trailing spaces 
     * if they are there. Also re-arrange the multiple data that each one  
     * is separated by a space.                                            
     */ 

    private String handleNumbers(String s) {

	StringBuffer stb = new StringBuffer();
	if (elementDelimiter != null) {
	  s = removeSpaces(s, elementDelimiter);
	  StringTokenizer st = new StringTokenizer(s, elementDelimiter);
	  while (st.hasMoreTokens()) {
	    String tmp = st.nextToken();
	    if (tmp.length() == 0) continue;
	    stb.append(tmp).append(" ");
	  }
	} else {
          StringTokenizer st = new StringTokenizer(s);
	  int iTokens = st.countTokens();
          if (iTokens > 0) {
            while (st.hasMoreTokens())
              stb.append(st.nextToken()).append(" ");
            stb.setLength(stb.length()-1);
          }
	}
        return stb.toString();

    }

    /**
     * It constructs the cdf file name based on the provided path and 
     * name. Checks if the new CDF file already exists. A new CDF is  
     * created.
     */
  
    private void createCDF(Attributes attrs) throws CDFException {

	if (newCDFName != null &&
	    outputDir == null) cdfName = newCDFName;
	else {
          if (attrs != null) {
            for (int i = 0; i < attrs.getLength(); i++) {
              String aName = attrs.getLocalName(i); // Attr name
              if ("".equals(aName)) aName = attrs.getQName(i);
              if (aName.equalsIgnoreCase("name")) 
		cdfName = attrs.getValue(i);
	    }
          }
	  cdfName = cdfName.substring(cdfName.lastIndexOf(
				      System.getProperty("file.separator"))+1);
	  if (outputDir != null)
		cdfName = outputDir + cdfName;
	  CDFML2CDF.checkCDFFile(cdfName);
        }

	if (backward) 
	  CDF.setFileBackward(BACKWARDFILEon);
	cdf  = CDF.create(cdfName);
	entryBlock = false;
	recordBlock = false;
	
    }

    /**
     * Properly close the newly created CDF file before program ends. 
     */

    private void closeCDF() throws CDFException {
        cdf.close();

    }

    /**
     * Modify the CDF's characteristics from its defaults.           
     */

    private void setCDF(Attributes attrs) throws CDFException {

        if (attrs != null) {
          for (int i = 0; i < attrs.getLength(); i++) {
              String aName = attrs.getLocalName(i); // Attr name
              if ("".equals(aName)) aName = attrs.getQName(i);
              if (aName.equalsIgnoreCase("fileFormat")) {
		long format = CDFUtils.getLongFormat(attrs.getValue(i));
		cdf.setFormat(format);
	      } else if (aName.equalsIgnoreCase("compression")) {
		String tmpCmp = attrs.getValue(i);
		long compression, compressionLevel;
		if (tmpCmp.length() > 4 && tmpCmp.substring(0, 4).equalsIgnoreCase("gzip")) {
		  compression = CDFUtils.getLongCompressionType(tmpCmp.substring(0,4));
		  compressionLevel = new Long(tmpCmp.substring(5)).longValue();
		} else {
		  compression = CDFUtils.getLongCompressionType(tmpCmp);
		  if (compression == NO_COMPRESSION) compressionLevel = 0;
		  else compressionLevel = 1;
		}
		cdf.setCompression(compression, new long[] {compressionLevel});
	      } else if (aName.equalsIgnoreCase("majority")) {
		long majority = CDFUtils.getLongMajority(attrs.getValue(i));
		cdf.setMajority(majority);
	      } else if (aName.equalsIgnoreCase("encoding")) {
		long encoding = CDFUtils.getLongEncoding(attrs.getValue(i));
		cdf.setEncoding(encoding);
              } else if (aName.equalsIgnoreCase("NEGtoPOSfp0")) {
                long nEGtoPOSfp0;
		if (attrs.getValue(i).equalsIgnoreCase("ENABLE")) nEGtoPOSfp0 = NEGtoPOSfp0on;
		else nEGtoPOSfp0 = NEGtoPOSfp0off;
                cdf.selectNegtoPosfp0(nEGtoPOSfp0);
              } else if (aName.equalsIgnoreCase("checksum")) {
                long checksum = CDFUtils.getLongChecksum(attrs.getValue(i));
                cdf.setChecksum(checksum);
              } else if (aName.equalsIgnoreCase("CDFCacheSize")) {
                long cdfCacheSize = new Long(attrs.getValue(i)).longValue();
                cdf.selectCDFCacheSize(cdfCacheSize);
              } else if (aName.equalsIgnoreCase("CompressCacheSize")) {
                long compressCacheSize = new Long(attrs.getValue(i)).longValue();
                cdf.selectCompressCacheSize(compressCacheSize);
              } else if (aName.equalsIgnoreCase("StageCacheSize")) {
                long stageCacheSize = new Long(attrs.getValue(i)).longValue();
                cdf.selectStageCacheSize(stageCacheSize);
	      }
          }
        }

    }

    /**
     * Create a new global attribute or a variable attribute if it is not 
     * already created.                                                   
     */

    private void createAttribute(Attributes attrs) throws CDFException {

        if (attrs != null) {
          for (int i = 0; i < attrs.getLength(); i++) {
              String aName = attrs.getLocalName(i); // Attr name
              if ("".equals(aName)) aName = attrs.getQName(i);
              if (aName.equalsIgnoreCase("name")) attrName = attrs.getValue(i);
          }
        }

        if (scope == GLOBAL_SCOPE) {
	  attribute  = Attribute.create(cdf, attrName, GLOBAL_SCOPE);
	  if (progressReport)
	    System.err.println("  Created global attribute: "+attrName+
			       " and start loading entries"); 
	} else { // Variable attribute
	  long id = cdf.getAttributeID(attrName);
	  if (id == -1) // do not yet exist 
	    attribute  = Attribute.create(cdf, attrName, VARIABLE_SCOPE);
	  else // already created
	    attribute  = cdf.getAttribute(id);
	}

    }

    /**
     * Set the characterisitcs of an attribute entry.        
     */

    private void setEntry(Attributes attrs) {

	entryNumFlag = false;
        if (attrs != null) {
          for (int i = 0; i < attrs.getLength(); i++) {
              String aName = attrs.getLocalName(i); // Attr name
              if ("".equals(aName)) aName = attrs.getQName(i);
              if (aName.equalsIgnoreCase("entryNum")) { // global attribute
		entryNumFlag = true;                    // entry only
                entryNum = new Long(attrs.getValue(i)).longValue();
              } else if (aName.equalsIgnoreCase("cdfDatatype")) {
                entryDataType = CDFUtils.getDataTypeValue(attrs.getValue(i));
              } else if (aName.equalsIgnoreCase("numElements")) {
		numElements = new Long(attrs.getValue(i)).longValue();
	      }
          }
        }

    }

    /**
     * Create an entry with the provided data.                
     */

    private void createEntry(String tmp) throws CDFException {

	Object entryValue = createEntryObject(tmp);
	if (entryValue != null)
          if (inScope == GAttributes) {    // use the provided entry number or
	    if (!entryNumFlag) entryNum++; // auto-increase the previous one 
            Entry.create(attribute, entryNum, entryDataType, entryValue);
          } else
            Entry.create(attribute, var.getID(), entryDataType, entryValue);
    }

    /**
     * Create a proper object based on the data type and the provided 
     * data. The object is an array of comparable type.               
     */
 
    private Object createEntryObject(String str) throws CDFException {

        String rts = new String(str);
        StringTokenizer st = new StringTokenizer(str, " ");
        int iTokens = st.countTokens();
	Object entryValue;

        if (entryDataType == CDF_INT1 || entryDataType == CDF_BYTE) {
	  entryValue = new byte[iTokens];
	  for (int i = 0; i < iTokens; i++) 
	    ((byte[]) entryValue)[i] = new Byte(st.nextToken()).byteValue();
        } else if (entryDataType == CDF_INT2 || entryDataType == CDF_UINT1) {
	  entryValue = new short[iTokens];
          for (int i = 0; i < iTokens; i++)
            ((short[]) entryValue)[i] = new Short(st.nextToken()).shortValue();
        } else if (entryDataType == CDF_INT4 || entryDataType == CDF_UINT2) {
	  entryValue = new int[iTokens];
          for (int i = 0; i < iTokens; i++)
            ((int[]) entryValue)[i] = new Integer(st.nextToken()).intValue();
        } else if (entryDataType == CDF_UINT4 || entryDataType == CDF_INT8) {
	  entryValue = new long[iTokens];
          for (int i = 0; i < iTokens; i++)
            ((long[]) entryValue)[i] = new Long(st.nextToken()).longValue();
        } else if (entryDataType == CDF_REAL4 || entryDataType == CDF_FLOAT) {
	  entryValue = new float[iTokens];
          for (int i = 0; i < iTokens; i++)
            ((float[]) entryValue)[i] = new Float(st.nextToken()).floatValue();
        } else if (entryDataType == CDF_REAL8 || entryDataType == CDF_DOUBLE) {
	  entryValue = new double[iTokens];
          for (int i = 0; i < iTokens; i++)
            ((double[]) entryValue)[i] = new Double(st.nextToken()).doubleValue();
        } else if (entryDataType == CDF_CHAR || entryDataType == CDF_UCHAR) {
	    entryValue = str;
        } else if (entryDataType == CDF_EPOCH) {
          StringTokenizer itks = new StringTokenizer(rts, " ");
          String sss = itks.nextToken();
          if (sss.length() != 23) { 
	    entryValue = new double[iTokens/2];
            for (int i = 0; i < iTokens/2; i++)
              ((double[]) entryValue)[i] = Epoch.parse(st.nextToken()+" "+st.nextToken());
          } else { /* ISO 8601 format. */
            entryValue = new double[iTokens];
            for (int i = 0; i < iTokens; i++)
              ((double[]) entryValue)[i] = Epoch.parse4(st.nextToken());
          }
        } else if (entryDataType == CDF_EPOCH16) {
          Object twoDouble = new double[2];
          StringTokenizer itks = new StringTokenizer(rts, " ");
          String sss = itks.nextToken();
          if (sss.length() != 32) {
            entryValue = new double[iTokens];
            for (int i = 0, j = 0; i < iTokens/2; i++) {
       	      twoDouble = Epoch16.parse(st.nextToken()+" "+st.nextToken());
              ((double[]) entryValue)[j++] = ((double[]) twoDouble)[0];
	      ((double[]) entryValue)[j++] = ((double[]) twoDouble)[1];
	    }
          } else { /* ISO 8601 format. */
            entryValue = new double[iTokens*2];
            for (int i = 0, j = 0; i < iTokens; i++) {
              twoDouble = Epoch16.parse4(st.nextToken());
              ((double[]) entryValue)[j++] = ((double[]) twoDouble)[0];
              ((double[]) entryValue)[j++] = ((double[]) twoDouble)[1];
            }
          }
        } else if (entryDataType == CDF_TIME_TT2000) {
          entryValue = new long[iTokens];
          for (int i = 0; i < iTokens; i++)
            ((long[]) entryValue)[i] = CDFTT2000.fromUTCstring(st.nextToken());
        } else
	  entryValue = null;
	return entryValue;

    }

    /**
     * Acquires the variable name from the variable element's attribute. 
     */

    private void setVariable(Attributes attrs) {

        if (attrs != null) {
          for (int i = 0; i < attrs.getLength(); i++) {
              String aName = attrs.getLocalName(i); // Attr name
              if ("".equals(aName)) aName = attrs.getQName(i);
              if (aName.equalsIgnoreCase("name")) {
                varName = attrs.getValue(i);
              }
          }
        }

    }

    /**
     * A variable's characterists are acquired from the attributes of the 
     * cdfVarInfo element. A variable is created and its defaults are     
     * reset if any optioanl attributes are provided.                     
     */

    private void createVariable(Attributes attrs) throws CDFException {

	long compression = 0, compressionLevel = 0;
	boolean dimSizesFlag, dimVariancesFlag;
	boolean compressionFlag, padValueFlag;
	boolean sparseRecordsFlag, blockingFactorFlag, initialRecordsFlag;
	boolean cacheSizeFlag, reservePercentFlag, recordsAllocateFlag;
	Object padValue = null;
	long sparseRecords = -1;
	long blockingFactor = -1;
	long initialRecords = -1;
	long recordsAllocate = -1;
	long cacheSize = -1;
	long reservePercent = -1;

        if (attrs != null) {
	  dimSizesFlag = false;
	  dimVariancesFlag = false;
	  compressionFlag = false;
	  padValueFlag = false;
	  sparseRecordsFlag = false;
	  blockingFactorFlag = false;
	  initialRecordsFlag = false;
	  recordsAllocateFlag = false;
          cacheSizeFlag = false;
          reservePercentFlag = false;
	  varDataType = -1;
	  dim = -1;

          for (int i = 0; i < attrs.getLength(); i++) {
              String aName = attrs.getLocalName(i); // Attr name
              if ("".equals(aName)) aName = attrs.getQName(i);
              if (aName.equalsIgnoreCase("cdfDatatype")) {
                varDataType = CDFUtils.getDataTypeValue(attrs.getValue(i));
              } else if (aName.equalsIgnoreCase("dim")) {
                dim = new Long(attrs.getValue(i)).longValue();
		int dimNo = (int) (dim > 0 ? dim : 1);
		dimSizes = new long[dimNo];
		dimVariances = new long[dimNo];
		dimIndex = new long[dimNo];
              } else if (aName.equalsIgnoreCase("dimSizes")) {
		dimSizesFlag = true;
                String tmp = attrs.getValue(i);
	        StringTokenizer st = new StringTokenizer(tmp, ",");
		int iTokens = st.countTokens();
		if (dim != -1) {
		  if (dim != iTokens) {
		    st = new StringTokenizer(tmp, " ");
		    iTokens = st.countTokens();
		    if (dim != iTokens) {
		      System.err.println("Var:"+varName+" dim="+dim+" dimSizes="+iTokens+" don't match");
		      return;
		    }
		  }
		}
		if (dimSizes == null) dimSizes = new long[st.countTokens()];
	        if (!"".equals(tmp) && iTokens > 0) {
		   int ii = 0;
	           while (st.hasMoreTokens()) 
		     dimSizes[ii++] = new Long((String)st.nextToken()).longValue();
		} else
                  dimSizes[0] = 0;
              } else if (aName.equalsIgnoreCase("dimVariances")) {
		dimVariancesFlag = true;
                String tmp = attrs.getValue(i);
                StringTokenizer st = new StringTokenizer(tmp, ",");
                int iTokens = st.countTokens();
                if (dim != -1) {
                  if (dim != iTokens) {
                    st = new StringTokenizer(tmp, " ");
                    iTokens = st.countTokens();
                    if (dim != iTokens) {
                      System.err.println("Var:"+varName+" dim="+dim+" dimVariances="+iTokens+" don't match");
                      return;
                    }
                  }
                }
		if (dimVariances == null) dimVariances = new long[st.countTokens()];
                if (!"".equals(tmp) && iTokens > 0) {
                   int ii = 0;
                   while (st.hasMoreTokens()) {
                     if (((String)st.nextToken()).equalsIgnoreCase("VARY")) 
			dimVariances[ii] = VARY;
                     else
			dimVariances[ii] = NOVARY;
		     ii++;
                   }
                } else
                dimVariances[0] = 0;
              } else if (aName.equalsIgnoreCase("recVariance")) {
                if ((attrs.getValue(i)).equalsIgnoreCase("VARY"))
		  recVariance = VARY;
                else
		  recVariance = NOVARY;
              } else if (aName.equalsIgnoreCase("numElements")) {
                numElements = new Long(attrs.getValue(i)).longValue();
              } else if (aName.equalsIgnoreCase("compression")) {
                String tmpCmp = attrs.getValue(i);
                if (tmpCmp.length() > 4 && tmpCmp.substring(0, 4).equalsIgnoreCase("gzip")) {
                  compression = CDFUtils.getLongCompressionType(tmpCmp.substring(0,4));
                  compressionLevel = new Long(tmpCmp.substring(5)).longValue();
                } else {
                  compression = CDFUtils.getLongCompressionType(tmpCmp);
                  if (compression == NO_COMPRESSION) compressionLevel = 0;
                  else compressionLevel = 1;
                }
		compressionFlag = true;
              } else if (aName.equalsIgnoreCase("sparseRecords")) {
                sparseRecordsFlag = true;
                sparseRecords = CDFUtils.getLongSparseRecord(attrs.getValue(i));
              } else if (aName.equalsIgnoreCase("blockingFactor")) {
                blockingFactorFlag = true;
                blockingFactor = new Long(attrs.getValue(i)).longValue();
              } else if (aName.equalsIgnoreCase("numInitialRecords")) {
                initialRecordsFlag = true;
                initialRecords = new Long(attrs.getValue(i)).longValue();
              } else if (aName.equalsIgnoreCase("numRecordsAllocate")) {
		if (!compressionFlag) {
                  recordsAllocateFlag = true;
                  recordsAllocate = new Long(attrs.getValue(i)).longValue();
		}
              } else if (aName.equalsIgnoreCase("cacheSize")) {
                cacheSizeFlag = true;
                cacheSize = new Long(attrs.getValue(i)).longValue();
              } else if (aName.equalsIgnoreCase("reservePercent")) {
		if (compressionFlag) {
                  reservePercentFlag = true;
                  reservePercent = new Long(attrs.getValue(i)).longValue();
		}
              } else if (aName.equalsIgnoreCase("padValue")) {
		padValueFlag = true;
                String tmp = attrs.getValue(i);
		if (varDataType == CDF_INT1 || varDataType == CDF_BYTE) {
		  padValue = new Byte(tmp);
		} else if (varDataType == CDF_INT2 || varDataType == CDF_UINT1) {
		  padValue = new Short(tmp);
		} else if (varDataType == CDF_INT4 || varDataType == CDF_UINT2) {
		  padValue = new Integer(tmp);
		} else if (varDataType == CDF_UINT4 || varDataType == CDF_INT8) {
		  padValue = new Long(tmp);
		} else if (varDataType == CDF_REAL4 || varDataType == CDF_FLOAT) {
		  padValue = new Float(tmp);
		} else if (varDataType == CDF_REAL8 || varDataType == CDF_DOUBLE) {
		  padValue = new Double(tmp);
		} else if (varDataType == CDF_CHAR || varDataType == CDF_UCHAR) {
		  padValue = tmp;
		} else if (varDataType == CDF_EPOCH) {
                  if (tmp.length() == 24)
		    padValue = new Double(Epoch.parse(tmp));
                  else /* ISO 8601 format. */
                    padValue = new Double(Epoch.parse4(tmp));
                } else if (varDataType == CDF_EPOCH16) {
                  padValue = new double[2];
                  if (tmp.length() == 36)
		    padValue = Epoch16.parse(tmp);
                  else /* ISO 8601 format. */
                    padValue = Epoch16.parse4(tmp);
                } else if (varDataType == CDF_TIME_TT2000) {
                    padValue = new Long(CDFTT2000.fromUTCstring(tmp));
                }
              }
          }
	  if (dim > 0) { 
	    if (!dimSizesFlag || !dimVariancesFlag) {
		System.err.println("\nError... dimSizes or dimVariances not provided for variable:"+varName+"!");
		return;
	    } 
	    phyDims = (int) dim;
	    physicalSize = 1;
	    for (int i = 0; i < dim; i++) physicalSize *= (int) dimSizes[i];
	  } else {
	     dimSizes[0] = 1;
	     dimVariances[0] = VARY;
             phyDims = 1;
             physicalSize = 1;
          }
	  var = Variable.create(cdf, varName, varDataType, numElements, dim, dimSizes,
				recVariance, dimVariances);
	  if (progressReport) {
	    String tail = (loadData) ? " and start loading its data" : " ";
	    System.err.println("  Created variable "+var.getID()+": "+varName+tail);
	  }
	  if (compressionFlag) 
		var.setCompression(compression, new long[] {compressionLevel});
	  if (padValueFlag) var.setPadValue(padValue);
	  if (sparseRecordsFlag) var.setSparseRecords(sparseRecords);
          if (blockingFactorFlag) var.setBlockingFactor(blockingFactor);
	  if (cacheSizeFlag) var.selectCacheSize(cacheSize);
	  if (reservePercentFlag) var.selectReservePercent(reservePercent);
	  if (initialRecordsFlag && (initialRecords > 0)) 
	    var.setInitialRecords(initialRecords);
	  if (recordsAllocateFlag && (recordsAllocate > 0)) 
  	    var.allocateRecords(recordsAllocate);
        }

    }

    /**
     * Converts a full variable record data, single or multiple values, 
     * into a proper object and loads the object into the variable.     
     * It needs to reset the arrays that are used for the variable's    
     * dimension information which time a new variable is encountered.  
     */ 

    private void loadVarRecordData(String tmp) throws CDFException {

        String pmt = new String(tmp);
	StringTokenizer st = new StringTokenizer(tmp, " ");
        int iTokens = st.countTokens();
	recInterval = 1;
	recCount = 1;

	if (dimIndices == null) {
          dimIndices = new long[phyDims];
          dimCounts = new long[phyDims];
          dimIntervals = new long[phyDims];
	  for (int i = 0; i < phyDims; i++) {
            dimIndices[i] = 0;
            dimCounts[i] = dimSizes[i];
            dimIntervals[i] = 1;
	  }
	  dataObject = createObject(phyDims, dimSizes, varDataType);
        }

	Object dataValues = null;
        int items = iTokens;
        if (varDataType == CDF_INT1 || varDataType == CDF_BYTE) {
	  dataValues = (byte[]) convertNumberData(iTokens, st, varDataType);
        } else if (varDataType == CDF_INT2 || varDataType == CDF_UINT1) {
	  dataValues = (short[]) convertNumberData(iTokens, st, varDataType);
        } else if (varDataType == CDF_INT4 || varDataType == CDF_UINT2) {
	  dataValues = (int[]) convertNumberData(iTokens, st, varDataType);
        } else if (varDataType == CDF_UINT4 || varDataType == CDF_INT8) {
	  dataValues = (long[]) convertNumberData(iTokens, st, varDataType);
        } else if (varDataType == CDF_REAL4 || varDataType == CDF_FLOAT) {
	  dataValues = (float[]) convertNumberData(iTokens, st, varDataType);
        } else if (varDataType == CDF_REAL8 || varDataType == CDF_DOUBLE) {
	  dataValues = (double[]) convertNumberData(iTokens, st, varDataType);
        } else if (varDataType == CDF_CHAR || varDataType == CDF_UCHAR) {
	  dataValues = new byte[(int)(physicalSize*numElements)];
	  byte[] dummy = new String(" ").getBytes();
	  byte sp = dummy[0];
	  for (int ij = 0; ij < physicalSize*numElements; ij++) 
	     ((byte[]) dataValues)[ij] = (byte) sp;
	  if (elementDelimiter != null) { // multiple data entries assumed
	    tmp = removeSpaces(tmp, elementDelimiter);
	    StringTokenizer st2 = new StringTokenizer(tmp, elementDelimiter);
	    String aTemp;
	    int ij = 0;
	    iTokens = 0;
	    while (st2.hasMoreTokens()) {
	      aTemp = st2.nextToken();
	      if (aTemp.length() == 0) continue;
	      System.arraycopy(aTemp.getBytes(), 0, (byte[]) dataValues, (int) (ij*numElements), 
			       aTemp.length());
	      ij++;
	      iTokens++;
	      if (ij == physicalSize) break;
	    }
	  } else { // single data entry assumed
	      iTokens = 1;
	      System.arraycopy(tmp.getBytes(), 0, (byte[]) dataValues, 0,
                               tmp.length());
          }
          items = iTokens;
        } else if (varDataType == CDF_EPOCH) {
          StringTokenizer itok = new StringTokenizer(pmt, " ");
          if (itok.nextToken().length() != 23) {
	    dataValues = new double[iTokens/2];
	    for (int ij = 0, ik = 0; ij < iTokens; ij=ij+2)
	      ((double[]) dataValues)[ik++] = Epoch.parse(st.nextToken()+" "+st.nextToken());
            items = iTokens/2;
          } else { /* ISO 8601 format. */
            dataValues = new double[iTokens];
            for (int ij = 0; ij < iTokens; ++ij)
              ((double[]) dataValues)[ij] = Epoch.parse4(st.nextToken());
            items = iTokens;
          }
        } else if (varDataType == CDF_EPOCH16) {
           StringTokenizer itok = new StringTokenizer(pmt, " ");
           Object twoDouble = new double[2];
           if (itok.nextToken().length() != 32) {
	     dataValues = new double[iTokens];
	     for (int ij = 0, ik = 0; ij < iTokens/2; ij++) {
	       twoDouble = Epoch16.parse(st.nextToken()+" "+st.nextToken());
	       ((double[]) dataValues)[ik++] = ((double[]) twoDouble)[0];
	       ((double[]) dataValues)[ik++] = ((double[]) twoDouble)[1];
	     }
             items =  iTokens/2;
           } else { /* ISO 8601 format. */
             dataValues = new double[iTokens*2];
             for (int ij = 0, ik = 0; ij < iTokens; ij++) {
               twoDouble = Epoch16.parse4(st.nextToken());
               ((double[]) dataValues)[ik++] = ((double[]) twoDouble)[0];
               ((double[]) dataValues)[ik++] = ((double[]) twoDouble)[1];
             }
             items =  iTokens;
           }
        } else if (varDataType == CDF_TIME_TT2000) {
           dataValues = new long[iTokens];
           for (int ij = 0; ij < iTokens; ++ij)
             ((long[]) dataValues)[ij] = CDFTT2000.fromUTCstring(st.nextToken());
        }
	if (dataValues != null) {
//	  arrayify(dataValues, dataObject, (int) (items*numElements), phyDims, dimSizes);
	  var.putHyperData(recNum, recCount, recInterval,
			   dimIndices, dimCounts, dimIntervals,
//		 	   dataObject);
			   dataValues);
	  recNum++;
	}

    }

    /**
     * Similar to loadVarRecordData. But, it only writes one data value at 
     * a time. Indices are provided through the attribute.                 
     */

    private void loadVarElementData(String tmp) throws CDFException {

	StringTokenizer st = new StringTokenizer(indexString, ",");
	int iTokens = st.countTokens();

	for (int i = 0; i < phyDims; i++) 
	  dimIndex[i] = new Long(st.nextToken()).longValue();

	Object dataValue = null;
        if (varDataType == CDF_INT1 || varDataType == CDF_BYTE) {
	  dataValue = new Byte(tmp);
        } else if (varDataType == CDF_INT2 || varDataType == CDF_UINT1) {
	  dataValue = new Short(tmp);
        } else if (varDataType == CDF_INT4 || varDataType == CDF_UINT2) {
	  dataValue = new Integer(tmp);
        } else if (varDataType == CDF_UINT4 || varDataType == CDF_INT8) {
	  dataValue = new Long(tmp);
        } else if (varDataType == CDF_REAL4 || varDataType == CDF_FLOAT) {
	  dataValue = new Float(tmp);
        } else if (varDataType == CDF_REAL8 || varDataType == CDF_DOUBLE) {
	  dataValue = new Double(tmp);
        } else if (varDataType == CDF_CHAR || varDataType == CDF_UCHAR) {
	  if (tmp.length() > numElements) dataValue = tmp.substring(0, (int) numElements);
	  else dataValue = tmp;
        } else if (varDataType == CDF_EPOCH) {
          if (tmp.trim().length() != 23)
	    dataValue = new Double(Epoch.parse(tmp));
          else /* ISO 8601 format. */
            dataValue = new Double(Epoch.parse4(tmp));
        } else if (varDataType == CDF_EPOCH16) {
          dataValue = new double[2];
          if (tmp.trim().length() != 32)
	    dataValue = Epoch16.parse(tmp);
          else /* ISO 8601 format. */
            dataValue = Epoch16.parse4(tmp);
        } else if (varDataType == CDF_TIME_TT2000) {
            dataValue = new Long(CDFTT2000.fromUTCstring(tmp));
        }
	if (dataValue != null) 
	  var.putSingleData(recNum, dimIndex, dataValue);

    }

    /**
     * An object representing the full variable record data is created based 
     * on its CDF data type and dimensionality. The object can be           
     * multi-dimensional.                                                  
     */

    private Object createObject(long dims, long[] dimSizes, long dataType) 
		   throws CDFException {

	Object one = null;
	Class arrayClass = null;
	int[] newDimsSizes = new int[(int)dims];
	for (int i=0; i<dims; i++) newDimsSizes[i] = (int) dimSizes[i];
	if (dataType == CDF_EPOCH16) {
	  if (dims == 0) 
	    newDimsSizes[0] = 2;
	  else
	    newDimsSizes[(int)dims-1] = 2 * newDimsSizes[(int)dims-1];
	}
    
        switch ((int)dataType) {
          case (int)CDF_CHAR:
          case (int)CDF_UCHAR:
            arrayClass = new String().getClass();
            break;
          case (int)CDF_BYTE:
          case (int)CDF_INT1:  arrayClass = Byte.TYPE;          break;
          case (int)CDF_INT2:
          case (int)CDF_UINT1: arrayClass = Short.TYPE;         break;
          case (int)CDF_INT4:
          case (int)CDF_UINT2: arrayClass = Integer.TYPE;       break;
          case (int)CDF_UINT4:
          case (int)CDF_INT8:
          case (int)CDF_TIME_TT2000: arrayClass = Long.TYPE;    break;
          case (int)CDF_REAL4:
          case (int)CDF_FLOAT: arrayClass = Float.TYPE;         break;
          case (int)CDF_REAL8:
          case (int)CDF_DOUBLE:
          case (int)CDF_EPOCH: 
	  case (int)CDF_EPOCH16: arrayClass = Double.TYPE;      break;
          default:
            break;
        } /* switch (dataType) */

        try {
            one = Array.newInstance(arrayClass, newDimsSizes);
        } catch (Exception e) {
            throw new CDFException(BAD_ARGUMENT);
        }
	return one;

    }

    /**
     * The data values that will be filled into one single variable record is 
     * moved to an 1-dimensional array of similar data type. This deals with
     * only number data.
     */

    private Object convertNumberData(int itemCounts, StringTokenizer st,
                                     long dataType) throws CDFException {

        Object one = null;

        switch ((int)dataType) {
          case (int)CDF_CHAR:
          case (int)CDF_UCHAR:
	    one = new String[itemCounts];
	    for (int i = 0; i< itemCounts; i++)
	      ((String[]) one)[i] = st.nextToken();
            break;
          case (int)CDF_BYTE:
          case (int)CDF_INT1:  
		one = new byte[itemCounts];
		for (int i = 0; i< itemCounts; i++) 
		  ((byte[]) one)[i] = new Byte(st.nextToken()).byteValue();
	        break;
          case (int)CDF_INT2:
          case (int)CDF_UINT1: 
                one = new short[itemCounts];          
                for (int i = 0; i< itemCounts; i++)
                  ((short[]) one)[i] = new Short(st.nextToken()).shortValue();
                break;
          case (int)CDF_INT4:
          case (int)CDF_UINT2:
                one = new int[itemCounts];          
                for (int i = 0; i< itemCounts; i++)
                  ((int[]) one)[i] = new Integer(st.nextToken()).intValue();
                break;
          case (int)CDF_UINT4: 
          case (int)CDF_INT8: 
                one = new long[itemCounts];          
                for (int i = 0; i< itemCounts; i++)
                  ((long[]) one)[i] = new Long(st.nextToken()).longValue();
                break;
          case (int)CDF_REAL4:
          case (int)CDF_FLOAT:
                one = new float[itemCounts];          
                for (int i = 0; i< itemCounts; i++)
                  ((float[]) one)[i] = new Float(st.nextToken()).floatValue();
                break;
          case (int)CDF_REAL8:
          case (int)CDF_DOUBLE:
                one = new double[itemCounts];          
                for (int i = 0; i< itemCounts; i++)
                  ((double[]) one)[i] = new Double(st.nextToken()).doubleValue();
                break;
          default:
            break;
        } /* switch (dataType) */
        return one;

    }

    /**
     * Translates a file into a common URL form.
     */

    private static String fileToURL(File f) throws IOException {

        String temp;
        if (!f.exists())
          throw new IOException("No such file: "+f.getName());
        temp = f.getAbsolutePath();
        if (File.separatorChar != '/')
          temp = temp.replace(File.separatorChar, '/');
        if (!temp.startsWith("/"))
          temp = "/" + temp;
        if (!temp.endsWith("/") && f.isDirectory())
          temp = temp + "/";
        return "file:"+temp;

    }

    /**
     * It assumes that each data is delimited by a pair of delimiters, a preceding
     * and a trailing one. It would remove any spaces between the trailing delimiter
     * from a preceding data and the preceding delimiter from its subceding data.
     */

    private String removeSpaces(String tmp, String delimiter) {
	
        StringBuffer sb = new StringBuffer();
        int dellen = delimiter.length();
        int loc1 = 0, loc2 = -1, len = tmp.length();
        int locX = -1;
        if (len == 0) return null;
        while (loc1 < len) {
          locX = tmp.substring(loc1).indexOf(delimiter);
          if (locX != -1) {
            loc1 = loc1 + locX;
            locX = tmp.substring(loc1+dellen).indexOf(delimiter);
            if (locX != -1) {
              loc2 = loc1 + dellen + locX;
              sb.append(tmp.substring(loc1, loc2+dellen));
              loc1 = loc2+dellen;
            } else 
              break;
          } else 
            break;
        }
        return sb.toString();
   
    }  

    /**
     * It provides the how-to information to properly use the program.
     */

    private static void howto() {
	
	System.err.println(" "); 
	System.err.println("Description:");
	System.err.println("    This program converts a cdfml (an xml) file that conforms to the");
        System.err.println("    cdf.dtd or cdf.xsd into a CDF file.");
	System.err.println(" ");
        System.err.println("Usage: java [-Dorg.xml.sax.driver=...] CDFML2CDF [Options] xmlFileName");
	System.err.println(" ");
	System.err.println("    The SAX2 parser is used to parse the cdfml document. ");
	System.err.println("    The default SAX2 parser can be set up through a system resource ");
	System.err.println("    at SAX2 JAR file's META-INF/services/org.xml.sax.driver if ");
        System.err.println("    your SAX2 distribution supports it through the class loader."); 
	System.err.println("    Otherwise, the system property for org.xml.sax.dirver has to be");
	System.err.println("    set for the parser. The parser classes are JVM/SAX2-dependent.");
        System.err.println("    If Sun's Java V1.4 and above is used, you don't have to specify");
        System.err.println("    the parser as its Crimson will be used as the default.\n");
	System.err.println("    For Xerces: org.apache.xerces.parsers.SAXParser");
	System.err.println("        Crimson (SUN 1.4): org.apache.crimson.parser.XMLReaderImpl");
	System.err.println("        AElfred2: gnu.xml.aelfred2.XmlReader (validating) or ");
	System.err.println("                  gnu.xml.aelfred2.SAXDriver (non-validating)");
	System.err.println("        Oracle: oracle.xml.parser.v2.SAXParser");
	System.err.println(" ");
	System.err.println("Options: ");
        System.err.println("   -[no]Validate       Whether to validate the cdfml file during the process. ");
        System.err.println("                       It's parser dependent. -noValidate is the default. ");
	System.err.println("   -[no]Delete         Whether to delete the CDF file if it already exists. ");
	System.err.println("                       -noDelete is the default. ");
	System.err.println("   -cdf:cdfFileName    A new CDF file name to replace the given name at CDF");
	System.err.println("     or cdfDirecotry   element's name attribute in the cdfml file. No .cdf."); 
	System.err.println("                       extension is needed. Or, provide an existing directory.");
	System.err.println("                       The directory  is to combine with the cdfml's CDF");
	System.err.println("                       element's name attribute to constitute the full pathname.");
	System.err.println("   -[no]Data           Whether to load the variable record data values if they");
	System.err.println("                       are in the cdfml file to the CDF file. -Data is the");
        System.err.println("                       default.");
	System.err.println("   -[no]Backward       Whether to create the CDF in an older, V2.7, version,");
	System.err.println("                       instead of the current library V3.* version. -noBackward");
        System.err.println("                       is the default. ");
        System.err.println("   -[no]Progress       Whether to display the progress.");
        System.err.println("                       -noProgress is the default. ");
	System.err.println(" ");
        System.err.println("Examples: \n");
	System.err.println(" 1) java CDFML2CDF -Validate testcdfml.xml ");
	System.err.println("         if your JAR file's META-INF/services/org.xml.sax.driver is defined,");
	System.err.println("         such as Xerces and AElfred2's parsers. ");
        System.err.println(" 2) java -Dorg.xml.sax.driver=org.apache.xerces.parsers.SAXParser CDFML2CDF"); 
        System.err.println("         -Delete -CDF:test1 testcdfml.xml ");
	System.err.println("         use Xerces's Parser and create a CDF file, test1.cdf. "); 
        System.err.println(" 3) java -Dorg.xml.sax.driver=org.apache.crimson.parser.XMLReaderImpl CDFML2CDF");
        System.err.println("         -CDF:test1 -noData testcdfml.xml ");
	System.err.println("         use Sun's Java V1.4 and create test1.cdf but passing the variable");
        System.err.println("         record data.");
        System.err.println(" 4) java -Dorg.xml.sax.driver=oracle.xml.parser.v2.SAXParser CDFML2CDF");
        System.err.println("         -CDF:/home/mydir -delete -noData myCDFML.xml ");
	System.err.println("          use Oracle's parser and create /home/mydir/xxxx.cdf where xxxx comes");
	System.err.println("          from the name attribute at the CDF element in myCDFML.xml."); 
	System.err.println(" ");
        System.exit(1);
        
   }

}
