// $Id: CDFTools.java,v 1.1 2012/05/18 17:48:50 liu Exp $
package gsfc.nssdc.cdf;

import java.lang.*;
import java.util.*;
import java.io.*;
import gsfc.nssdc.cdf.util.*;

/**
 * CDFTools.java
 *
 *
 * Created: Tue Nov 24 16:14:50 1998
 *
 * @author Phil Williams
 * @version $Id: CDFTools.java,v 1.1 2012/05/18 17:48:50 liu Exp $
 */

public class CDFTools implements CDFConstants {
    
    private static String cdfbase   = System.getProperty("cdf.base");
    private static String separator = System.getProperty("file.separator");
    private static String cmdBase   = cdfbase+separator+"bin"+separator;

    // Where to send messages
    private static java.io.OutputStream err = System.err;

    public static final int NO_VALUES     = 0;
    public static final int NRV_VALUES    = 1;
    public static final int RV_VALUES     = 2;
    public static final int ALL_VALUES    = 3;
    public static final int NAMED_VALUES   = 4;

    public static final int NO_REPORTS    = 0;
    public static final int REPORT_ERRORS = 1;
    public static final int REPORT_WARNINGS = 2;
    public static final int REPORT_INFORMATION = 4;

    /**
     * skeletonTable produces a skeleton table from a CDF.  A
     * skeleton table is a text file which can be read by the
     * SkeletonCDF program to build a skeleton CDF.
     *
     * @param skeletonName is the pathname of the skeleton table
     *            to be created.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named <cdfName>.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the CDF from which the skeleton table
     *                will be created.  Do not enter an extension.<BR><BR>
     *               
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *            
     * @param format Specifies whether or not the FORMAT attribute is used
     *            when writing variable values (if the FORMAT attribute
     *            exists and an entry exists for the variable).<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *                 
     * @param screen Specifies whether or not the skeleton table is displayed
     *            on the terminal screen (written to the "standard output").
     *            If not, the skeleton table is written to a file.<BR><BR>
     *                       
     * @param page If the skeleton table is being displayed on the terminal
     *            screen, specifies whether or not the output is displayed
     *            one page (screen) at a time.<BR><BR>
     *                       
     * @param values Specifies which variable values are to
     *           be put in the skeleton table.  It may be one of
     *           the following...<BR><BR>
     *           <DL>
     *              <DT>CDFTools.NO_VALUES
     *                 <DD>Ignore all NRV data values.
     *              <DT>CDFTools.NRV_VALUES
     *                 <DD>Put NRV data values in the skeleton table.
     *              <DT>CDFTools.RV_VALUES
     *                 <DD>Put RV variable values in the skeleton table.
     *               <DT>CDFTools.ALL_VALUES
     *                 <DD>Put all variable values in the skeleton table.
     *               <DT>CDFTools.NAMED_VALUES
     *                  <DD>Put named variables values in the skeleton table.
     *                      This requires that valueList be non-null
     *           </DL><BR><BR>
     *
     * @param valueList the named variables to list values.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSize The number of 512-byte buffers to be used for the CDF's
     *            dotCDF file, staging file, and compression scratch file.
     *            If this qualifier is absent, default cache sizes chosen by
     *            the CDF library are used.  The cache sizes are specified
     *            with a comma-separated list of <number><type> pairs where
     *            <number> is the number of cache buffers and <type> is the
     *            type of file.  The file <type>'s are as follows: `d' for
     *            the dotCDF file, `s' for the staging file, and `c' for the
     *            compression scratch file.  For example, `200d,100s'
     *            specifies 200 cache buffers for the dotCDF file and 100
     *            cache buffers for the staging file.  The dotCDF file cache
     *            size can also be specified without the `d' <type> for
     *            compatibility with older CDF releases (eg. `200,100s').
     *            Note that not all of the file types must be specified.
     *            Those not specified will receive a default cache size.<BR><BR>
     *
     */
    public static void skeletonTable(String skeletonName,
				     String cdfName,
				     boolean log,
				     boolean format,
				     boolean neg2posfp0,
				     boolean statistics,
				     boolean screen,
				     boolean page,
				     int values,
				     String [] valueList,
				     int zMode,
				     int reportType,
				     int cacheSize)
	throws java.io.IOException, InterruptedException
    {

	// StringBuffers are faster than using String and +
	StringBuffer cmd = new StringBuffer();  
	cmd.append(cmdBase).append("skeletontable ");
	
	if (skeletonName != null) {
	    cmd.append("-skeleton ");
	    cmd.append(skeletonName+" ");
	}

	switch (values) {
	case NO_VALUES:
	    break;
	case NRV_VALUES:
	    cmd.append("-values nrv ");
	    break;
	case RV_VALUES:
	    cmd.append("-values rv ");
	    break;
	case ALL_VALUES:
	    cmd.append("-values all");
	    break;
	case NAMED_VALUES:
	    cmd.append("-values ");
	    for (int i=0; i < valueList.length; i++)
		cmd.append(valueList[i]+",");
	    cmd.setLength(cmd.length()-1);
	    cmd.append(" ");
	    break;
	}

	cmd.append("-");
	if (!log)
	    cmd.append("no");
	cmd.append("log ");
	    
	cmd.append("-zmode "+zMode+" ");

	cmd.append("-");
	if (!format)
	    cmd.append("no");
	cmd.append("format ");

	cmd.append("-");
	if (!neg2posfp0)
	    cmd.append("no");
	cmd.append("neg2posfp0 ");

	if (reportType != 0) {
	    cmd.append("-report \"");
	    if ((reportType & REPORT_ERRORS) != 0)
		cmd.append("errors, ");
	    if ((reportType & REPORT_WARNINGS) != 0)
		cmd.append("warnings, ");
	    if ((reportType & REPORT_INFORMATION) != 0)
		cmd.append("informationals, ");
	    cmd.setLength(cmd.length()-2); // chop off the last 2 chars
	    cmd.append("\" ");
	}

	if (cacheSize != 0) {
	    cmd.append("-cache ");
	    cmd.append("\""+cacheSize+"\"");
	}

	cmd.append("-");
	if (!statistics)
	    cmd.append("no");
	cmd.append("statistics ");

	cmd.append("-");
	if (!screen)
	    cmd.append("no");
	cmd.append("screen ");

	cmd.append("-");
	if (!page)
	    cmd.append("no");
	cmd.append("page ");
	
	cmd.append(cdfName);

	Process p = Runtime.getRuntime().exec(cmd.toString());
	int retcode = p.waitFor();

	// Echo any errors
	BufferedReader inStream = 
	    new BufferedReader(new InputStreamReader(p.getInputStream()));
	String line;
	while ((line = inStream.readLine()) != null) {
	    if (err instanceof PrintStream)
		((PrintStream)err).println(line);
	}
    }

    /**
     * skeletonTable produces a skeleton table from a CDF.  A
     * skeleton table is a text file which can be read by the
     * SkeletonCDF program to build a skeleton CDF.
     *
     * @param skeletonName is the pathname of the skeleton table
     *            to be created.  (Do not enter an extension because ".skt"
     *            is appended automatically).  If <B>null</B> is specified, the
     *            skeleton table is named <cdfName>.skt in the current
     *            directory<BR><BR>
     *
     * @param cdfName The pathname of the CDF from which the skeleton table
     *                will be created.  Do not enter an extension.<BR><BR>
     *
     * @param delete specifies whether or not the CDF should be deleted
     *               if it already exists.
     *
     * @param log   Specifies whether or not messages are displayed as the
     *             program executes.<BR><BR>
     *
     * @param neg2posfp0
     *            Specifies whether or not -0.0 is converted to 0.0 by the CDF
     *            library when read from a CDF.  -0.0 is an illegal floating
     *            point value on VAXes and DEC Alphas running OpenVMS.<BR><BR>
     *
     * @param statistics Specifies whether or not caching statistics
     *            are displayed at the end of each CDF.<BR><BR>
     *
     * @param zMode Specifies which zMode should be used.  May be one
     *            of the following...<BR><BR>
     *
     *               <DL>
     *               <DT>0
     *                 <DD>Indicates that zMode is disabled.
     *               <DT>1
     *                  <DD>Indicates that zMode/1 should be used (the
     *                       dimension variances of rVariables will be
     *                       preserved).
     *               <DT>2
     *                  <DD>Indicates that zMode/2 should be used (the
     *                       dimensions of rVariables having a variance
     *                       of NOVARY (false) are hidden.
     *               </DL><BR><BR>
     *
     * @param reportType Specifies the types of return status codes from the CDF
     *            library which should be reported/displayed.  <tt>report</tt>
     *            is a bit mask made up from the following
     *            CDFTools.NO_REPORTS, CDFTools.REPORT_ERRORS, 
     *            CDFTools.REPORT_WARNINGS and CDFTools.REPORT_INFORMATION<BR><BR>
     *
     * @param cacheSize The number of 512-byte buffers to be used for the CDF's
     *            dotCDF file, staging file, and compression scratch file.
     *            If this qualifier is absent, default cache sizes chosen by
     *            the CDF library are used.  The cache sizes are specified
     *            with a comma-separated list of <number><type> pairs where
     *            <number> is the number of cache buffers and <type> is the
     *            type of file.  The file <type>'s are as follows: `d' for
     *            the dotCDF file, `s' for the staging file, and `c' for the
     *            compression scratch file.  For example, `200d,100s'
     *            specifies 200 cache buffers for the dotCDF file and 100
     *            cache buffers for the staging file.  The dotCDF file cache
     *            size can also be specified without the `d' <type> for
     *            compatibility with older CDF releases (eg. `200,100s').
     *            Note that not all of the file types must be specified.
     *            Those not specified will receive a default cache size.<BR><BR>
     *
     */
    public static void skeletonCDF(String skeletonName,
				   String cdfName,
				   boolean delete,
				   boolean log,
				   boolean neg2posfp0,
				   boolean statistics,
				   int zMode,
				   int reportType,
				   int cacheSize)
	throws java.io.IOException, InterruptedException
    {

	// StringBuffers are faster than using String and +
	StringBuffer cmd = new StringBuffer();  
	cmd.append(cmdBase).append("skeletoncdf ");
	
	if (cdfName != null) {
	    cmd.append("-cdf ");
	    cmd.append(cdfName+" ");
	}

	cmd.append("-");
	if (!delete)
	    cmd.append("no");
	cmd.append("delete ");
	    
	cmd.append("-");
	if (!log)
	    cmd.append("no");
	cmd.append("log ");
	    
	cmd.append("-zmode "+zMode+" ");

	cmd.append("-");
	if (!neg2posfp0)
	    cmd.append("no");
	cmd.append("neg2posfp0 ");

	if (reportType != 0) {
	    cmd.append("-report ");
	    if ((reportType & REPORT_ERRORS) != 0)
		cmd.append("errors, ");
	    if ((reportType & REPORT_WARNINGS) != 0)
		cmd.append("warnings, ");
	    if ((reportType & REPORT_INFORMATION) != 0)
		cmd.append("informationals, ");
	    cmd.setLength(cmd.length()-2); // chop off the last 2 chars
	    cmd.append(" ");
	}

	if (cacheSize != 0) {
	    cmd.append("-cache ");
	    cmd.append("\""+cacheSize+"\"");
	}

	cmd.append("-");
	if (!statistics)
	    cmd.append("no");
	cmd.append("statistics ");
	
	cmd.append(skeletonName);

	Process p = Runtime.getRuntime().exec(cmd.toString());
	int returnCode = p.waitFor();

	// Echo any errors
	BufferedReader inStream = 
	    new BufferedReader(new InputStreamReader(p.getInputStream()));
	String line;
	while ((line = inStream.readLine()) != null) {
	    if (err instanceof PrintStream)
		((PrintStream)err).println(line);
	}
    }

} // CDFTools
