//$Id: DataTable.java,v 1.1.1.1 2009/01/20 16:27:23 liu Exp $

import javax.swing.JTable;
import javax.swing.table.TableModel;
import javax.swing.DefaultCellEditor;

/**
 * A Table which accepts arrays of primative or string values.
 *
 * <BR><BR>The DefaultDataTable provided in the JFC does not support arrays
 * of primatives.  The user must ensure that all primative values are
 * wrapped in the corresponding <code>java.lang.Number</code> class. This 
 * class takes care of this problem.
 *
 *
 *  <BR><BR><FONT SIZE=-1>
 *  1999, NASA/Goddard Space Flight Center
 *  This software may be used, copied, or redistributed as long as it is not
 *  sold or incorporated in any product meant for profit.  This copyright 
 *  notice must be reproduced on each copy made.  This routine is provided 
 *  as is without any express or implied warranties whatsoever.
 *  </FONT>
 *
 * @author Phil Williams
 * @version $Revision: 1.1.1.1 $
 *
 */
public class DataTable extends JTable {
    
    static final long serialVersionUID = 1L;

    // Constructors

    /**
     * Construct a default DataTable.
     */
    public DataTable() {
	this(new DefaultDataTableModel());
    }

    /**
     * Construct a default DataTable.
     * @param cellEditable boolean indicating whether the cells may be editable
     */
    public DataTable(boolean cellEditable) {
        this(new DefaultDataTableModel(cellEditable));
    }

    /**
     * Construct a DataTable using the given model.
     *
     * @param model A DefaultDataTableModel
     */
    public DataTable(DefaultDataTableModel model) {
	super(model);
	setUpFloatEditor();
	setUpDoubleEditor();
	setUpByteEditor();
	setUpShortEditor();
	setUpIntegerEditor();
	setUpLongEditor();
    }

    /**
     * Construct a DataTable using the given model.
     *
     * @param model A DefaultDataTableModel
     * @param cellEditable boolean indicating whether the cells may be editable
     */
    public DataTable(DefaultDataTableModel model, boolean cellEditable) {
        super(model);
	model.setCellEditable(cellEditable);
        setUpFloatEditor();
        setUpDoubleEditor();
        setUpByteEditor();
        setUpShortEditor();
        setUpIntegerEditor();
        setUpLongEditor();
    }

    /**
     * Construct a DataTable containing the given data.
     *
     * @param data the table data (must be an array).
     */
    public DataTable(Object data) {
	this(new DefaultDataTableModel(data));
    }

    /**
     * Construct a DataTable containing the given data.
     *
     * @param data the table data (must be an array).
     * @param cellEditable boolean indicating whether the cells may be editable
     */
    public DataTable(Object data, boolean cellEditable) {
        this(new DefaultDataTableModel(data, cellEditable));
    }

    /**
     * Construct a DataTable containing the given data using the specified
     * majority.
     *
     * @param data the table data (must be an array).
     * @param majority either DefaultDataTableModel.ROW_MAJOR or
     *      DefaultDataTableModel.COLUMN_MAJOR
     */
    public DataTable(Object data, int majority) {
	this(new DefaultDataTableModel(data, majority));
    }

    /**
     * Construct a DataTable containing the given data using the specified
     * majority.
     *
     * @param data the table data (must be an array).
     * @param majority either DefaultDataTableModel.ROW_MAJOR or
     *      DefaultDataTableModel.COLUMN_MAJOR
     * @param cellEditable boolean indicating whether the cells may be editable
     */
    public DataTable(Object data, int majority, boolean cellEditable) {
        this(new DefaultDataTableModel(data, majority, cellEditable));
    }

    // Private methods to set up the editors for the Number objects

    private void setUpFloatEditor() {
        //Set up the editor for the float cells.
        final FloatNumberField floatField = new FloatNumberField(0, 5);
        floatField.setHorizontalAlignment(FloatNumberField.RIGHT);

        DefaultCellEditor floatEditor = 
            new DefaultCellEditor(floatField) {
                //Override DefaultCellEditor's getCellEditorValue method
                //to return an Float, not a String:

                static final long serialVersionUID = 1L;

                public Object getCellEditorValue() {
                    return floatField.getFloat();
                }
            };
        this.setDefaultEditor(Float.class, floatEditor);
    }

    private void setUpDoubleEditor() {
        //Set up the editor for the double cells.
        final FloatNumberField floatField = new FloatNumberField(0, 5);
        floatField.setHorizontalAlignment(FloatNumberField.RIGHT);

        DefaultCellEditor doubleEditor = 
            new DefaultCellEditor(floatField) {
                //Override DefaultCellEditor's getCellEditorValue method
                //to return an Float, not a String:

                static final long serialVersionUID = 1L;

                public Object getCellEditorValue() {
                    return floatField.getDouble();
                }
            };
        this.setDefaultEditor(Double.class, doubleEditor);
    }

    private void setUpByteEditor() {
        //Set up the editor for the byte cells.
        final WholeNumberField wholeField = new WholeNumberField(0, 5);
        wholeField.setHorizontalAlignment(WholeNumberField.RIGHT);

        DefaultCellEditor byteEditor = 
            new DefaultCellEditor(wholeField) {
                //Override DefaultCellEditor's getCellEditorValue method
                //to return an Whole, not a String:

                static final long serialVersionUID = 1L;

                public Object getCellEditorValue() {
                    return wholeField.getByte();
                }
            };
        this.setDefaultEditor(Byte.class, byteEditor);
    }

    private void setUpShortEditor() {
        //Set up the editor for the short cells.
        final WholeNumberField wholeField = new WholeNumberField(0, 5);
        wholeField.setHorizontalAlignment(WholeNumberField.RIGHT);

        DefaultCellEditor shortEditor = 
            new DefaultCellEditor(wholeField) {
                //Override DefaultCellEditor's getCellEditorValue method
                //to return an Whole, not a String:

                static final long serialVersionUID = 1L;

                public Object getCellEditorValue() {
                    return wholeField.getShort();
                }
            };
        this.setDefaultEditor(Short.class, shortEditor);
    }

    private void setUpIntegerEditor() {
        //Set up the editor for the integer cells.
        final WholeNumberField wholeField = new WholeNumberField(0, 5);
        wholeField.setHorizontalAlignment(WholeNumberField.RIGHT);

        DefaultCellEditor integerEditor = 
            new DefaultCellEditor(wholeField) {
                //Override DefaultCellEditor's getCellEditorValue method
                //to return an Whole, not a String:

                static final long serialVersionUID = 1L;

                public Object getCellEditorValue() {
                    return wholeField.getInteger();
                }
            };
        this.setDefaultEditor(Integer.class, integerEditor);
    }

    private void setUpLongEditor() {
        //Set up the editor for the long cells.
        final WholeNumberField wholeField = new WholeNumberField(0, 5);
        wholeField.setHorizontalAlignment(WholeNumberField.RIGHT);

        DefaultCellEditor longEditor = 
            new DefaultCellEditor(wholeField) {
                //Override DefaultCellEditor's getCellEditorValue method
                //to return an Whole, not a String:

                static final long serialVersionUID = 1L;

                public Object getCellEditorValue() {
                    return wholeField.getLong();
                }
            };
        this.setDefaultEditor(Long.class, longEditor);
    }

} // DataTable
