// $Id: Epoch.java,v 1.1.1.1 2008/11/19 21:09:25 liu Exp $
package gsfc.nssdc.cdf.util;

import gsfc.nssdc.cdf.CDFException;
import java.text.NumberFormat;
import java.text.DecimalFormat;

import java.util.*;
import java.text.*;

/**
 *
 * <B>Example:</B>
 * <PRE> 
 * // Get the milliseconds to Aug 5, 1990 at 5:00
 * double ep = Epoch.compute(1990, 8, 5, 5, 0, 0, 0);
 * //Get the year, month, day, hour, minutes, seconds, milliseconds for ep
 * long times[] = Epoch.breakdown(ep);
 * for (int i=0;i&lt;times.length;i++)
 *     System.out.print(times[i]+" ");
 * System.out.println();
 * // Printout the epoch in various formats
 * System.out.println(Epoch.encode(ep));
 * System.out.println(Epoch.encode1(ep));
 * System.out.println(Epoch.encode2(ep));
 * System.out.println(Epoch.encode3(ep));
 * // Print out the date using format
 * String format = "<month> <dom.02>, <year> at <hour>:<min>";
 * System.out.println(Epoch.encodex(ep,format));
 * </PRE>
 */
public class Epoch implements gsfc.nssdc.cdf.CDFConstants {

    private static double MAX_EPOCH_BINARY = 3.15569519999998e14;

    private static int MAX_ePART_LEN =	25;

    private static String [] _monthToken = {    
	"Jan",
	"Feb",
	"Mar",
	"Apr",
	"May",
	"Jun",
	"Jul",
	"Aug",
	"Sep",
	"Oct",
	"Nov",
	"Dec"
    };

    /**
     * 
     * This function parses an input date/time string and returns an EPOCH
     * value.  The format must be exactly as shown below.  
     * Month abbreviations may be in any case and are always the first 
     * three letters of the month.
     *
     * <PRE>
     * Format:   dd-mmm-yyyy hh:mm:ss.mmm
     * Examples:  1-Apr-1990 03:05:02.000
     *           10-Oct-1993 23:45:49.999
     * </PRE>
     *
     * The expected format is the same as that produced by encodeEPOCH.
     *
     * @param inString the epoch in string representation
     * @return the value of the epoch represented by inString
     *
     * @exception CDFException if a bad epoch value is passed in inString
     */
    public static double parse(String inString)
	throws CDFException
    {
	long 
	    year   = 0, 
	    month  = 0, 
	    day    = 0, 
	    hour   = 0, 
	    minute = 0, 
	    second = 0, 
	    msec   = 0;
	String monthStr, secondStr;

	try {

  	    StringTokenizer st = new StringTokenizer(inString, " ");
	    String date = st.nextToken();
	    String time = st.nextToken();

	    // Get the date portion of the string
	    st = new StringTokenizer(date, "-");

	    day = Long.parseLong(st.nextToken());
	    monthStr = st.nextToken();
	    year = Long.parseLong(st.nextToken());
	    for (int monthX = 1; monthX <= 12; monthX++) {
		if (monthStr.equals(_monthToken[monthX-1])) {
		    month = monthX;
		    break;
		}
	    }
	    if (month == 0) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    
	    // Get the time portion
	    st = new StringTokenizer(time, ":");
	    hour   = Long.parseLong(st.nextToken());
	    minute = Long.parseLong(st.nextToken());
	    secondStr = st.nextToken();
	    st = new StringTokenizer(secondStr, ".");
	    second = Long.parseLong(st.nextToken());
	    msec   = Long.parseLong(st.nextToken());
	    
	    return compute(year, month, day, hour, minute, second, msec);
	} catch (Exception e) {
	    throw new CDFException(ILLEGAL_EPOCH_FIELD);
	}
    }

    /**
     * 
     * This function parses an input date/time string and returns an EPOCH
     * value.  The format must be exactly as shown below.  Note that if 
     * there are less than 7 digits after the decimal point, zeros (0's)
     * are assumed for the missing digits.
     *
     * <PRE>
     * Format:    yyyymmdd.ttttttt
     * Examples:  19950508.0000000
     *            19671231.58      (== 19671213.5800000)
     * </PRE>
     *
     * The expected format is the same as that produced by encodeEPOCH1.
     *
     * @param inString the epoch in string representation
     * @return the value of the epoch represented by inString
     *
     * @exception  CDFException if a bad epoch value is passed in inString
     */
    public static double parse1 (String inString)
	throws CDFException
    {
	StringTokenizer st = new StringTokenizer(inString, ".");
	String date = st.nextToken();
	String time = st.nextToken();
	long year = 0, month = 0, day = 0, hour = 0, minute = 0, 
	    second = 0, msec = 0, fractionL = 0;
	double fraction = 0.0;

	try {
	    year   = Long.parseLong(date.substring(0,4));
	    month  = Long.parseLong(date.substring(4,6));
	    day    = Long.parseLong(date.substring(6));
	    fractionL = Long.parseLong(time);
	    
	    fraction = ((double) fractionL) / 10000000.0;
	    hour = (long) (fraction * 24.0);
	    fraction -= (double) (hour / 24.0);
	    minute = (long) (fraction * 1440.0);
	    fraction -= (double) (minute / 1440.0);
	    second = (long) (fraction * 86400.0);
	    fraction -= (double) (second / 86400.0);
	    msec = (long) (fraction * 86400000.0);
	    return compute(year, month, day, hour, minute, second, msec);
	} catch (java.lang.NumberFormatException e) {
	    throw new CDFException(ILLEGAL_EPOCH_FIELD);
	}
    }

    /**
     * 
     * This function parses an input date/time string and returns an EPOCH
     * value.  The format must be exactly as shown below.
     *
     * <PRE>
     * Format:   yyyymmddhhmmss
     * Examples: 19950508000000
     *           19671231235959
     * </PRE>
     *
     * The expected format is the same as that produced by encodeEPOCH2.
     *
     * @param inString the epoch in string representation
     * @return the value of the epoch represented by inString
     *
     * @exception CDFException if a bad epoch value is passed in inString
     */
    public static double parse2 (String inString)
	throws CDFException
    {
	long year = 0, month = 0, day = 0, hour = 0, minute = 0, second = 0;
	try {
	    year   = Long.parseLong(inString.substring(0,4));
	    month  = Long.parseLong(inString.substring(4,6));
	    day    = Long.parseLong(inString.substring(6,8));
	    hour   = Long.parseLong(inString.substring(8,10));
	    minute = Long.parseLong(inString.substring(10,12));
	    second = Long.parseLong(inString.substring(12));
	    return compute(year, month, day, hour, minute, second, 0L);
	} catch (java.lang.NumberFormatException e) {
	    throw new CDFException(ILLEGAL_EPOCH_FIELD);
	}
    }
    
    /**
     * 
     * This function parses an input date/time string and returns an EPOCH
     * value.  The format must be exactly as shown below.
     *
     * <PRE>
     * Format:    yyyy-mm-ddThh:mm:ss.cccZ
     * Examples:  1990-04-01T03:05:02.000Z
     *            1993-10-10T23:45:49.999Z
     * </PRE>
     *
     * The expected format is the same as that produced by encodeEPOCH3.
     *
     * @param inString the epoch in string representation
     * @return the value of the epoch represented by inString
     *
     * @exception CDFException if a bad epoch value is passed in inString
     */
    public static double parse3 (String inString)
	throws CDFException
    {
	long year = 0, month = 0, day = 0, hour = 0, 
	    minute = 0, second = 0, msec = 0;

	String secondStr;

	try {
	    // Breakup the date and time fields
	    StringTokenizer st = new StringTokenizer(inString, "T");
	    String date = st.nextToken();
	    String tt = st.nextToken();
	    String time = tt.substring(0,tt.length() - 1);

	    // Breakup the date portion
	    st = new StringTokenizer(date, "-");
	    year = Long.parseLong(st.nextToken());
	    month = Long.parseLong(st.nextToken());
	    day = Long.parseLong(st.nextToken());

	    // Get the time portion
	    st = new StringTokenizer(time, ":");
	    hour   = Long.parseLong(st.nextToken());
	    minute = Long.parseLong(st.nextToken());
	    secondStr = st.nextToken();
	    st = new StringTokenizer(secondStr, ".");
	    second = Long.parseLong(st.nextToken());
	    msec   = Long.parseLong(st.nextToken());

	    return compute(year, month, day, hour, minute, second, msec);
	} catch (java.lang.Exception e) {
	    throw new CDFException(ILLEGAL_EPOCH_FIELD);
	}
    }

    /**
     * 
     * Converts an EPOCH value into a readable date/time string.
     *
     * <PRE>
     * Format:    dd-mmm-yyyy hh:mm:ss.ccc
     * Examples:  01-Apr-1990 03:05:02.000
     *            10-Oct-1993 23:45:49.999
     * </PRE>
     *
     * This format is the same as that expected by parse.
     *
     * @param epoch the epoch value
     *
     * @return A string representation of the epoch
     */
    public static String encode(double epoch)
    {
        if (epoch == -1.0E31) {
          return "31-Dec-9999 23:59:59.999";
        }

	return encodex (epoch, 
			"<dom.02>-<month>-<year> <hour>:<min>:<sec>.<fos>");
    }

    /**
     * 
     * Converts an EPOCH value into a readable date/time string.
     *
     * <PRE>
     * Format:    yyyymmdd.ttttttt
     * Examples:  19900401.3658893
     *            19611231.0000000
     * </PRE>
     *
     * This format is the same as that expected by parse1.
     *
     * @param epoch the epoch value
     *
     * @return A string representation of the epoch
     */
    public static String encode1 (double epoch)
    {
        if (epoch == -1.0E31) {
          return "99991231.9999999";
        }
        
	return encodex (epoch, "<year><mm.02><dom.02>.<fod.7>");
    }

    /**
     * 
     * Converts an EPOCH value into a readable date/time string.
     *
     * <PRE>
     * Format:     yyyymmddhhmmss
     * Examples:   19900401235959
     *             19611231000000
     * </PRE>
     *
     * This format is the same as that expected by parse2.
     *
     * @param epoch the epoch value
     *
     * @return A string representation of the epoch
     */
    public static String encode2 (double epoch)
    {
        if (epoch == -1.0E31) {
          return "99991231235959";
        }
        
	return encodex (epoch, "<year><mm.02><dom.02><hour><min><sec>");
    }

    /**
     * 
     * Converts an EPOCH value into a readable date/time string.
     *
     * <PRE>
     * Format:    yyyy-mm-ddThh:mm:ss.cccZ
     * Examples:  1990-04-01T03:05:02.000Z
     *            1993-10-10T23:45:49.999Z
     * </PRE>
     *
     * This format is the same as that expected by parse3.
     *
     * @param epoch the epoch value
     *
     * @return A string representation of the epoch
     */
    public static String encode3 (double epoch)
    {
        if (epoch == -1.0E31) {
          return "9999-12-31T23:59:59.999Z";
        }
        
	return 
	    encodex (epoch, 
		     "<year>-<mm.02>-<dom.02>T<hour>:<min>:<sec>.<fos>Z");
    }

    /**
     * 
     * Converts an EPOCH value into a readable date/time string using the
     * specified format.  See the C Reference Manual section 8.7 for details
     *
     * @param epoch the epoch value
     * @param formatString a string representing the desired 
     *      format of the epoch
     *
     * @return A string representation of the epoch according to formatString
     */
    public static String encodex(double epoch, String formatString)
    {
	StringBuffer 
	    encoded = new StringBuffer(), // Fully encoded epString
	    part = new StringBuffer(),    // Part being encoded
	    mod  = new StringBuffer();    // Part modifier.
	int ptr = 0;                // Current position in format string.
	int ptrD;		    // Location of decimal point.
	int ptrE;		    // Location of ending right angle bracket.
	int p;                      // temp position
	long [] components =  	    // EPOCH components.
	    new long[7]; //year, month, day, hour, minute, second, msec;

	if (formatString == null || formatString.equals(""))
	    return encode(epoch);
	
	char [] format = formatString.toCharArray();
	components = breakdown(epoch);

	// Scan format string.
	for (ptr = 0;ptr<format.length;ptr++) {
	    if (format[ptr] == '<') {

		// If next character is also a `<' (character stuffing), 
		// then append a `<' and move on.
		if (format[ptr+1] == '<') {
		    encoded.append("<");
		    System.out.println("append a literal \"<\"");

		    // char is not a literal '<'
		} else { 

		    // Find ending right angle bracket.
		    ptrE = formatString.indexOf('>',ptr + 1);
		    if (ptrE == -1) {
			encoded.append("?");
			System.out.println("appending ? (1)");
			return encoded.toString();
		    }

		    part.setLength(0);
		    for (p = ptr+1; p != ptrE; p++) 
			part.append(format[p]);
		    
		    // Check for a modifier.
		    ptrD = formatString.indexOf(".",ptr + 1);
		    mod = new StringBuffer();
		    if (ptrD != -1 && ptrD < ptrE) { // Modifier present
			for (p = ptrD+1; p != ptrE; p++) 
			    mod.append(format[p]);
		    }
		    ptr = ptrE;
		    String sPart = part.toString();

		    // Day (of month), <dom>.
		    if (sPart.indexOf("dom") == 0) {
			appendIntegerPart(encoded,components[2],0,
					  false,mod.toString());
		    } 
		    
		    // Day of year, <doy>.
		    else if (sPart.indexOf("doy") == 0) {
			long doy = 
			    JulianDay(components[0],components[1],components[2]) - 
			    JulianDay(components[0],1L,1L) + 1;
			
			appendIntegerPart(encoded, doy, 3,
					  true, mod.toString());
		    }
		    
		    // Month (3-character), <month>.
		    else if (sPart.indexOf("month") == 0) {
			encoded.append(_monthToken[(int)components[1] - 1]);
		    }
		    
		    // Month (digits), <mm>.
		    else if (sPart.indexOf("mm") ==0) {
			appendIntegerPart(encoded,components[1], 0,
					  false, mod.toString());
		    }
		    
		    // Year (full), <year>.
		    else if (sPart.indexOf("year") == 0) {
			appendIntegerPart(encoded, components[0], 4,
					  true,mod.toString());
		    }
		    
		    // Year (2-digit), <yr>.
		    else if (sPart.indexOf("yr") == 0) {
			long yr = components[0] % 100L;
			appendIntegerPart(encoded, yr, 2,
					  true, mod.toString());
		    }
		    
		    // Hour, <hour>.
		    else if (sPart.indexOf("hour") == 0) {
			appendIntegerPart(encoded, components[3], 2,
					  true,mod.toString());
		    }
		    
		    // Minute, <min>.
		    else if (sPart.indexOf("min") == 0) {
			appendIntegerPart(encoded, components[4], 2,
					  true,mod.toString());
		    }
		    
		    // Second, <sec>.
		    else if (sPart.indexOf("sec") == 0) {
			appendIntegerPart(encoded, components[5], 2,
					  true,mod.toString());
		    }
		    
		    // Fraction of second, <fos>.
		    else if (sPart.indexOf("fos") == 0) {
			double fos = ((double) components[6]) / 1000.0;
			appendFractionPart(encoded, fos, 3, mod.toString());
		    }
		    
		    // Fraction of day, <fod>.
		    else if (sPart.indexOf("fod") == 0) {
			double fod = 
			    ((double) components[3] / 24.0) +
			    ((double) components[4] / 1440.0) +
			    ((double) components[5] / 86400.0) +
			    ((double) components[6] / 86400000.0);
			appendFractionPart(encoded,fod,8,mod.toString());
		    }
		    
		    // Unknown/unsupported part.
		    else {
			encoded.append("?");
			System.out.println("append ? (2)");
		    }
		}
	    } else {
		if (ptr >= format.length) break;
		encoded.append(format[ptr]);
	    }
	}
	return encoded.toString();
    }

    private static void appendFractionPart (StringBuffer encoded, 
					    double fraction, 
					    int defaultWidth, 
					    String modifier)
    {
	StringBuffer ePart = new StringBuffer(MAX_ePART_LEN+1);
	int width;
	NumberFormat nf = NumberFormat.getNumberInstance(Locale.US);
	NumberFormat df;
	StringBuffer format = new StringBuffer();

	nf.setParseIntegerOnly(true);
	if (!modifier.equals("")) {            // modifier is present
	    try {
		width = nf.parse(modifier).intValue();
		if (width < 1) {
		    encoded.append("?");
		    System.out.println("append ? (3)");
		    return;
		}
	    } catch (java.text.ParseException e) {
		encoded.append("?");
		System.out.println("append ? (4)");
		return;
	    }
	} else
	    width = defaultWidth;
	
	for (int i = 0; i< width + 2; i++)
	    format.append("0");

	format.append(".");

	for (int i = 0; i < width; i++)
	    format.append("0");

	df = new DecimalFormat(format.toString());
	ePart.append(df.format(fraction));

	// If the encoded value was rounded up to 1.000..., then replace 
	// all of the digits after the decimal with `9's before appending.
	if (ePart.charAt(0) == '1') {
	    for (int i = 0; i < width; i++) ePart.setCharAt(i+2, '9');
	}
	String ePartStr = ePart.toString();
	char sp = new DecimalFormat().getDecimalFormatSymbols().getDecimalSeparator();
	appendPart(encoded,
//		   ePartStr.substring(ePartStr.indexOf(".")+1),
		   ePartStr.substring(ePartStr.indexOf(sp)+1),		   
		   width,false);
    }

    /**
     * Will append an integer to encoded.
     */
    private static void appendIntegerPart(StringBuffer encoded, 
					  long integer, 
					  int defaultWidth,
					  boolean defaultLeading0, 
					  String modifier)
    {
	StringBuffer ePart = new StringBuffer(MAX_ePART_LEN+1); 
	char [] modArray = modifier.toCharArray();
	int width; 
	boolean leading0;
	NumberFormat nf = NumberFormat.getNumberInstance(Locale.US);
	nf.setParseIntegerOnly(true);
	if (!modifier.equals("")) {
	    try {
		width = nf.parse(modifier).intValue();
		if (width < 0) {
		    encoded.append("?");
		    System.out.println("append ? (5)");
		    return;
		}
		leading0 = (modArray[0] == '0');
	    } catch (java.text.ParseException e) {
		encoded.append("?");
		System.out.println("append ? (6)");
		return;
	    }
	} else {
	    width = defaultWidth;
	    leading0 = defaultLeading0;
	}
	ePart.append(""+integer);
	appendPart(encoded, ePart.toString(), width, leading0);
    }
    
    /**
     * Append ePart to encoded.
     *
     * @param encoded The encoded epoch string.
     * @param ePart The part to append to encoded.
     * @param width The string length that the ePart should occupy. A width
     *        of zero indicates that the length of ePart should be used.
     * @param leading0 If true, that pad ePart with leading zeros.
     */
    private static void appendPart (StringBuffer encoded, 
				    String ePart, 
				    int width, 
				    boolean leading0)
    {
	int i;
	if (width == 0) {
	    encoded.append(ePart);
	} else {
	    int length = ePart.length();
	    if (length > width) {
		for (i = 0; i < width; i++) 
		    encoded.append("*");
	    } else {
		int pad = width - length;
		if (pad > 0) {
		    for (i = 0; i < pad; i++) 
			encoded.append((leading0 ? "0" : " "));
		}
		encoded.append(ePart);
	    }
	}
    }

    /**
     * 
     * Computes an EPOCH value based on its component parts.
     *
     * @param year the year
     * @param month the month
     * @param day the day
     * @param hour the hour
     * @param minute the minute
     * @param second the second 
     * @param msec the millisecond
     *
     * @return the epoch value
     *
     * @exception CDFException an ILLEGAL_EPOCH_FIELD if an illegal 
     *    component value is detected.
     */
    public static double compute(long year, 
				 long month, 
				 long day, 
				 long hour,
				 long minute, 
				 long second,
				 long msec)
	throws CDFException
    {
	long daysSince0AD, msecInDay;

	if (year == 9999 && month == 12 && day == 31 && hour == 23 &&
            minute == 59 && second == 59 && msec == 999) 
          return -1.0E31;
	/*
	 * Calculate the days since 0 A.D (1-Jan-0000).  If a value of zero 
	 * is passed in for `month', assume that `day' is the day-of-year
	 * (DOY) with January 1st being day 1.
	 */
	if (year < 0 || year > 9999) 
	    throw new CDFException(ILLEGAL_EPOCH_FIELD);
	if (month < 0 || month > 12)
	    throw new CDFException(ILLEGAL_EPOCH_FIELD);

	if (month == 0) {
	    if (day < 1 || day > 366) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    daysSince0AD = (JulianDay(year,1L,1L) + (day-1)) - 1721060L;
	} else {
	    if (day < 1 || day > 31) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    daysSince0AD = JulianDay(year,month,day) - 1721060L;
	}

	/*
	 * Calculate the millisecond in the day (with the first millisecond
	 * being 0). If values of zero are passed in for `hour', `minute',
	 * and `second', assume that `msec' is the millisecond in the day.
	 */
	if (hour == 0 && minute == 0 && second == 0) {
	    if (msec < 0 || msec > 86399999L) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    msecInDay = msec;
	} else {
	    if (hour < 0 || hour > 23) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    if (minute < 0 || minute > 59) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    if (second < 0 || second > 59) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    if (msec < 0 || msec > 999) 
		throw new CDFException(ILLEGAL_EPOCH_FIELD);
	    msecInDay = 
		(3600000L * hour) + 
		(60000L * minute) + 
		(1000 * second) + 
		msec;
	}
	
	// Return the milliseconds since 0 A.D.
	return ((86400000L * ((double) daysSince0AD)) + ((double) msecInDay));
    }

    /**
     * 
     * Breaks an EPOCH value down into its component parts.
     *
     * @param epoch the epoch value to break down
     * @return an array containing the epoch parts:
     *  <TABLE BORDER="0">
     *    <TR><TD ALIGN="CENTER">Index</TD><TD ALIGN="CENTER">Part</TD></TR>
     *    <TR><TD>0</TD><TD>year</TD></TR>
     *    <TR><TD>1</TD><TD>month</TD></TR>
     *    <TR><TD>2</TD><TD>day</TD></TR>
     *    <TR><TD>3</TD><TD>hour</TD></TR>
     *    <TR><TD>4</TD><TD>minute</TD></TR>
     *    <TR><TD>5</TD><TD>second</TD></TR>
     *    <TR><TD>6</TD><TD>msec</TD></TR>
     *  </TABLE>
     */
    public static long [] breakdown (double epoch)
    {
	long [] components = new long[7];
	long jd,i,j,k,l,n;
	double msec_AD, second_AD, minute_AD, hour_AD, day_AD;
	
	if (epoch == -1.0E31) {
	  components[0] = 9999;
	  components[1] = 12;
	  components[2] = 31;
	  components[3] = 23;
	  components[4] = 59;
	  components[5] = 59;
	  components[6] = 999;
          return components;
	}

	if (epoch < 0.0) epoch = -epoch;
	epoch = (MAX_EPOCH_BINARY < epoch ? MAX_EPOCH_BINARY : epoch);

	msec_AD = epoch;
	second_AD = msec_AD / 1000.0;
	minute_AD = second_AD / 60.0;
	hour_AD = minute_AD / 60.0;
	day_AD = hour_AD / 24.0;
	
	jd = (long) (1721060 + day_AD);
	l=jd+68569;
	n=4*l/146097;
	l=l-(146097*n+3)/4;
	i=4000*(l+1)/1461001;
	l=l-1461*i/4+31;
	j=80*l/2447;
	k=l-2447*j/80;
	l=j/11;
	j=j+2-12*l;
	i=100*(n-49)+i+l;
	
	components[0] = i;
	components[1] = j;
	components[2] = k;
	
	components[3] = (long) (hour_AD   % (double) 24.0);
	components[4] = (long) (minute_AD % (double) 60.0);
	components[5] = (long) (second_AD % (double) 60.0);
	components[6] = (long) (msec_AD   % (double) 1000.0);
	
	return components;
    }
    
    /**
     * JulianDay.
     * The year, month, and day are assumed to have already been validated.
     * This is the day since 0 AD/1 BC.  (Julian day may not be the
     * proper term.)
     */
    private static long JulianDay (long y, long m, long d)
    {
	return (367*y -
		7*(y+(m+9)/12)/4 -
		3*((y+(m-9)/7)/100+1)/4 +
		275*m/9 + 
		d +
		1721029);
    }
}
