#!/usr/local/bin/perl -w

#
# testPerlCDFii
#
#   This program tests the PerlCDF internal interface.  PerlCDF must be installed.
#   See README.
#
#  Usage:
#     perl testPerlCDFii.pl
#
# Written By:
#
#    Emily A. Greene
#    NSSDC / Hughes STX
#
#    17 October, 1995
#
#    Version 1.0
#    Modified by: Michael Liu  - 20 March, 1998
#    Version 2.0
#    Modified by: Michael Liu  - 07 January, 2005
#
#############################################################################
#
#  Translated from qst2ic.c v. 1.10 by J. Love
#
#  Differences include:
#     z Variable is 5 strings each 8 characters long not a 40 character buffer
#          tests changed accordingly
#     maxiumum records written is a variable
#     Arrays are all 1 Dimensional.  2 dimensionality handled within CDF.
#     Added section confirming rVARs recCount, etc.
#     Added a few informational messages
#     Tests Epoch routines
# Note: As Perl uses double, a variable of data type CDF_REAL4/FLOAT will 
#	cause some minor value deviation due to different floating-point type 
#	representations. If the value is returned from the CDF's C routine 
#	and compared with Perl value, an epsilon is used for checkng their
#	equality. 10**(-9) is used for double and 10**(-5) for 
#	float/real. No problem for CDF_REAL8 data type.
#     
#############################################################################

use strict;

BEGIN { unshift @INC,'/home/cdf/PerlCDF32_3/blib/arch',
                     '/home/cdf/PerlCDF32_3/blib/lib'; }
use CDF;

my $N_DIMS = 2;
my $DIM_0_SIZE = 2;
my $DIM_1_SIZE = 3;

my $zN_DIMSa = 1;
my $zDIM_0_SIZEa = 5;
my $zNUM_ELEMSa = 8;

my $encoding = NETWORK_ENCODING;
my $actual_encoding = NETWORK_ENCODING;
my $majority = ROW_MAJOR;
my $numDims = $N_DIMS;
my @dimSizes = ( $DIM_0_SIZE, $DIM_1_SIZE );
my $zNumDimsA = $zN_DIMSa;
my @zDimSizesA = ( $zDIM_0_SIZEa );
my $var1DataType = CDF_INT2;
my $var1DataTypeNew = CDF_UINT2 ;
my $var2DataType = CDF_REAL8 ;
my $zVarAdataType = CDF_CHAR ;
my $zVarAdataTypeNew = CDF_UCHAR ;
my $var1NumElements = 1 ;
my $var1NumElementsNew = 1 ;
my $var2NumElements = 1 ;
my $zVarAnumElements = $zNUM_ELEMSa ;
my $zVarAnumElementsNew = $zNUM_ELEMSa ;
my @var1Values = (1,2,3,4,5,6);
my @var2Values = (1.,2.,3.,4.,5.,6.);
my @zVarAvalues = (
  '11111111',
  '22222222',
  '33333333',
  '44444444',
  '55555555'
);
my @entry1Values = (1,2,3,4,5,6);
my $recNum = 0 ;
my $recStart = 0 ;
my $recCount = 1 ;
my $recInterval = 1 ;
my @counts = ( $DIM_0_SIZE, $DIM_1_SIZE );
my @intervals = ( 1, 1 );
my $zRecNum =  0;
my $zRecStart =  0;
my $zRecCount =  1;
my $zRecInterval = 1 ;
my @zCounts = ( $zDIM_0_SIZEa );
my @zIntervals = ( 1 );
my $entryNum = 2 ;
my $attrScope = GLOBAL_SCOPE ;
my $attrScope2 = VARIABLE_SCOPE ;
my $attrScope3 = VARIABLE_SCOPE ;
my $attrScope4 = VARIABLE_SCOPE ;
my $entryDataType = CDF_INT2 ;
my $entryDataTypeNew = CDF_UINT2 ;
my $entryNumElems = 1 ;
my $entryValue = 1 ;
my $var1RecVariance = VARY ;
my $var1RecVarianceNew = NOVARY ;
my $var2RecVariance = VARY ;
my $zVarArecVariance = VARY ;
my $zVarArecVarianceNew = NOVARY ;
my @var1DimVariances = ( VARY, VARY );
my @var1DimVariancesNew = ( NOVARY, NOVARY );
my @var2DimVariances = ( VARY, VARY );
my @zVarAdimVariances = ( VARY );
my @zVarAdimVariancesNew = ( NOVARY );
my $var1Name = "VAR1a";
my $var2Name = "VAR2a";
my $zVarAname = "zVARa1";
my $new_var1Name = "VAR1b";
my $new_var2Name = "VAR2b";
my $new_zVarAname = "zVARa2";
my $attrName = "ATTR1";
my $attrName2 = "ATTR2";
my $attrName3 = "ATTR3";
my $attrName4 = "ATTR4";
my $new_attrName = "ATTR1a";
my $rEntryValue =  4 ;
my $zEntryValue = 4.0 ;
my $pad1 = -999 ;
my $pad2 = -8.0 ;
my $pad3 =  "********" ;
my $extendRecs1 = 3;
my $extendRecs2 = 4;
my $extendRecs3 = 5;
my $allocRecs1 = 10 ;
my $allocRecs2 = 15 ;
my $allocRecs3 = 8 ;
my $nRvars = 2 ;
my @rVarNs = ( 0, 1 );
my @rVarsRecBuffer = (
  1,2,3,4,5,6,2.2,4.2,6.2,8.2,10.2,12.2
);
my $nZvars = 1 ;
my @zVarNs = ( 0 );
my @zVarsRecBuffer = (
  'aaaaaaaa',
  'bbbbbbbb',
  'cccccccc',
  'dddddddd',
  'eeeeeeee'
);

#############################################################################
# Display title.
#############################################################################

print "\nTesting Perl-CDF Internal/C interface\n\n";

#############################################################################
# Create CDF. 
#############################################################################
# Uncomment the following line if you want to create a V2.7 CDF file.
# CDF::CDFsetFileBackward(FILEBACKWARDon);

my $id; my $status;
$status = CDF::CDFlib(CREATE_, CDF_, "TEST",$numDims,\@dimSizes,\$id,
	   	      PUT_, CDF_ENCODING_, $encoding,
 			    CDF_MAJORITY_, $majority,
 		      NULL_);
if ($status < CDF_OK) {
  if ($status == CDF_EXISTS) {
    $status = CDF::CDFlib (OPEN_, CDF_, "TEST", \$id,
			   NULL_);
    QuitCDF("1.0", $status) if ($status < CDF_OK) ;

    $status = CDF::CDFlib (DELETE_, CDF_,
			   NULL_);
    QuitCDF ("1.1", $status) if ($status < CDF_OK) ;

    $status = CDF::CDFlib (CREATE_, CDF_, "TEST", $numDims, \@dimSizes, \$id,
			   PUT_, CDF_ENCODING_, $encoding,
				 CDF_MAJORITY_, $majority,
		    	   NULL_);
    QuitCDF ("1.2", $status) if ($status < CDF_OK) ;
  }
  else {
    QuitCDF ("1.3", $status);
    }
}

#############################################################################
# Create variables and set/confirm cache sizes, etc.
#############################################################################

my ($var1Num_out, $var2Num_out, $zVarAnum_out);
$status = CDF::CDFlib (CREATE_, rVAR_, $var1Name, $var1DataType, $var1NumElements,
				$var1RecVariance, \@var1DimVariances,
				\$var1Num_out, 
		       PUT_, rVAR_PADVALUE_, \$pad1,
		       NULL_);
QuitCDF ("2.0a", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (CREATE_, rVAR_, $var2Name, $var2DataType, $var2NumElements,
				$var2RecVariance, \@var2DimVariances,
				\$var2Num_out,
		       PUT_, rVAR_PADVALUE_, \$pad2,
		       NULL_);
QuitCDF ("2.0b", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (CREATE_, zVAR_, $zVarAname, $zVarAdataType, $zVarAnumElements,
				$zNumDimsA, \@zDimSizesA, $zVarArecVariance,
				\@zVarAdimVariances, \$zVarAnum_out,
		       PUT_, zVAR_PADVALUE_, \$pad3,
	      	       NULL_);
QuitCDF ("2.0c", $status) if ($status < CDF_OK) ;

my ($cacheOut1, $cacheOut2, $cacheOut3);
my ($pad1out, $pad2out, $pad3out);
$status = CDF::CDFlib (SELECT_, CDF_, $id,
				rVAR_, 0,
		       GET_, rVAR_PADVALUE_, \$pad1out,
		       SELECT_, rVAR_, 1,
		       GET_, rVAR_PADVALUE_, \$pad2out,
		       SELECT_, zVAR_, 0,
		       GET_, zVAR_PADVALUE_, \$pad3out,
		       NULL_);
QuitCDF ("2a.0", $status) if ($status < CDF_OK) ;
QuitCDF ("2a.3a", $status) if ($pad1out != $pad1) ;
QuitCDF ("2a.3b", $status) if ($pad2out != $pad2) ;
QuitCDF ("2a.3c", $status) if ($pad3out ne $pad3) ;

$status = CDF::CDFlib (SELECT_, rVAR_, 0,
			 	rVAR_CACHESIZE_, 4,
				zVAR_, 0,
				zVAR_CACHESIZE_, 8,
		       NULL_);
QuitCDF ("2a.4", $status) if ($status < CDF_OK) ;

#############################################################################
# Modify variables.
#############################################################################

$status = CDF::CDFlib (SELECT_, rVAR_, 0,
		       PUT_, rVAR_DATASPEC_, $var1DataTypeNew, $var1NumElementsNew,
			     rVAR_RECVARY_, $var1RecVarianceNew,
			     rVAR_DIMVARYS_, \@var1DimVariancesNew,
			     rVAR_INITIALRECS_, 1,
		       SELECT_, zVAR_, 0,
		       PUT_, zVAR_DATASPEC_, $zVarAdataTypeNew, $zVarAnumElementsNew,
			     zVAR_RECVARY_, $zVarArecVarianceNew,
			     zVAR_DIMVARYS_, \@zVarAdimVariancesNew,
			     zVAR_INITIALRECS_, 1,
		       NULL_);
QuitCDF ("2c.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Close CDF.
#############################################################################

$status = CDF::CDFlib (CLOSE_, CDF_,
		       NULL_);
QuitCDF ("3.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Reopen CDF.
#############################################################################

$status = CDF::CDFlib (OPEN_, CDF_, "TEST", \$id,
		       NULL_);
QuitCDF ("4.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Delete CDF.
#############################################################################
$status = CDF::CDFlib (DELETE_, CDF_,
		       NULL_);
QuitCDF ("5.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Create CDF again (previous delete will allow this).
#############################################################################

$status = CDF::CDFlib (CREATE_, CDF_, "TEST", $numDims, \@dimSizes, \$id,
		       PUT_, CDF_ENCODING_, $encoding,
			     CDF_MAJORITY_, $majority,
		       NULL_);
QuitCDF ("6.0", $status) if ($status < CDF_OK) ;

# Uncomment the following lines if MD5 checksum option is needed.
$status = CDF::CDFlib (PUT_, CDF_CHECKSUM_, MD5_CHECKSUM,
                       NULL_);
QuitCDF ("6.1", $status) if ($status < CDF_OK) ;
# Or, use this way:
#$status = CDF::CDFsetChecksum ($id, MD5_CHECKSUM);
QuitCDF ("6.1", $status) if ($status < CDF_OK) ;

#############################################################################
# Create variables.
#############################################################################

$status = CDF::CDFlib (CREATE_, rVAR_, $var1Name, $var1DataType, $var1NumElements,
				 $var1RecVariance, \@var1DimVariances,
				 \$var1Num_out,
		       PUT_, rVAR_ALLOCATERECS_, $allocRecs1,
			     rVAR_BLOCKINGFACTOR_, $extendRecs1,
		       NULL_);
QuitCDF ("7.0a", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (CREATE_, rVAR_, $var2Name, $var2DataType, $var2NumElements,
				 $var2RecVariance, \@var2DimVariances,
				 \$var2Num_out,
		       PUT_, rVAR_ALLOCATERECS_, $allocRecs2,
			     rVAR_BLOCKINGFACTOR_, $extendRecs2,
		       NULL_);
QuitCDF ("7.0b", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (CREATE_, zVAR_, $zVarAname, $zVarAdataType, $zVarAnumElements,
				$zNumDimsA, \@zDimSizesA, $zVarArecVariance,
				\@zVarAdimVariances, \$zVarAnum_out,
	       	       PUT_, zVAR_ALLOCATERECS_, $allocRecs3,
			     zVAR_BLOCKINGFACTOR_, $extendRecs3,
		       NULL_);
QuitCDF ("7.0c", $status) if ($status < CDF_OK) ;

#############################################################################
# PUT to variables.
#############################################################################

$status = CDF::CDFlib (SELECT_, rVARs_RECNUMBER_, $recNum,
		NULL_);
QuitCDF ("8.0", $status) if ($status < CDF_OK) ;

my ($x0, $x1, $i, @indices, $temp1, $temp2);
for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
      $indices[0] = $x0;
      $indices[1] = $x1;
      $temp1 = $var1Values[$x0*$DIM_1_SIZE + $x1];
      $temp2 = $var2Values[$x0*$DIM_1_SIZE + $x1];
      $status = CDF::CDFlib (SELECT_, rVARs_DIMINDICES_, \@indices,
				      rVAR_, $var1Num_out,
			     PUT_, rVAR_DATA_, \$temp1,
			     SELECT_, rVAR_, $var2Num_out,
			     PUT_, rVAR_DATA_, \$temp2,
			     NULL_);
      QuitCDF ("8.1", $status) if ($status < CDF_OK) ;
   }
 }
my $maxRecWritten = 0;

$status = CDF::CDFlib (SELECT_, zVAR_, $zVarAnum_out,
				zVAR_RECNUMBER_, $zRecNum,
		       NULL_);
QuitCDF ("8.0z", $status) if ($status < CDF_OK) ;

my @zIndicesA;
for ($x0 = 0; $x0 < $zDIM_0_SIZEa; $x0++) {
   $zIndicesA[0] = $x0;
   $temp1 = $zVarAvalues[$x0];
   $status = CDF::CDFlib (SELECT_, zVAR_DIMINDICES_, \@zIndicesA,
			  PUT_, zVAR_DATA_, \$temp1,
			  NULL_);
   QuitCDF ("8.1z", $status) if ($status < CDF_OK) ;
}

#############################################################################
# GET from the variables.
#############################################################################

my ($var1Value_out, $var2Value_out);
for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
      $indices[0] = $x0;
      $indices[1] = $x1;
      $status = CDF::CDFlib (SELECT_, rVARs_DIMINDICES_, \@indices,
				      rVAR_, $var1Num_out,
			     GET_, rVAR_DATA_, \$var1Value_out,
			     SELECT_, rVAR_, $var2Num_out,
			     GET_, rVAR_DATA_, \$var2Value_out,
			     NULL_);
      QuitCDF ("9.0", $status) if ($status < CDF_OK) ;

      QuitCDF ("9.1", $status) if ($var1Value_out != $var1Values[$x0*$DIM_1_SIZE + $x1]) ;
      QuitCDF ("9.2", $status) if ($var2Value_out != $var2Values[$x0*$DIM_1_SIZE + $x1]) ;
   }
 }

my $zVarAvalue_out;
for ($x0 = 0; $x0 < $zDIM_0_SIZEa; $x0++) {
   $zIndicesA[0] = $x0;
   $status = CDF::CDFlib (SELECT_, zVAR_DIMINDICES_, \@zIndicesA,
			  GET_, zVAR_DATA_, \$zVarAvalue_out,
			  NULL_);
   QuitCDF ("9.1z", $status) if ($status < CDF_OK) ;
   QuitCDF ("9.2z", $status) if ($zVarAvalue_out ne $zVarAvalues[$x0]) ;
}

#############################################################################
# HyperPUT to the variables.
#############################################################################

for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
      $var1Values[$x0*$DIM_1_SIZE + $x1] = -$var1Values[$x0*$DIM_1_SIZE + $x1];
      $var2Values[$x0*$DIM_1_SIZE + $x1] = -$var2Values[$x0*$DIM_1_SIZE + $x1];
   }
 }

$indices[0] = 0;
$indices[1] = 0;

$status = CDF::CDFlib (SELECT_, rVARs_RECNUMBER_, $recStart,
				rVARs_RECCOUNT_, $recCount,
				rVARs_RECINTERVAL_, $recInterval,
				rVARs_DIMINDICES_, \@indices,
				rVARs_DIMCOUNTS_, \@counts,
				rVARs_DIMINTERVALS_, \@intervals,
				rVAR_, $var1Num_out,
		       PUT_, rVAR_HYPERDATA_, \@var1Values,
		       SELECT_, rVAR_, $var2Num_out,
		       PUT_, rVAR_HYPERDATA_, \@var2Values,
		       NULL_);
QuitCDF ("10.0", $status) if ($status < CDF_OK) ;
for ($x0 = 0; $x0 < $zDIM_0_SIZEa; $x0++) {
      $zVarAvalues[$x0] = $zVarAvalues[$x0] + $zVarAvalues[0];
 }
$zIndicesA[0] = 0; 
$status = CDF::CDFlib (SELECT_, zVAR_RECNUMBER_, $zRecStart,
				zVAR_RECCOUNT_, $zRecCount,
				zVAR_RECINTERVAL_, $zRecInterval,
				zVAR_DIMINDICES_, \@zIndicesA,
				zVAR_DIMCOUNTS_, \@zCounts,
				zVAR_DIMINTERVALS_, \@zIntervals,
		       PUT_, zVAR_HYPERDATA_, \@zVarAvalues,
		       NULL_);
QuitCDF ("10.0z", $status) if ($status < CDF_OK) ;
############################################################################g
# HyperGET from variables.
#############################################################################

my (@var1Buffer_out, @var2Buffer_out);
$status = CDF::CDFlib (SELECT_, rVARs_RECNUMBER_, $recStart,
				rVARs_RECCOUNT_, $recCount,
				rVARs_RECINTERVAL_, $recInterval,
				rVARs_DIMINDICES_, \@indices,
				rVARs_DIMCOUNTS_, \@counts,
				rVARs_DIMINTERVALS_, \@intervals,
				rVAR_, $var1Num_out,
		       GET_, rVAR_HYPERDATA_, \@var1Buffer_out,
		       SELECT_, rVAR_, $var2Num_out,
		       GET_, rVAR_HYPERDATA_, \@var2Buffer_out,
		       NULL_);
QuitCDF ("11.0", $status) if ($status < CDF_OK) ;

for ($x0 = 0; $x0 < $DIM_0_SIZE; $x0++) {
   for ($x1 = 0; $x1 < $DIM_1_SIZE; $x1++) {
   	QuitCDF ("11.1", $status) if ($var1Buffer_out[$x0*$DIM_1_SIZE + $x1] != $var1Values[$x0*$DIM_1_SIZE + $x1]);
   	QuitCDF ("11.2", $status) if ($var2Buffer_out[$x0*$DIM_1_SIZE + $x1] != $var2Values[$x0*$DIM_1_SIZE + $x1]);
   }
  }
  
#############################################################################
# Confirm hyper parameters for a rVariables.
#############################################################################

my ($recStartOut, $recCountOut, $recIntervalOut, @indicesOut, @countsOut, @intervalsOut);
$status = CDF::CDFlib (SELECT_, rVAR_NAME_, $var1Name,
		       CONFIRM_, rVARs_RECNUMBER_, \$recStartOut,
				 rVARs_RECCOUNT_, \$recCountOut,
				 rVARs_RECINTERVAL_, \$recIntervalOut,
				 rVARs_DIMINDICES_, \@indicesOut,
				 rVARs_DIMCOUNTS_, \@countsOut,
				 rVARs_DIMINTERVALS_, \@intervalsOut,
		       NULL_);
QuitCDF ("11a.0r", $status) if ($status < CDF_OK) ;

QuitCDF ("11a.1r", $status) if ($recStartOut != $recStart) ;
QuitCDF ("11a.2r", $status) if ($recCountOut != $recCount) ;
QuitCDF ("11a.3r", $status) if ($recIntervalOut != $recInterval) ;

my $dimN;
for ($dimN = 0; $dimN < $N_DIMS; $dimN++) {
   QuitCDF ("11a.4r", $status) if ($indicesOut[$dimN] != $indices[$dimN]) ;
   QuitCDF ("11a.5r", $status) if ($countsOut[$dimN] != $counts[$dimN]) ;
   QuitCDF ("11a.6r", $status) if ($intervalsOut[$dimN] != $intervals[$dimN]) ;
}

#############################################################################

my @zVarAbuffer_out;
$status = CDF::CDFlib (GET_, zVAR_HYPERDATA_, \@zVarAbuffer_out,
		       NULL_);
QuitCDF ("11.0z", $status) if ($status < CDF_OK) ;

for ($x0 = 0; $x0 < $zDIM_0_SIZEa; $x0++) {
	QuitCDF ("11.1z", $status) if ($zVarAbuffer_out[$x0] ne $zVarAvalues[$x0]);
 }

#############################################################################
# Confirm hyper parameters for a zVariable.
#############################################################################

$status = CDF::CDFlib (CONFIRM_, zVAR_RECNUMBER_, \$recStartOut,
				 zVAR_RECCOUNT_, \$recCountOut,
				 zVAR_RECINTERVAL_, \$recIntervalOut,
				 zVAR_DIMINDICES_, \@indicesOut,
				 zVAR_DIMCOUNTS_, \@countsOut,
				 zVAR_DIMINTERVALS_, \@intervalsOut,
		       NULL_);
QuitCDF ("11a.0", $status) if ($status < CDF_OK) ;

QuitCDF ("11a.1", $status) if ($recStartOut != $zRecStart) ;
QuitCDF ("11a.2", $status) if ($recCountOut != $zRecCount) ;
QuitCDF ("11a.3", $status) if ($recIntervalOut != $zRecInterval) ;

for ($dimN = 0; $dimN < $zN_DIMSa; $dimN++) {
   QuitCDF ("11a.4", $status) if ($indicesOut[$dimN] != $zIndicesA[$dimN]) ;
   QuitCDF ("11a.5", $status) if ($countsOut[$dimN] != $zCounts[$dimN]) ;
   QuitCDF ("11a.6", $status) if ($intervalsOut[$dimN] != $zIntervals[$dimN]) ;
}

#############################################################################
# Set/confirm sequential access position for a zVariable (and read/write a
# value).
#############################################################################

my $recNumOut;
$status = CDF::CDFlib (SELECT_, zVAR_SEQPOS_, $zRecStart, \@zIndicesA,
		       GET_, zVAR_SEQDATA_, \$zVarAvalue_out,
		       PUT_, zVAR_SEQDATA_, \$zVarAvalue_out,
		       CONFIRM_, zVAR_SEQPOS_, \$recNumOut, \@indicesOut,
		       NULL_);
QuitCDF ("11b.0", $status) if ($status < CDF_OK) ;

QuitCDF ("11b.1", $status) if ($recNumOut != $zRecStart) ;
QuitCDF ("11b.2", $status) if ($indicesOut[0] != $zIndicesA[0] + 2) ;

#############################################################################
# Create attributes.
#############################################################################

my $attrNum_out;
$status = CDF::CDFlib (CREATE_, ATTR_, $attrName, $attrScope, \$attrNum_out,
				ATTR_, $attrName2, $attrScope2, \$attrNum_out,
				ATTR_, $attrName3, $attrScope3, \$attrNum_out,
				ATTR_, $attrName4, $attrScope4, \$attrNum_out,
		       NULL_);
QuitCDF ("12.0", $status) if ($status < CDF_OK) ;

#############################################################################
# PUT to attributes.
#############################################################################

$status = CDF::CDFlib (SELECT_, ATTR_, 0,
				gENTRY_, $entryNum,
		       PUT_, gENTRY_DATA_, $entryDataType, $entryNumElems,
				     \$entryValue,
		       SELECT_, ATTR_, 1,
				rENTRY_NAME_, $var2Name,
		       PUT_, rENTRY_DATA_, CDF_BYTE, 1, \$rEntryValue,
		       SELECT_, ATTR_, 2,
				zENTRY_NAME_, $zVarAname,
		       PUT_, zENTRY_DATA_, CDF_REAL8, 1, \$zEntryValue,
		       SELECT_, ATTR_, 3,
		       PUT_, zENTRY_DATA_, CDF_INT2, 6, \@entry1Values,
		       NULL_);
QuitCDF ("13.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Confirm entry numbers.
#############################################################################

my ($entryNumOut1, $entryNumOut2, $entryNumOut3);
$status = CDF::CDFlib (CONFIRM_, gENTRY_, \$entryNumOut1,
				 rENTRY_, \$entryNumOut2,
				 zENTRY_, \$entryNumOut3,
		       NULL_);
QuitCDF ("13a.0", $status) if ($status < CDF_OK) ;

QuitCDF ("13a.1", $status) if ($entryNumOut1 != 1) ;
QuitCDF ("13a.2", $status) if ($entryNumOut2 != 1) ;
QuitCDF ("13a.3", $status) if ($entryNumOut3 != 0) ;

#############################################################################
# GET from attributes.
#############################################################################

my ($entryValue_out, $rEntryValueOut, $zEntryValueOut, @zEntryValuesOut);
$status = CDF::CDFlib (SELECT_, ATTR_, 0,
				gENTRY_, $entryNum,
		       CONFIRM_, CURgENTRY_EXISTENCE_,
		       GET_, gENTRY_DATA_, \$entryValue_out,
		       NULL_);
QuitCDF ("14.0.1", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 1,
				rENTRY_, 1,
		       CONFIRM_, CURrENTRY_EXISTENCE_,
		       GET_, rENTRY_DATA_, \$rEntryValueOut,
		       NULL_);
QuitCDF ("14.0.2", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 2,
				zENTRY_, 0,
		       CONFIRM_, CURzENTRY_EXISTENCE_,
		       GET_, zENTRY_DATA_, \$zEntryValueOut,
		       NULL_);
QuitCDF ("14.0.3", $status) if ($status < CDF_OK) ;
QuitCDF ("14.1", $status) if ($entryValue_out != $entryValue) ;
QuitCDF ("14.2", $status) if ($rEntryValue != $rEntryValueOut) ;
QuitCDF ("14.3", $status) if ($zEntryValue != $zEntryValueOut) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 3,
                                zENTRY_, 0,
                       CONFIRM_, CURzENTRY_EXISTENCE_,
                       GET_, zENTRY_DATA_, \@zEntryValuesOut,
                       NULL_);
QuitCDF ("14.4.3", $status) if ($status < CDF_OK) ;
for ($i=0; $i<6; $i++) {
  QuitCDF ("14.4", $status) if ($entry1Values[$i] != $zEntryValuesOut[$i]) ;
}
#############################################################################
# Confirm existence of variables/attributes/entries.
#############################################################################

$status = CDF::CDFlib (CONFIRM_, zVAR_EXISTENCE_, $zVarAname,
				 rVAR_EXISTENCE_, $var1Name,
				 ATTR_EXISTENCE_, $attrName3,
		       NULL_);
QuitCDF ("14a.0", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 0,
		       CONFIRM_, gENTRY_EXISTENCE_, $entryNum,
		       SELECT_, ATTR_, 1,
		       CONFIRM_, rENTRY_EXISTENCE_, 1,
		       SELECT_, ATTR_, 2,
		       CONFIRM_, zENTRY_EXISTENCE_, 0,
		       NULL_);
QuitCDF ("14a.1", $status) if ($status < CDF_OK) ;

#############################################################################
# Get CDF documentation.
#############################################################################

my ($version_out, $release_out, $increment_out, $subincrement_out, $CopyRightText);
$status = CDF::CDFlib (GET_, LIB_VERSION_, \$version_out,
			     LIB_RELEASE_, \$release_out,
			     LIB_INCREMENT_, \$increment_out,
			     LIB_subINCREMENT_, \$subincrement_out,
			     LIB_COPYRIGHT_, \$CopyRightText,
		       NULL_);
QuitCDF ("15.0", $status) if ($status < CDF_OK) ;
print "CDF Library Version $version_out, Release $release_out
Increment $increment_out and subIncrement $subincrement_out
Copyright: $CopyRightText\n";

#############################################################################
# Inquire CDF.
#############################################################################

my ($formatOut, $numDims_out, @dimSizes_out, $encoding_out, $majority_out);
my ($maxRec_out, $numRvars, $numZvars, $numAttrs_out);
$status = CDF::CDFlib (GET_, CDF_FORMAT_, \$formatOut,
			     rVARs_NUMDIMS_, \$numDims_out,
			     rVARs_DIMSIZES_, \@dimSizes_out,
			     CDF_ENCODING_, \$encoding_out,
			     CDF_MAJORITY_, \$majority_out,
		 	     rVARs_MAXREC_, \$maxRec_out,
			     CDF_NUMrVARS_, \$numRvars,
			     CDF_NUMzVARS_, \$numZvars,
			     CDF_NUMATTRS_, \$numAttrs_out,
		       NULL_);
QuitCDF ("16.0", $status) if ($status < CDF_OK) ;

QuitCDF ("16.1o", $status) if ($formatOut != SINGLE_FILE) ;
QuitCDF ("16.1", $status) if ($numDims_out != $numDims) ;

my $x;
for ($x = 0; $x < $N_DIMS; $x++) {
   QuitCDF ("16.2", $status) if ($dimSizes_out[$x] != $dimSizes[$x]) ;
   }
QuitCDF ("16.3", $status) if ($encoding_out != $actual_encoding) ;
QuitCDF ("16.4", $status) if ($majority_out != $majority) ;
QuitCDF ("16.5", $status) if ($maxRec_out != $maxRecWritten) ;
QuitCDF ("16.6", $status) if ($numRvars != 2) ;
QuitCDF ("16.6z", $status) if ($numZvars != 1) ;
QuitCDF ("16.7", $status) if ($numAttrs_out != 4) ;

#############################################################################
# Inquire numbers.
#############################################################################

my ($varNum_out1, $varNum_out2);
$status = CDF::CDFlib (GET_, ATTR_NUMBER_, $attrName3, \$attrNum_out,
			     rVAR_NUMBER_, $var2Name, \$varNum_out1,
			     zVAR_NUMBER_, $zVarAname, \$varNum_out2,
		       NULL_);
QuitCDF ("16a.0", $status) if ($status < CDF_OK) ;

QuitCDF ("16a.1", $status) if ($attrNum_out != 2) ;
QuitCDF ("16a.2", $status) if ($varNum_out1 != 1) ;
QuitCDF ("16a.3", $status) if ($varNum_out2 != 0) ;

#############################################################################
# Rename variables.
#############################################################################

$status = CDF::CDFlib (SELECT_, rVAR_NAME_, $var1Name,
		       PUT_, rVAR_NAME_, $new_var1Name,
		       NULL_);
QuitCDF ("17.0a", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, rVAR_NAME_, $var2Name,
		       PUT_, rVAR_NAME_, $new_var2Name,
		       NULL_);
QuitCDF ("17.0b", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, zVAR_NAME_, $zVarAname,
		       PUT_, zVAR_NAME_, $new_zVarAname,
		       NULL_);
QuitCDF ("17.0c", $status) if ($status < CDF_OK) ;

#############################################################################
# Read/write multiple variable data. 
#############################################################################

$status = CDF::CDFlib (SELECT_, rVARs_RECNUMBER_, 2,
		       PUT_, rVARs_RECDATA_, $nRvars, \@rVarNs, \@rVarsRecBuffer,
		       SELECT_, zVARs_RECNUMBER_, 2,
		       PUT_, zVARs_RECDATA_, $nZvars, \@zVarNs, \@zVarsRecBuffer,
		       NULL_);
QuitCDF ("17.0d", $status) if ($status < CDF_OK) ;
$maxRecWritten = $maxRecWritten + 2;

my @rVarsRecBufferOut;
my @zVarsRecBufferOut;
my $j;
$status = CDF::CDFlib (GET_, rVARs_RECDATA_, $nRvars, \@rVarNs, \@rVarsRecBufferOut,
			 NULL_);
QuitCDF ("17.0e", $status) if ($status < CDF_OK) ;
for ($i=0; $i<12; $i++) {
# print "$rVarsRecBufferOut[$i] vs $rVarsRecBuffer[$i] \n"; 
  QuitCDF ("17.0f", $status) # epsilon of 10^(-5) for float, 10^(-9) for double 
 	if (abs($rVarsRecBufferOut[$i]-$rVarsRecBuffer[$i]) > 10**(-5)); 
#print $rVarsRecBuffer[$i],' ',$rVarsRecBufferOut[$i],"\n";
}
$status = CDF::CDFlib (GET_, zVARs_RECDATA_, $nZvars, \@zVarNs, \@zVarsRecBufferOut,
			 NULL_);
QuitCDF ("17.0g", $status) if ($status < CDF_OK) ;
for ($j=0; $j<5; $j++) {
  QuitCDF ("17.0h", $status)
        if ($zVarsRecBufferOut[$j] ne $zVarsRecBuffer[$j]);
}
#
#############################################################################
# Inquire variables.
#############################################################################

my ($var1Name_out, $var1DataType_out, $var1NumElements_out, $var1RecVariance_out);
my (@var1DimVariances_out, $extendRecsOut1, $maxAllocOut1, $maxRecOut1);
my ($nIndexRecsOut1, $nIndexEntriesOut1);
$status = CDF::CDFlib (SELECT_, rVAR_, $var1Num_out,
		       GET_, rVAR_NAME_, \$var1Name_out,
			     rVAR_DATATYPE_, \$var1DataType_out,
			     rVAR_NUMELEMS_, \$var1NumElements_out,
			     rVAR_RECVARY_, \$var1RecVariance_out,
			     rVAR_DIMVARYS_, \@var1DimVariances_out,
			     rVAR_BLOCKINGFACTOR_, \$extendRecsOut1,
			     rVAR_MAXallocREC_, \$maxAllocOut1,
			     rVAR_MAXREC_, \$maxRecOut1,
			     rVAR_nINDEXRECORDS_, \$nIndexRecsOut1,
			     rVAR_nINDEXENTRIES_, \$nIndexEntriesOut1,
		       CONFIRM_, rVAR_, \$var1Num_out,
		       NULL_);
QuitCDF ("18.0a", $status) if ($status < CDF_OK) ;

QuitCDF ("18.11", $status) if ($var1Name_out ne $new_var1Name) ;
QuitCDF ("18.12", $status) if ($var1DataType_out != $var1DataType) ;

QuitCDF ("18.13", $status) if ($var1NumElements_out != $var1NumElements) ;
QuitCDF ("18.14", $status) if ($var1RecVariance_out != $var1RecVariance) ;
QuitCDF ("18.14a", $status) if ($var1Num_out != 0) ;
QuitCDF ("18.14b", $status) if ($extendRecsOut1 != $extendRecs1) ;
QuitCDF ("18.14c", $status) if ($maxAllocOut1 + 1 != $allocRecs1) ;
QuitCDF ("18.14d", $status) if ($maxRecOut1 != $maxRecWritten) ;

my $dim_n;
for ($dim_n = 0; $dim_n < $numDims; $dim_n++) {
   if ($var1DimVariances_out[$dim_n] != $var1DimVariances[$dim_n]) {
     QuitCDF ("18.14", $status);
   }
}

my ($var2Name_out, $var2DataType_out, $var2NumElements_out, $var2RecVariance_out);
my (@var2DimVariances_out, $extendRecsOut2, $maxAllocOut2, $maxRecOut2);
my ($nIndexRecsOut2, $nIndexEntriesOut2);
$status = CDF::CDFlib (SELECT_, rVAR_, $var2Num_out,
		       GET_, rVAR_NAME_, \$var2Name_out,
			     rVAR_DATATYPE_, \$var2DataType_out,
			     rVAR_NUMELEMS_, \$var2NumElements_out,
			     rVAR_RECVARY_, \$var2RecVariance_out,
			     rVAR_DIMVARYS_, \@var2DimVariances_out,
			     rVAR_BLOCKINGFACTOR_, \$extendRecsOut2,
			     rVAR_MAXallocREC_, \$maxAllocOut2,
			     rVAR_MAXREC_, \$maxRecOut2,
			     rVAR_nINDEXRECORDS_, \$nIndexRecsOut2,
			     rVAR_nINDEXENTRIES_, \$nIndexEntriesOut2,
		       CONFIRM_, rVAR_, \$var2Num_out,
		       NULL_);
QuitCDF ("18.0b", $status) if ($status < CDF_OK) ;

QuitCDF ("18.21", $status) if ($var2Name_out ne $new_var2Name) ;
QuitCDF ("18.22", $status) if ($var2DataType_out != $var2DataType) ;
QuitCDF ("18.23", $status) if ($var2NumElements_out != $var2NumElements) ;
QuitCDF ("18.24", $status) if ($var2RecVariance_out != $var2RecVariance) ;
QuitCDF ("18.24a", $status) if ($var2Num_out != 1) ;
QuitCDF ("18.24b", $status) if ($extendRecsOut2 != $extendRecs2) ;
QuitCDF ("18.24c", $status) if ($maxAllocOut2 + 1 != $allocRecs2) ;
QuitCDF ("18.24d", $status) if ($maxRecOut2 != $maxRecWritten) ;

for ($dim_n = 0; $dim_n < $numDims; $dim_n++) {
   if ($var2DimVariances_out[$dim_n] != $var2DimVariances[$dim_n]) {
     QuitCDF ("18.25", $status);
   }
}

my ($zVarAname_out, $zVarAdataType_out, $zVarAnumElements_out, $zVarArecVariance_out);
my (@zVarAdimVariances_out, @zDimSizesA_out);
my ($zNumDimsA_out, $extendRecsOut3, $maxAllocOut3, $maxRecOut3, $nIndexRecsOut3, $nIndexEntriesOut3);
$status = CDF::CDFlib (SELECT_, zVAR_, $zVarAnum_out,
		       GET_, zVAR_NAME_, \$zVarAname_out,
		             zVAR_DATATYPE_, \$zVarAdataType_out,
		             zVAR_NUMELEMS_, \$zVarAnumElements_out,
		             zVAR_RECVARY_, \$zVarArecVariance_out,
		             zVAR_DIMVARYS_, \@zVarAdimVariances_out,
		             zVAR_NUMDIMS_, \$zNumDimsA_out,
		             zVAR_DIMSIZES_, \@zDimSizesA_out,
		             zVAR_BLOCKINGFACTOR_, \$extendRecsOut3,
		             zVAR_MAXallocREC_, \$maxAllocOut3,
		             zVAR_MAXREC_, \$maxRecOut3,
		             zVAR_nINDEXRECORDS_, \$nIndexRecsOut3,
		             zVAR_nINDEXENTRIES_, \$nIndexEntriesOut3,
		       NULL_);
QuitCDF ("18.0c1", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (CONFIRM_, zVAR_, \$zVarAnum_out,
	 	       NULL_);
QuitCDF ("18.0c2", $status) if ($status < CDF_OK) ;

QuitCDF ("18.21z", $status) if ($zVarAname_out ne $new_zVarAname) ;
QuitCDF ("18.22z", $status) if ($zVarAdataType_out != $zVarAdataType) ;
QuitCDF ("18.23z", $status) if ($zVarAnumElements_out != $zVarAnumElements) ;
QuitCDF ("18.24z", $status) if ($zVarArecVariance_out != $zVarArecVariance) ;
QuitCDF ("18.25z", $status) if ($zNumDimsA_out != $zNumDimsA) ;
QuitCDF ("18.27z", $status) if ($zVarAnum_out != 0) ;
QuitCDF ("18.27z1", $status) if ($extendRecsOut3 != $extendRecs3) ;
QuitCDF ("18.27z2", $status) if ($maxAllocOut3 + 1 != $allocRecs3) ;
QuitCDF ("18.27z3", $status) if ($maxRecOut3 != $maxRecWritten) ;

for ($dim_n = 0; $dim_n < $zNumDimsA; $dim_n++) {
   if ($zDimSizesA_out[$dim_n] != $zDimSizesA[$dim_n]) {
     QuitCDF ("18.27z", $status);
   }
   if ($zVarAdimVariances_out[$dim_n] != $zVarAdimVariances[$dim_n]) {
     QuitCDF ("18.28z", $status);
   }
}

#############################################################################
# Rename attribute.
#############################################################################

$status = CDF::CDFlib (SELECT_, ATTR_NAME_, $attrName,
		       PUT_, ATTR_NAME_, $new_attrName,
		       NULL_);
QuitCDF ("20.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Inquire attribute.
#############################################################################

my ($attrName_out, $attrScope_out, $maxEntry_out);
$status = CDF::CDFlib (GET_, ATTR_NAME_, \$attrName_out,
			     ATTR_SCOPE_, \$attrScope_out,
			     ATTR_MAXgENTRY_, \$maxEntry_out,
		       CONFIRM_, ATTR_, \$attrNum_out,
		       NULL_);
QuitCDF ("22.0", $status) if ($status < CDF_OK) ;

QuitCDF ("22.1", $status) if ($attrName_out ne $new_attrName) ;
QuitCDF ("22.2", $status) if ($attrScope_out != $attrScope) ;
QuitCDF ("22.3", $status) if ($maxEntry_out != $entryNum) ;
QuitCDF ("22.4", $status) if ($attrNum_out != 0) ;

#############################################################################
# Inquire attribute entries.
#############################################################################

my ($entryDataType_out, $entryNumElems_out);
$status = CDF::CDFlib (SELECT_, ATTR_, 0,
				gENTRY_, $entryNum,
		       GET_, gENTRY_DATATYPE_, \$entryDataType_out,
			     gENTRY_NUMELEMS_, \$entryNumElems_out,
		       NULL_);
QuitCDF ("23.0", $status) if ($status < CDF_OK) ;

QuitCDF ("23.1", $status) if ($entryDataType_out != $entryDataType) ;
QuitCDF ("23.1", $status) if ($entryNumElems_out != $entryNumElems) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 1,
				rENTRY_, 1,
		       GET_, rENTRY_DATATYPE_, \$entryDataType_out,
			     rENTRY_NUMELEMS_, \$entryNumElems_out,
		       NULL_);
QuitCDF ("23a.0", $status) if ($status < CDF_OK) ;

QuitCDF ("23a.1", $status) if ($entryDataType_out != CDF_BYTE) ;
QuitCDF ("23a.1", $status) if ($entryNumElems_out != 1) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 2,
				zENTRY_, 0,
		       GET_, zENTRY_DATATYPE_, \$entryDataType_out,
			     zENTRY_NUMELEMS_, \$entryNumElems_out,
		       NULL_);
QuitCDF ("23b.0", $status) if ($status < CDF_OK) ;

QuitCDF ("23b.1", $status) if ($entryDataType_out != CDF_REAL8) ;
QuitCDF ("23b.1", $status) if ($entryNumElems_out != 1) ;

#############################################################################
# Get error text.
#############################################################################

my $errorText;
$status = CDF::CDFlib (SELECT_, CDF_STATUS_, CDF_OK,
		       GET_, STATUS_TEXT_, \$errorText,
		       NULL_);
		 
QuitCDF ("24.0", $status) if ($status < CDF_OK) ;

#############################################################################
# Select zMode and inquire CDF.
#############################################################################

$status = CDF::CDFlib (SELECT_, CDF_zMODE_, zMODEon2,
		       NULL_);
QuitCDF ("25.0a", $status) if ($status < CDF_OK) ;

my ($numGattrs, $numVattrs, $maxGentry, $numGentries, $maxRecOut);
$status = CDF::CDFlib (SELECT_, ATTR_, 0,
		       GET_, CDF_NUMgATTRS_, \$numGattrs,
		             CDF_NUMvATTRS_, \$numVattrs,
		             CDF_NUMrVARS_, \$numRvars,
		             CDF_NUMzVARS_, \$numZvars,
		             ATTR_MAXgENTRY_, \$maxGentry,
		             ATTR_NUMgENTRIES_, \$numGentries,
		             zVARs_MAXREC_, \$maxRecOut,
		       NULL_);
QuitCDF ("25.0b", $status) if ($status < CDF_OK) ;

my ($maxRentry, $numRentries, $maxZentry, $numZentries);
$status = CDF::CDFlib (SELECT_, ATTR_, 1,
		       GET_, ATTR_MAXrENTRY_, \$maxRentry,
			     ATTR_NUMrENTRIES_, \$numRentries,
	 		     ATTR_MAXzENTRY_, \$maxZentry,
			     ATTR_NUMzENTRIES_, \$numZentries,
		       NULL_);
QuitCDF ("25.0c", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, CDF_zMODE_, zMODEoff,
		NULL_);
QuitCDF ("25.0d", $status) if ($status < CDF_OK) ;

QuitCDF ("25.1", $status) if ($numGattrs != 1) ; 
QuitCDF ("25.1a", $status) if ($numVattrs != 3) ;
QuitCDF ("25.1b", $status) if ($numRvars != 0) ;
QuitCDF ("25.2", $status) if ($numZvars != 3) ;
QuitCDF ("25.3", $status) if ($maxGentry != $entryNum) ;
QuitCDF ("25.4", $status) if ($numGentries != 1) ;
QuitCDF ("25.5", $status) if ($maxRentry != -1) ;
QuitCDF ("25.6", $status) if ($numRentries != 0) ;
QuitCDF ("25.7", $status) if ($maxZentry != 1);
QuitCDF ("25.8", $status) if ($numZentries != 1) ;
QuitCDF ("25.9", $status) if ($maxRecOut != $maxRecWritten) ;

#############################################################################
# Attempt to close variables.
#############################################################################

$status = CDF::CDFlib (SELECT_, rVAR_, 0,
		       CLOSE_, rVAR_,
		       NULL_);
QuitCDF ("27.0", $status) if ($status != SINGLE_FILE_FORMAT) ;

$status = CDF::CDFlib (SELECT_, zVAR_, 0,
		       CLOSE_, zVAR_,
		       NULL_);
QuitCDF ("27.1", $status) if ($status != SINGLE_FILE_FORMAT) ;

#############################################################################
# Modify entries/attribute.
#############################################################################

$status = CDF::CDFlib (SELECT_, ATTR_, 0,
				gENTRY_, $entryNum,
		       PUT_, gENTRY_DATASPEC_, $entryDataTypeNew, $entryNumElems,
		       NULL_);
QuitCDF ("27a.0a", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 1,
				rENTRY_, 1,
		       PUT_, rENTRY_DATASPEC_, CDF_UINT1, 1,
		       NULL_);
QuitCDF ("27a.0b", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 2,
				zENTRY_, 0,
		       PUT_, zENTRY_DATASPEC_, CDF_EPOCH, 1,
		NULL_);
QuitCDF ("27a.0c", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 0,
		       PUT_, ATTR_SCOPE_, VARIABLE_SCOPE,
			     ATTR_SCOPE_, GLOBAL_SCOPE,
		       NULL_);
QuitCDF ("27a.0d", $status) if ($status < CDF_OK) ;

#############################################################################
# Delete entries/attribute/variables.
#############################################################################

$status = CDF::CDFlib (SELECT_, ATTR_, 0,
				gENTRY_, $entryNum,
		       DELETE_, gENTRY_,
		       SELECT_, ATTR_, 1,
				rENTRY_, 1,
		       DELETE_, rENTRY_,
		       SELECT_, ATTR_, 2,
				zENTRY_, 0,
		       DELETE_, zENTRY_,
		       NULL_);
QuitCDF ("27.1", $status) if ($status < CDF_OK) ;

$status = CDF::CDFlib (SELECT_, ATTR_, 0,
		       DELETE_, ATTR_,
		       SELECT_, rVAR_, 0,
#		DELETE_, rVAR_,
		       SELECT_, zVAR_, 0,
#		DELETE_, zVAR_,
		       NULL_);
QuitCDF ("27.2", $status) if ($status < CDF_OK) ;

#############################################################################
# Checksum.
#############################################################################
my $chksum;

#$status = CDF::CDFlib (GET_, CDF_CHECKSUM_, \$chksum,
#                       NULL_);
$status = CDF::CDFgetChecksum ($id, \$chksum);
QuitCDF ("28.0", $status) if ($status < CDF_OK) ;

QuitCDF ("28.1", $chksum) if ($chksum != MD5_CHECKSUM) ;

#############################################################################
# Close CDF.
#############################################################################

$status = CDF::CDFlib (CLOSE_, CDF_,
		       NULL_);
QuitCDF ("28.2", $status) if ($status < CDF_OK) ;

#############################################################################
# Test Epoch Routines
#############################################################################
print "Test Epoch...\n";
my $year = 1994;
my $month = 10;
my $day = 13;
my $hour = 1;
my $minute = 2;
my $second = 3;
my $msec = 987;

my $epoch = CDF::computeEPOCH ($year, $month, $day, $hour, $minute,  
                               $second, $msec);

my $epStringTrue = '13-Oct-1994 01:02:03.987';
my $epString1True = '19941013.0431017';
my $epString2True = '19941013010203';
my $epString3True = '1994-10-13T01:02:03.987Z';

my ($epString, $epString1, $epString2, $epString3);

CDF::encodeEPOCH ($epoch, $epString);
QuitEPOCH ("30.0") if ($epString ne $epStringTrue);

CDF::encodeEPOCH1 ($epoch, $epString1);
QuitEPOCH ("30.1") if ($epString1 ne $epString1True);

CDF::encodeEPOCH2 ($epoch, $epString2);
QuitEPOCH ("30.2") if ($epString2 ne $epString2True);

CDF::encodeEPOCH3 ($epoch, $epString3);
QuitEPOCH ("30.3") if ($epString3 ne $epString3True);

my $epochOut = CDF::parseEPOCH ($epString);
QuitEPOCH ("31.0") if ($epochOut != $epoch) ;

my ($yearOut, $monthOut, $dayOut, $hourOut, $minuteOut, $secondOut, 
    $msecOut);

CDF::EPOCHbreakdown ($epoch, $yearOut, $monthOut, $dayOut, $hourOut, 
                     $minuteOut, $secondOut, $msecOut);

QuitEPOCH ("32.1") if ($yearOut != $year) ;
QuitEPOCH ("32.2") if ($monthOut != $month) ;
QuitEPOCH ("32.3") if ($dayOut != $day) ;
QuitEPOCH ("32.4") if ($hourOut != $hour) ;
QuitEPOCH ("32.5") if ($minuteOut != $minute) ;
QuitEPOCH ("32.6") if ($secondOut != $second) ;
QuitEPOCH ("32.7") if ($msecOut != $msec) ;

#############################################################################
# Test Epoch16 Routines
#############################################################################
print "Test Epoch16...\n";

my $usec = 876;
my $nsec = 765;
my $psec = 654;
my @epoch16;
my $retco = CDF::computeEPOCH16 ($year, $month, $day, $hour, $minute, $second, 
                                 $msec, $usec, $nsec, $psec, \@epoch16);

my $ep16StringTrue = '13-Oct-1994 01:02:03.987.876.765.654';
my $ep16String1True = '19941013.043101711536640';
my $ep16String2True = '19941013010203';
my $ep16String3True = '1994-10-13T01:02:03.987.876.765.654Z';

my ($ep16String, $ep16String1, $ep16String2, $ep16String3);

CDF::encodeEPOCH16 (\@epoch16, $ep16String);
QuitEPOCH ("40.0") if ($ep16String ne $ep16StringTrue);

CDF::encodeEPOCH16_1 (\@epoch16, $ep16String1);
#print "encodeEPOCH16_1:$ep16String1";
QuitEPOCH ("40.1") if ($ep16String1 ne $ep16String1True);

CDF::encodeEPOCH16_2 (\@epoch16, $ep16String2);
QuitEPOCH ("40.2") if ($ep16String2 ne $ep16String2True);

CDF::encodeEPOCH16_3 (\@epoch16, $ep16String3);
QuitEPOCH ("40.3") if ($ep16String3 ne $ep16String3True);

my @epoch16Out;
$retco = CDF::parseEPOCH16 ($ep16String, \@epoch16Out);
QuitEPOCH ("41.0") if ($epoch16Out[0] != $epoch16[0]) ;
QuitEPOCH ("41.1") if ($epoch16Out[1] != $epoch16[1]) ;

my ($year16Out, $month16Out, $day16Out, $hour16Out, $minute16Out, 
    $second16Out, $msec16Out, $usec16Out, $nsec16Out, $psec16Out);

CDF::EPOCH16breakdown (\@epoch16, $year16Out, $month16Out, $day16Out, 
                       $hour16Out, $minute16Out, $second16Out, $msec16Out,
                       $usec16Out, $nsec16Out, $psec16Out);

QuitEPOCH ("42.0") if ($year16Out != $year) ;
QuitEPOCH ("42.1") if ($month16Out != $month) ;
QuitEPOCH ("42.2") if ($day16Out != $day) ;
QuitEPOCH ("42.3") if ($hour16Out != $hour) ;
QuitEPOCH ("42.4") if ($minute16Out != $minute) ;
QuitEPOCH ("42.5") if ($second16Out != $second) ;
QuitEPOCH ("42.6") if ($msec16Out != $msec) ;
QuitEPOCH ("42.7") if ($usec16Out != $usec) ;
QuitEPOCH ("42.8") if ($nsec16Out != $nsec) ;
QuitEPOCH ("42.9") if ($psec16Out != $psec) ;

#############################################################################
# Successful completion.
#############################################################################
print "All tests completed successfully\n";
exit;

#############################################################################
# QuitCDF.
#############################################################################
sub QuitCDF {
	my ($where, $status)=@_;

  print "Aborting at $where ...\n";
  if ($status < CDF_OK) {
    my $text;
    CDF::CDFlib (SELECT_, CDF_STATUS_, $status,
		   GET_, STATUS_TEXT_, \$text,
		   NULL_);
    print $text;
  }
  CDF::CDFlib (CLOSE_, CDF_,
	  NULL_);
  print "...test aborted.\n";
  exit;
	
}#endsub QuitCDF

#############################################################################
#  QuitEPOCH
#############################################################################
sub QuitEPOCH {
  my ($where) = @_;
  print "Aborting at $where...test aborted.\n";
  exit;

}#endsub QuitEPOCH
